package ir.map.sdk_map.maps;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.util.AttributeSet;
import android.view.Gravity;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.mapbox.mapboxsdk.maps.MapboxMap;
import com.mapbox.mapboxsdk.maps.MapboxMapOptions;
import com.mapbox.mapboxsdk.maps.OnMapReadyCallback;
import com.mapbox.mapboxsdk.maps.Style;

import java.util.Calendar;
import java.util.Date;

import ir.map.mapsdk.R;
import ir.map.sdk_map.Mapir;
import ir.map.sdk_map.util.autonightmode.AutoNightModeConfiguration;
import ir.map.sdk_map.utils.MapLoadUtils;
import ir.map.sdk_map.utils.NetworkUtils;
import okhttp3.OkHttpClient;

import static ir.map.sdk_map.MapirStyle.CARMANIA;
import static ir.map.sdk_map.MapirStyle.HYRCANIA;
import static ir.map.sdk_map.MapirStyle.ISATIS;
import static ir.map.sdk_map.MapirStyle.VERNA;
import static ir.map.sdk_map.MapirStyle.WORLD_HYRCANIA;
import static ir.map.sdk_map.util.autonightmode.SunriseSunset.getSunriseSunset;

@Keep
public class MapView extends com.mapbox.mapboxsdk.maps.MapView {

    private MapboxMap map;
    private boolean isAutoNightModeEnabled = false;
    private AutoNightModeConfiguration styleConfiguration = new AutoNightModeConfiguration.Builder().build();

    BroadcastReceiver tickReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (intent.getAction() != null && intent.getAction().equals(Intent.ACTION_TIME_TICK) && isAutoNightModeEnabled) {
                changeMapStyle(getSunriseSunset(Calendar.getInstance(), styleConfiguration.getLocation().getLatitude(), styleConfiguration.getLocation().getLongitude()));
            }
        }
    };

    public MapView(@NonNull Context context) {
        super(context);
    }

    public MapView(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
    }

    public MapView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    public MapView(@NonNull Context context, @Nullable MapboxMapOptions options) {
        super(context, options);
    }

    private boolean isInitialOnResume = true;

    @Override
    protected void initialize(@NonNull Context context, @NonNull MapboxMapOptions options) {
        super.initialize(context, options);

        ImageView logoView = findViewById(com.mapbox.mapboxsdk.R.id.logoView);
        logoView.setImageResource(R.drawable.ic_map_logo_light);
        logoView.setMaxWidth(150);
        logoView.setMaxHeight(50);
        options.compassImage(getResources().getDrawable(R.drawable.ic_compass));

        TextView copyRight = new TextView(context);
        copyRight.setText("© Map © OpenStreetMap");
        copyRight.setGravity(Gravity.RIGHT | Gravity.BOTTOM);
        copyRight.setTextSize(12);
        copyRight.setPadding(0, 0, 12, 12);
        addView(copyRight);

        ImageView attrView = findViewById(com.mapbox.mapboxsdk.R.id.attributionView);
        attrView.setClickable(false);
        attrView.setEnabled(false);
        attrView.setAlpha(0f);
        attrView.setVisibility(GONE);

        addOnDidFinishLoadingStyleListener(new OnDidFinishLoadingStyleListener() {
            @Override
            public void onDidFinishLoadingStyle() {
                if (map != null) {
                    map.getStyle(new Style.OnStyleLoaded() {
                        @Override
                        public void onStyleLoaded(@NonNull Style style) {
                            // Change mapir logo base on style
                            ImageView logoView = findViewById(com.mapbox.mapboxsdk.R.id.logoView);
                            switch (style.getUri()) {
                                case CARMANIA:
                                    logoView.setImageResource(R.drawable.ic_map_logo_dark);
                                    logoView.setMaxWidth(150);
                                    logoView.setMaxHeight(50);
                                    break;
                                case VERNA:
                                case ISATIS:
                                    logoView.setImageResource(R.drawable.ic_map_logo_light);
                                    logoView.setMaxWidth(150);
                                    logoView.setMaxHeight(50);
                                    break;
                                default:
//                                    if (style.getJson().equals(HYRCANIA)) {
                                    logoView.setImageResource(R.drawable.ic_map_logo_light);
                                    logoView.setMaxWidth(150);
                                    logoView.setMaxHeight(50);
//                                    }
                            }
                        }
                    });
                }
            }
        });
    }

    /**
     * Sets a callback object which will be triggered when the {@link MapboxMap} instance is ready to be used.
     *
     * @param callback The callback object that will be triggered when the map is ready to be used.
     */
    @Override
    public void getMapAsync(@NonNull final OnMapReadyCallback callback) {
        super.getMapAsync(new OnMapReadyCallback() {
            @Override
            public void onMapReady(@NonNull final MapboxMap mapboxMap) {
                map = mapboxMap;
                map.setMinZoomPreference(1);
                map.setMaxZoomPreference(22);
                if (isAutoNightModeEnabled) {
                    changeMapStyle(getSunriseSunset(Calendar.getInstance(), styleConfiguration.getLocation().getLatitude(), styleConfiguration.getLocation().getLongitude()));
                    enableAutoNightMode(styleConfiguration);
                } else
                    map.setStyle(new Style.Builder().fromJson(WORLD_HYRCANIA));

                callback.onMapReady(map);
            }
        });
    }

    private void changeMapStyle(Calendar[] sunriseSunset) {
        final String styleUrl;
        Date currentTime = Calendar.getInstance().getTime();
        Date sunrise = sunriseSunset[0].getTime();
        Date sunset = sunriseSunset[1].getTime();
        if (currentTime.after(sunrise) && currentTime.before(sunset)) {
            styleUrl = styleConfiguration.getLightStyle();
        } else if (currentTime.after(sunset)) {
            styleUrl = styleConfiguration.getDarkStyle();
        } else {
            styleUrl = styleConfiguration.getLightStyle();
        }
        if (map != null) {
            map.setStyle(styleUrl);
        }
    }

    void enableAutoNightMode(AutoNightModeConfiguration configuration) {
        isAutoNightModeEnabled = true;
        this.styleConfiguration = configuration;
        try {
            if (getContext() != null)
                getContext().registerReceiver(tickReceiver, new IntentFilter(Intent.ACTION_TIME_TICK));
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        }
    }

    void disableAutoNightMode() {
        isAutoNightModeEnabled = false;
        try {
            if (tickReceiver != null)
                getContext().unregisterReceiver(tickReceiver);
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        }
    }

    public void enableAutoNightMode(boolean enable, AutoNightModeConfiguration configuration) {
        if (enable)
            enableAutoNightMode(configuration);
        else
            disableAutoNightMode();
    }

    @Override
    public void onResume() {
        super.onResume();
        if (!isInitialOnResume) {
            OkHttpClient client = new NetworkUtils(getContext()).getOkHttpClient(Mapir.getApiKey());
            new MapLoadUtils().sendUsageRequest(client);
        }
        isInitialOnResume = false;
        if (isAutoNightModeEnabled) {
            changeMapStyle(getSunriseSunset(Calendar.getInstance(), styleConfiguration.getLocation().getLatitude(), styleConfiguration.getLocation().getLongitude()));
            enableAutoNightMode(styleConfiguration);
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        try {
            if (tickReceiver != null)
                if (getContext() != null)
                    getContext().unregisterReceiver(tickReceiver);
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();
    }
}
