package ir.msob.jima.cloud.rsocket.beans;

import ir.msob.jima.cloud.rsocket.commons.model.InstanceInfo;
import ir.msob.jima.cloud.rsocket.commons.util.RoundRobinList;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

@Service
@Log4j2
@RequiredArgsConstructor
public class ApplicationCacheService {
    @Getter
    private final Map<String, RoundRobinList<InstanceInfo>> applications = new HashMap<>();
    private final Object lockObject = new Object();

    public RoundRobinList<InstanceInfo> getInstanceInfos(String applicationName) {
        return applications.getOrDefault(applicationName, new RoundRobinList<>());
    }

    public InstanceInfo getInstanceInfo(String applicationName) {
        return getInstanceInfos(applicationName).next();
    }

    public InstanceInfo getInstanceInfoExcept(String applicationName, InstanceInfo instanceInfo) {
        return getInstanceInfos(applicationName).nextExcept(instanceInfo);
    }

    public void add(Collection<InstanceInfo> inputs) {
        log.info("Register application. instanceInfos {}", inputs);
        synchronized (lockObject) {
            for (InstanceInfo instanceInfo : inputs) {
                RoundRobinList<InstanceInfo> instanceInfos = applications.get(instanceInfo.getApplicationName());
                if (instanceInfos == null) instanceInfos = new RoundRobinList<>();
                if (instanceInfos.stream().anyMatch(ii -> Objects.equals(ii.getInstanceId(), instanceInfo.getInstanceId())))
                    return;
                instanceInfos.add(instanceInfo);
                applications.put(instanceInfo.getApplicationName(), instanceInfos);
            }

        }
    }

    public void remove(Collection<InstanceInfo> inputs) {
        log.info("Unregister application. instanceInfos {}", inputs);
        synchronized (lockObject) {
            for (InstanceInfo instanceInfo : inputs) {
                RoundRobinList<InstanceInfo> instanceInfos = applications.get(instanceInfo.getApplicationName());
                if (instanceInfos == null) return;
                instanceInfos.removeIf(ii -> Objects.equals(ii.getInstanceId(), instanceInfo.getInstanceId()));
                if (instanceInfos.isEmpty()) {
                    applications.remove(instanceInfo.getApplicationName());
                } else {
                    applications.put(instanceInfo.getApplicationName(), instanceInfos);
                }
            }
        }
    }
}
