package ir.msob.jima.cloud.rsocket.beans;

import io.rsocket.RSocket;
import io.rsocket.SocketAcceptor;
import ir.msob.jima.cloud.rsocket.commons.model.ConnectionInfo;
import ir.msob.jima.core.commons.logger.Logger;
import ir.msob.jima.core.commons.logger.LoggerFactory;
import lombok.NoArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.ToString;
import org.springframework.boot.rsocket.server.RSocketServer;
import org.springframework.messaging.rsocket.RSocketRequester;
import org.springframework.messaging.rsocket.RSocketStrategies;
import org.springframework.messaging.rsocket.annotation.support.RSocketMessageHandler;
import org.springframework.stereotype.Service;
import reactor.util.retry.Retry;
import reactor.util.retry.RetryBackoffSpec;

import java.net.URI;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

@Service
@RequiredArgsConstructor
public class RequesterBuilder {
    private final RSocketRequester.Builder rsocketRequesterBuilder;
    private final RSocketStrategies rsocketStrategies;
    private final RetryBackoffSpec retryBackoffSpec = Retry.backoff(1000, Duration.ofMillis(500));
    Logger log = LoggerFactory.getLog(RequesterBuilder.class);

    public Builder builder() {
        return new Builder();
    }

    @NoArgsConstructor
    @ToString
    public class Builder {
        private final List<Object> candidateHandlers = new ArrayList<>();
        private ConnectionInfo connectionInfo;
        private String setupRoute;
        private Object setupData;

        public Builder connectionInfo(ConnectionInfo connectionInfo) {
            this.connectionInfo = connectionInfo;
            return this;
        }

        public Builder setupRoute(String setupRoute) {
            this.setupRoute = setupRoute;
            return this;
        }

        public Builder setupData(Object setupData) {
            this.setupData = setupData;
            return this;
        }

        public Builder addHandler(Object candidateHandlers) {
            this.candidateHandlers.add(candidateHandlers);
            return this;
        }

        public Builder addHandlers(List<Object> candidateHandlers) {
            if (candidateHandlers != null && !candidateHandlers.isEmpty())
                this.candidateHandlers.addAll(candidateHandlers);
            return this;
        }

        public RSocketRequester build() {
            RSocketRequester.Builder builder = rsocketRequesterBuilder
                    .rsocketStrategies(rsocketStrategies)
                    .rsocketConnector(connector -> connector.acceptor(getSocketAcceptor(candidateHandlers)).reconnect(retryBackoffSpec));

            RSocketRequester requester = initTransport(connectionInfo, builder);

            requester.rsocketClient()
                    .source()
                    .flatMap(RSocket::onClose)
                    .repeat()
                    .log()
                    .retryWhen(retryBackoffSpec)
                    .doOnError(error -> log.warn("Connection closed. builder {}", error, this))
                    .doFinally(consumer -> log.info("Connection disconnected. consumer {} builder {}", consumer, this))
                    .subscribe();

            return requester;
        }

        private SocketAcceptor getSocketAcceptor(List<Object> candidateHandlers) {
            RSocketMessageHandler handler = new RSocketMessageHandler();
            handler.setHandlers(candidateHandlers);
            handler.setRSocketStrategies(rsocketStrategies);
            handler.afterPropertiesSet();
            return handler.responder();
        }

        private RSocketRequester initTransport(ConnectionInfo connectionInfo, RSocketRequester.Builder builder) {
            if (connectionInfo.getTransport() == RSocketServer.Transport.TCP) {
                return builder
                        .tcp(connectionInfo.getHost(), connectionInfo.getPort()); // TODO by rSocketServerInfo.getServerUrl()
            } else if (connectionInfo.getTransport() == RSocketServer.Transport.WEBSOCKET) {
                return builder
                        .websocket(URI.create(connectionInfo.getServerUrl()));
            } else {
                throw new RuntimeException("Can not support transport: " + connectionInfo.getTransport());
            }
        }
    }
}
