package ir.msob.jima.cloud.rsocket.client;

import ir.msob.jima.cloud.rsocket.commons.model.ClientPayload;
import ir.msob.jima.cloud.rsocket.commons.model.RequestPayload;
import ir.msob.jima.core.commons.logger.Logger;
import ir.msob.jima.core.commons.logger.LoggerFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.messaging.rsocket.RSocketRequester;
import org.springframework.stereotype.Service;
import reactor.core.publisher.Mono;

import static ir.msob.jima.cloud.rsocket.commons.Constants.*;

@Service
@RequiredArgsConstructor
public class ClientRequester {
    private final RSocketRequester.Builder rsocketRequesterBuilder;
    private final Logger log = LoggerFactory.getLog(ClientRequester.class);

    public RSocketRequester.Builder buildRequester(String clientId) {
        log.debug("Building requester for clientId: {}", clientId);
        return rsocketRequesterBuilder
                .setupRoute(GATEWAY_SERVER_ROUTE + "." + CLIENT_CONNECT_ROUTE)
                .setupData(prepareSetupPayload(clientId));
    }


    public <R> Mono<R> requestResponse(RSocketRequester requester, String clientId, String applicationName, String route, Object data, Class<R> responseClass) {
        log.debug("Sending request-response with clientId: {}, applicationName: {}, route: {}", clientId, applicationName, route);
        return requester
                .route(GATEWAY_SERVER_ROUTE + "." + CLIENT_REQUEST_RESPONSE_ROUTE)
                .data(buildRequestPayload(clientId, applicationName, route, data))
                .retrieveMono(responseClass)
                .doOnSuccess(response -> log.debug("Request-response succeeded with response: {}", response))
                .doOnError(error -> log.error("Request-response failed with error: {}", error.getMessage()));
    }


    public Mono<Void> fireAndForget(RSocketRequester requester, String clientId, String applicationName, String route, Object data) {
        log.debug("Sending fire-and-forget with clientId: {}, applicationName: {}, route: {}", clientId, applicationName, route);
        return requester
                .route(GATEWAY_SERVER_ROUTE + "." + CLIENT_FIRE_AND_FORGET_ROUTE)
                .data(buildRequestPayload(clientId, applicationName, route, data))
                .send()
                .doOnSuccess(signal -> log.debug("Fire-and-forget completed successfully"))
                .doOnError(error -> log.error("Fire-and-forget failed with error: {}", error.getMessage()));
    }


    private RequestPayload buildRequestPayload(String clientId, String applicationName, String route, Object data) {
        return RequestPayload.builder()
                .clientId(clientId)
                .applicationName(applicationName)
                .route(route)
                .data(data)
                .build();
    }

    private ClientPayload prepareSetupPayload(String clientId) {
        return ClientPayload.builder().clientId(clientId).build();
    }


}
