package ir.msob.jima.cloud.rsocket.gateway.server.client;

import ir.msob.jima.cloud.rsocket.beans.ApplicationCacheService;
import ir.msob.jima.cloud.rsocket.beans.RequesterBuilder;
import ir.msob.jima.cloud.rsocket.commons.model.ClientInfo;
import ir.msob.jima.cloud.rsocket.commons.model.ClientPayload;
import ir.msob.jima.cloud.rsocket.commons.model.InstanceInfo;
import ir.msob.jima.cloud.rsocket.servicediscovery.client.properties.ServiceDiscoveryClientProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.messaging.rsocket.RSocketRequester;
import org.springframework.stereotype.Service;

import java.util.Objects;

import static ir.msob.jima.cloud.rsocket.commons.Constants.*;

@Service
@Log4j2
@RequiredArgsConstructor
public class GatewayClient {

    private final ApplicationCacheService applicationCacheService;
    private final ServiceDiscoveryClientProperties serviceDiscoveryClientProperties;
    private final RequesterBuilder requesterBuilder;
    @Value("${spring.application.name}")
    private String applicationName;

    public void registerClient(ClientInfo clientInfo) {
        applicationCacheService.getInstanceInfos(applicationName)
                .stream()
                .filter(instanceInfo -> !Objects.equals(serviceDiscoveryClientProperties.getInstanceId(), instanceInfo.getInstanceId()))
                .map(this::prepareRequester)
                .forEach(ii -> ii.getRequester()
                        .route(GATEWAY_SERVER_ROUTE + "." + CLIENT_REGISTER_ROUTE)
                        .data(ClientPayload.builder()
                                .clientId(clientInfo.getClientId())
                                .gatewayId(serviceDiscoveryClientProperties.getInstanceId())
                                .build())
                        .retrieveMono(ClientPayload.class)
                        .log()
                        .subscribe());

    }

    private InstanceInfo prepareRequester(InstanceInfo instanceInfo) {
        if (instanceInfo.getRequester() == null) {
            RSocketRequester requester = requesterBuilder.builder()
                    .connectionInfo(instanceInfo.getConnectionInfo())
                    .build();
            instanceInfo.setRequester(requester);
        }
        return instanceInfo;
    }

    public void unregisterClient(ClientInfo clientInfo) {
        applicationCacheService.getInstanceInfos(applicationName)
                .stream()
                .filter(instanceInfo -> !Objects.equals(serviceDiscoveryClientProperties.getInstanceId(), instanceInfo.getInstanceId()))
                .forEach(ii -> ii.getRequester()
                        .route(GATEWAY_SERVER_ROUTE + "." + CLIENT_UNREGISTER_ROUTE)
                        .data(ClientPayload.builder()
                                .clientId(clientInfo.getClientId())
                                .gatewayId(serviceDiscoveryClientProperties.getInstanceId())
                                .build())
                        .retrieveMono(ClientPayload.class)
                        .log()
                        .subscribe());
    }

}
