package ir.msob.jima.cloud.rsocket.gateway.server.controller;

import ir.msob.jima.cloud.rsocket.commons.model.ClientInfo;
import ir.msob.jima.cloud.rsocket.commons.model.ClientPayload;
import ir.msob.jima.cloud.rsocket.gateway.server.ClientCacheService;
import ir.msob.jima.cloud.rsocket.gateway.server.client.GatewayClient;
import ir.msob.jima.cloud.rsocket.servicediscovery.client.properties.ServiceDiscoveryClientProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.messaging.handler.annotation.MessageMapping;
import org.springframework.messaging.handler.annotation.Payload;
import org.springframework.messaging.rsocket.RSocketRequester;
import org.springframework.messaging.rsocket.annotation.ConnectMapping;
import org.springframework.stereotype.Controller;

import java.util.Objects;

import static ir.msob.jima.cloud.rsocket.commons.Constants.*;

@Controller
@MessageMapping(GATEWAY_SERVER_ROUTE)
@RequiredArgsConstructor
@Log4j2
public class ClientController {

    private final ClientCacheService clientCacheService;
    private final GatewayClient gatewayClient;
    private final ServiceDiscoveryClientProperties serviceDiscoveryClientProperties;

    @ConnectMapping(CLIENT_SETUP_ROUTE)
    public void connecting(RSocketRequester requester, @Payload ClientPayload payload) {
        log.info("{} ConnectMapping. {}", CLIENT_CONNECT_ROUTE, payload);

        if (requester.rsocket() == null)
            throw new RuntimeException("Requester socket is null");

        Objects.requireNonNull(requester.rsocket())
                .onClose()
                .doFirst(() -> {
                    log.info("Client connected. {}", payload);
                    connect(requester, payload);
                })
                .doOnError(error -> {
                    log.info("Client connection has an error. payload {}, error {}", payload, error);
                    disconnect(payload);
                })
                .doFinally(consumer -> {
                    log.info("Client disconnected. payload {}, consumer {}", payload, consumer);
                    disconnect(payload);
                })
                .subscribe();

    }

    @MessageMapping(CLIENT_REGISTER_ROUTE)
    public void register(@Payload ClientPayload payload) {
        log.info("Received {} request: {}", CLIENT_REGISTER_ROUTE, payload);
        ClientInfo clientInfo = ClientInfo.builder()
                .clientId(payload.getClientId())
                .gatewayId(payload.getGatewayId())
                .build();
        clientCacheService.add(clientInfo);
    }

    @MessageMapping(CLIENT_UNREGISTER_ROUTE)
    public void unregister(@Payload ClientPayload payload) {
        log.info("Received {} request: {}", CLIENT_UNREGISTER_ROUTE, payload);
        ClientInfo clientInfo = ClientInfo.builder()
                .clientId(payload.getClientId())
                .gatewayId(payload.getGatewayId())
                .build();
        clientCacheService.remove(clientInfo);
    }


    private void connect(RSocketRequester requester, ClientPayload payload) {
        payload.setGatewayId(serviceDiscoveryClientProperties.getInstanceId());

        ClientInfo clientInfo = ClientInfo.builder()
                .clientId(payload.getClientId())
                .gatewayId(payload.getGatewayId())
                .requester(requester)
                .build();
        clientCacheService.add(clientInfo);
        gatewayClient.registerClient(clientInfo);
        returnPayload(requester, payload);
    }

    private void returnPayload(RSocketRequester requester, ClientPayload payload) {
        requester
                .route(CLIENT_SETUP_ROUTE)
                .data(payload)
                .send()
                .subscribe();
    }

    private void disconnect(ClientPayload payload) {
        ClientInfo clientInfo = ClientInfo.builder()
                .clientId(payload.getClientId())
                .gatewayId(serviceDiscoveryClientProperties.getInstanceId())
                .build();
        clientCacheService.remove(clientInfo);
        gatewayClient.unregisterClient(clientInfo);
    }


}
