package ir.msob.jima.cloud.rsocket.gateway.server.controller;

import ir.msob.jima.cloud.rsocket.beans.ApplicationCacheService;
import ir.msob.jima.cloud.rsocket.beans.RequesterBuilder;
import ir.msob.jima.cloud.rsocket.commons.model.ClientInfo;
import ir.msob.jima.cloud.rsocket.commons.model.InstanceInfo;
import ir.msob.jima.cloud.rsocket.commons.model.RequestPayload;
import ir.msob.jima.cloud.rsocket.gateway.server.ClientCacheService;
import ir.msob.jima.cloud.rsocket.servicediscovery.client.properties.ServiceDiscoveryClientProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.messaging.handler.annotation.MessageMapping;
import org.springframework.messaging.handler.annotation.Payload;
import org.springframework.stereotype.Controller;
import reactor.core.publisher.Mono;

import java.util.Objects;

import static ir.msob.jima.cloud.rsocket.commons.Constants.*;

@Controller
@RequiredArgsConstructor
@MessageMapping(GATEWAY_SERVER_ROUTE)
@Log4j2
public class ApplicationRequestController {

    private final ClientCacheService clientCacheService;
    private final ServiceDiscoveryClientProperties serviceDiscoveryClientProperties;
    private final ApplicationCacheService applicationCacheService;
    private final RequesterBuilder requesterBuilder;
    @Value("${spring.application.name}")
    private String applicationName;

    @MessageMapping(APPLICATION_REQUEST_RESPONSE_ROUTE)
    public Mono<Object> requestResponse(@Payload RequestPayload payload) {
        log.info("Received {} request: {}", APPLICATION_REQUEST_RESPONSE_ROUTE, payload);

        ClientInfo clientInfo = clientCacheService.getClientInfo(payload.getClientId());
        if (clientInfo.getGatewayId().equals(serviceDiscoveryClientProperties.getInstanceId())) {
            return clientInfo.getRequester()
                    .route(payload.getRoute())
                    .data(payload.getData())
                    .retrieveMono(Object.class);
        } else {
            InstanceInfo instanceInfo = applicationCacheService.getInstanceInfos(applicationName).stream()
                    .filter(ii -> Objects.equals(ii.getInstanceId(), clientInfo.getGatewayId()))
                    .findFirst()
                    .orElseThrow();

            if (instanceInfo.getRequester() == null)
                instanceInfo.setRequester(requesterBuilder.builder().connectionInfo(instanceInfo.getConnectionInfo()).build());

            return instanceInfo.getRequester()
                    .route(APPLICATION_REQUEST_RESPONSE_ROUTE)
                    .data(payload)
                    .retrieveMono(Object.class);
        }
    }

    @MessageMapping(APPLICATION_FIRE_AND_FORGET_ROUTE)
    public Mono<Void> fireAndForget(@Payload RequestPayload payload) {
        log.info("Received {} request: {}", APPLICATION_FIRE_AND_FORGET_ROUTE, payload);

        ClientInfo clientInfo = clientCacheService.getClientInfo(payload.getClientId());
        if (clientInfo.getGatewayId().equals(serviceDiscoveryClientProperties.getInstanceId())) {
            return clientInfo.getRequester()
                    .route(payload.getRoute())
                    .data(payload.getData())
                    .send();
        } else {
            InstanceInfo instanceInfo = applicationCacheService.getInstanceInfos(applicationName).stream()
                    .filter(ii -> Objects.equals(ii.getInstanceId(), clientInfo.getGatewayId()))
                    .findFirst()
                    .orElseThrow();

            if (instanceInfo.getRequester() == null)
                instanceInfo.setRequester(requesterBuilder.builder().connectionInfo(instanceInfo.getConnectionInfo()).build());

            return instanceInfo.getRequester()
                    .route(APPLICATION_FIRE_AND_FORGET_ROUTE)
                    .data(payload)
                    .send();
        }
    }

    @MessageMapping(APPLICATION_FIRE_AND_FORGET_WITHOUT_WAITING_ROUTE)
    public Mono<Void> fireAndForgetWithoutWaiting(@Payload RequestPayload payload) {
        log.info("Received {} request: {}", APPLICATION_FIRE_AND_FORGET_WITHOUT_WAITING_ROUTE, payload);

        ClientInfo clientInfo = clientCacheService.getClientInfo(payload.getClientId());
        if (clientInfo.getGatewayId().equals(serviceDiscoveryClientProperties.getInstanceId())) {
            clientInfo.getRequester()
                    .route(payload.getRoute())
                    .data(payload.getData())
                    .send()
                    .subscribe();
        } else {
            InstanceInfo instanceInfo = applicationCacheService.getInstanceInfos(applicationName).stream()
                    .filter(ii -> Objects.equals(ii.getInstanceId(), clientInfo.getGatewayId()))
                    .findFirst()
                    .orElseThrow();

            if (instanceInfo.getRequester() == null)
                instanceInfo.setRequester(requesterBuilder.builder().connectionInfo(instanceInfo.getConnectionInfo()).build());

            instanceInfo.getRequester()
                    .route(APPLICATION_FIRE_AND_FORGET_WITHOUT_WAITING_ROUTE)
                    .data(payload)
                    .send()
                    .subscribe();
        }
        return Mono.empty();
    }

}
