package ir.msob.jima.href.service;

import ir.msob.jima.core.beans.properties.JimaProperties;
import ir.msob.jima.core.commons.annotation.domain.DomainService;
import ir.msob.jima.core.commons.exception.badrequest.BadRequestException;
import ir.msob.jima.core.commons.exception.domainnotfound.DomainNotFoundException;
import ir.msob.jima.core.commons.model.criteria.BaseCriteria;
import ir.msob.jima.core.commons.model.dto.BaseDto;
import ir.msob.jima.core.commons.operation.BaseBeforeAfterOperation;
import ir.msob.jima.core.commons.operation.Operations;
import ir.msob.jima.core.commons.security.BaseUser;
import ir.msob.jima.href.commons.Href;
import ir.msob.jima.href.commons.HrefModel;
import ir.msob.jima.href.commons.Link;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

@Service
@RequiredArgsConstructor
public class HrefBeforeAfterOperation implements BaseBeforeAfterOperation {
    private final JimaProperties jimaProperties;

    @Override
    public <ID extends Comparable<ID> & Serializable, USER extends BaseUser, DTO extends BaseDto<ID>, C extends BaseCriteria<ID>>
    void afterGet(Collection<ID> ids, Collection<DTO> dtos, C criteria, Optional<USER> user) throws DomainNotFoundException, BadRequestException {
        if (dtos.isEmpty()) {
            return;
        }
        AtomicReference<Class<DTO>> dtoClass = new AtomicReference<>();
        dtos.stream().findFirst().ifPresent(dto -> dtoClass.set((Class<DTO>) dto.getClass()));
        Href href = Href.info.getAnnotation(dtoClass.get());
        DomainService domainService = DomainService.info.getAnnotation(dtoClass.get());
        if (href != null && domainService != null) {
            for (DTO dto : dtos) {
                if (dto instanceof HrefModel hrefModel) {
                    hrefModel.add(prepareLinks(domainService, dto));
                }
            }
        }
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> Iterable<Link> prepareLinks(DomainService domainService, DTO dto) {
        return List.of(prepareSelfLink(domainService, dto), preparePageLink(domainService), prepareCollectionLink(domainService));
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> Link prepareSelfLink(DomainService domainService, DTO dto) {
        return Link.of(getUrl(domainService, "/" + dto.getDomainId().toString()), "SELF");
    }

    private Link preparePageLink(DomainService domainService) {
        return Link.of(getUrl(domainService, ""), "PAGE");
    }

    private Link prepareCollectionLink(DomainService domainService) {
        return Link.of(getUrl(domainService, "/" + Operations.GET_MANY), "COLLECTION");
    }

    private String getUrl(DomainService domainService, String suffix) {
        return String.format("%s/%s/%s/%s/%s%s"
                , jimaProperties.getHref().getBaseUrl()
                , domainService.serviceName()
                , "api"
                , domainService.version()
                , domainService.domainName()
                , suffix);
    }
}
