package ir.msob.jima.lock.ral.mongo.beans;

import ir.msob.jima.core.ral.mongo.commons.criteria.MongoCriteria;
import lombok.RequiredArgsConstructor;
import org.springframework.data.mongodb.core.ReactiveMongoTemplate;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.util.Collection;
import java.util.concurrent.ExecutionException;


@Repository
@RequiredArgsConstructor
public class LockLogService {

    private final ReactiveMongoTemplate reactiveMongoTemplate;

    @Retryable(retryFor = {Exception.class}, maxAttemptsExpression = "${jima.lock.retry.max-attempts}", backoff = @Backoff(
            delayExpression = "${jima.lock.retry.delay}",
            maxDelayExpression = "${jima.lock.retry.max-delay}",
            multiplierExpression = "${jima.lock.retry.multiplier}",
            randomExpression = "${jima.lock.retry.random}"
    ))
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void saveMany(Collection<LockLog> lockLogs) throws ExecutionException, InterruptedException {
        Instant now = Instant.now();
        lockLogs.forEach(documentLock -> documentLock.setTs(now));
        reactiveMongoTemplate.insertAll(lockLogs)
                .collectList()
                .toFuture()
                .get();
    }


    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void deleteMany(Collection<String> ids) throws ExecutionException, InterruptedException {
        Query query = new Query();
        query.addCriteria(MongoCriteria.in(LockLog.FN.id, ids));
        reactiveMongoTemplate.remove(query, LockLog.class)
                .map(deleteResult -> deleteResult.getDeletedCount() > 0)
                .toFuture()
                .get();
    }
}
