package ir.msob.jima.search.client;

import ir.msob.jima.core.beans.properties.JimaProperties;
import ir.msob.jima.core.commons.client.BaseAsyncClient;
import ir.msob.jima.core.commons.model.channel.ChannelMessage;
import ir.msob.jima.core.commons.model.dto.BaseDto;
import ir.msob.jima.core.commons.operation.Operations;
import ir.msob.jima.core.commons.security.BaseUser;
import ir.msob.jima.search.commons.annotaion.CurrentClass;
import ir.msob.jima.search.commons.annotaion.DomainSearch;
import ir.msob.jima.search.commons.model.SearchMessage;
import org.modelmapper.ModelMapper;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.Optional;

public interface BaseSearchClient {

    ModelMapper getModelMapper();

    BaseAsyncClient getAsyncClient();

    JimaProperties getJimaProperties();

    default <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>, USER extends BaseUser<ID>> void insert(DTO dto, Optional<USER> user) {
        sendSearchMessage(dto, user, Operations.SAVE);
    }

    default <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>, USER extends BaseUser<ID>> void update(DTO dto, Optional<USER> user) {
        sendSearchMessage(dto, user, Operations.UPDATE);
    }

    default <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>, USER extends BaseUser<ID>> void delete(ID id, Class<DTO> dtoClass, Optional<USER> user) throws InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {
        DTO dto = prepareDto(id, dtoClass);
        sendSearchMessage(dto, user, Operations.DELETE);
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> DTO prepareDto(ID id, Class<DTO> dtoClass) throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        DTO dto = dtoClass.getDeclaredConstructor().newInstance();
        dto.setDomainId(id);
        return dto;
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>, USER extends BaseUser<ID>> void sendSearchMessage(DTO dto, Optional<USER> user, String operation) {
        DomainSearch domainSearch = getAuditLogAnnotation(dto);
        if (domainSearch == null) return;
        ChannelMessage<ID, USER, SearchMessage> message = prepareChannelMessage(dto, domainSearch, operation);
        send(message, user);
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> Object prepareData(DTO dto, DomainSearch domainSearch) {
        Object data;
        if (domainSearch.model() == CurrentClass.class) {
            data = dto;
        } else {
            data = getModelMapper().map(dto, domainSearch.model());
        }
        return data;
    }

    private <ID extends Comparable<ID> & Serializable, USER extends BaseUser<ID>, DTO extends BaseDto<ID>> ChannelMessage<ID, USER, SearchMessage> prepareChannelMessage(DTO dto, DomainSearch domainSearch, String operation) {
        SearchMessage searchMessage = new SearchMessage();
        searchMessage.setDto(prepareData(dto, domainSearch));
        searchMessage.setOperation(operation);
        if (domainSearch.model() == CurrentClass.class) {
            searchMessage.setDtoClass(dto.getClass());
        } else {
            searchMessage.setDtoClass(domainSearch.model());
        }
        ChannelMessage<ID, USER, SearchMessage> channelMessage = new ChannelMessage<>();
        channelMessage.setData(searchMessage);
        return channelMessage;
    }

    private <ID extends Comparable<ID> & Serializable, USER extends BaseUser<ID>> void send(ChannelMessage<ID, USER, SearchMessage> message, Optional<USER> user) {
        getAsyncClient().send(message, getJimaProperties().getSearch().getChannel(), user);
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> DomainSearch getAuditLogAnnotation(DTO dto) {
        return getAuditLogAnnotation(dto.getClass());
    }

    private <ID extends Comparable<ID> & Serializable, DTO extends BaseDto<ID>> DomainSearch getAuditLogAnnotation(Class<DTO> dtoClass) {
        return DomainSearch.info.getAnnotation(dtoClass);
    }

}
