package sdk.main.core;


import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.Nullable;

import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskCompletionSource;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

/**
 * Abstraction around downloading an image in a background executor.
 *
 * @hide
 */
public class ImageDownload implements Closeable {

    private static final String TAG = "ImageDownload";

    /**
     * Maximum image size to download in bytes (1 MiB).
     */
    private static final int MAX_IMAGE_SIZE_BYTES = 1024 * 1024;

    private final URL url;
    @Nullable
    private volatile Future<?> future;
    @Nullable
    private Task<Bitmap> task;

    private ImageDownload(URL url) {
        this.url = url;
    }

    @Nullable
    public static ImageDownload create(String imageUrl) {
        if (TextUtils.isEmpty(imageUrl)) {
            return null;
        }
        try {
            return new ImageDownload(new URL(imageUrl));
        } catch (MalformedURLException e) {
            Log.w(TAG, "Not downloading image, bad URL: " + imageUrl);
            return null;
        }
    }

    public void start(ExecutorService executor) {
        final TaskCompletionSource<Bitmap> taskCompletionSource = new TaskCompletionSource<>();
        future =
                executor.submit(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            Bitmap bitmap = blockingDownload();
                            taskCompletionSource.setResult(bitmap);
                        } catch (Exception e) {
                            taskCompletionSource.setException(e);
                        }
                    }
                });
        task = taskCompletionSource.getTask();
    }

    public Task<Bitmap> getTask() {
        if (task == null) {
            throw new NullPointerException();
        }
        return task; // will throw if getTask() is called without a call to start()
    }

    public Bitmap blockingDownload() throws IOException {
        if (Log.isLoggable(TAG, Log.INFO)) {
            Log.i(TAG, "Starting download of: " + url);
        }

        byte[] imageBytes = blockingDownloadBytes();
        Bitmap bitmap = BitmapFactory.decodeByteArray(imageBytes, /* offset= */ 0, imageBytes.length);
        if (bitmap == null) {
            throw new IOException("Failed to decode image: " + url);
        }

        if (Log.isLoggable(TAG, Log.DEBUG)) {
            Log.d(TAG, "Successfully downloaded image: " + url);
        }
        return bitmap;
    }

    @SuppressWarnings("UrlConnectionChecker")
    private byte[] blockingDownloadBytes() throws IOException {
        URLConnection connection = url.openConnection();

        // First check the content length, and fail if it's too high
        int contentLength = connection.getContentLength();
        if (contentLength > MAX_IMAGE_SIZE_BYTES) {
            throw new IOException("Content-Length exceeds max size of " + MAX_IMAGE_SIZE_BYTES);
        }

        // Now actually try to download the content
        byte[] bytes = new byte[0];
        InputStream connectionInputStream = null;
        try {
            connectionInputStream = connection.getInputStream();
            // Read one byte over the limit so we can tell if the data is too big, as in many cases
            // BitmapFactory will happily decode a partial image.
            bytes =
                    ByteStreams.toByteArray(
                            ByteStreams.limit(connectionInputStream, MAX_IMAGE_SIZE_BYTES + 1));
        } catch (Exception e) {
        } finally {
            if (connectionInputStream != null)
                connectionInputStream.close();
        }

        if (Log.isLoggable(TAG, Log.VERBOSE)) {
            Log.v(TAG, "Downloaded " + bytes.length + " bytes from " + url);
        }

        if (bytes.length > MAX_IMAGE_SIZE_BYTES) {
            throw new IOException("Image exceeds max size of " + MAX_IMAGE_SIZE_BYTES);
        }
        return bytes;
    }

    @Override
    public void close() {
        future.cancel(true);
    }
}
