package sdk.main.core;

import static sdk.main.core.PushHandler.EXTRA_ACTION_INDEX;
import static sdk.main.core.PushHandler.EXTRA_INTENT;
import static sdk.main.core.PushHandler.EXTRA_MESSAGE;
import static sdk.main.core.PushHandler.useAdditionalIntentRedirectionChecks;

import android.app.NotificationManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.os.Bundle;

import sdk.main.core.pushmessaging.PushMessage;
import sdk.main.core.pushmessaging.TimerNotificationService;

public final class MessageIntentHandler {
    public static final String SECURE_NOTIFICATION_BROADCAST = "ir.intrack.android.sdk.PushHandler.SECURE_NOTIFICATION_BROADCAST";

    private MessageIntentHandler() {
    } //make this class a static utility

    private static boolean isHttpOrHttpsUri(Uri uri) {
        if (uri == null) {
            return false;
        }
        String scheme = uri.getScheme();
        return scheme != null && (scheme.equalsIgnoreCase("http") || scheme.equalsIgnoreCase("https"));
    }

    public static void handleIntent(Context context, Intent targetIntent) {
        CoreInternal.sharedInstance().L.d("[PushHandler, NotificationBroadcastReceiver] Push broadcast receiver receiving message");
        final NotificationManager manager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

        int notificationId = targetIntent.getIntExtra(TimerNotificationService.EXTRA_NOTIFICATION_ID, -1);
        if (notificationId != -1) {
            Intent serviceIntent = new Intent(context, TimerNotificationService.class);
            serviceIntent.setAction(TimerNotificationService.ACTION_STOP_TIMER);
            serviceIntent.putExtra(TimerNotificationService.EXTRA_NOTIFICATION_ID, notificationId);
            manager.cancel(notificationId);
            context.startService(serviceIntent);
        }

        targetIntent.setExtrasClassLoader(PushHandler.class.getClassLoader());

        Intent intent = targetIntent.getParcelableExtra(EXTRA_INTENT);

        if (intent == null) {
            CoreInternal.sharedInstance().L.e("[PushHandler, NotificationBroadcastReceiver] Received a null Intent, stopping execution");
            return;
        }

        Intent broadcast = new Intent(SECURE_NOTIFICATION_BROADCAST, null);
        broadcast.setPackage(context.getApplicationContext().getPackageName());
        broadcast.putExtra(EXTRA_INTENT, intent);
        context.sendBroadcast(broadcast);

        int flags = intent.getFlags();
        if (((flags & Intent.FLAG_GRANT_READ_URI_PERMISSION) != 0) || ((flags & Intent.FLAG_GRANT_WRITE_URI_PERMISSION) != 0)) {
            CoreInternal.sharedInstance().L.w("[PushHandler, NotificationBroadcastReceiver] Attempt to get URI permissions");
            return;
        }

        if (useAdditionalIntentRedirectionChecks) { //TODO: ask about it, it was always false
            ComponentName componentName = intent.getComponent();
            String intentPackageName = componentName.getPackageName();
            String intentClassName = componentName.getClassName();
            String contextPackageName = context.getPackageName();

            if (intentPackageName != null && !intentPackageName.equals(contextPackageName)) {
                CoreInternal.sharedInstance().L.w("[PushHandler, NotificationBroadcastReceiver] Untrusted intent package");
                return;
            }

            if (!intentClassName.startsWith(intentPackageName)) {
                CoreInternal.sharedInstance().L.w("[PushHandler, NotificationBroadcastReceiver] intent class name and intent package names do not match");
                return;
            }
        }

        CoreInternal.sharedInstance().L.d("[PushHandler, NotificationBroadcastReceiver] Push broadcast, after filtering");

        intent.setExtrasClassLoader(PushHandler.class.getClassLoader());

        int index = intent.getIntExtra(EXTRA_ACTION_INDEX, 0);
        Bundle bundle = intent.getParcelableExtra(EXTRA_MESSAGE);
        if (bundle == null) {
            CoreInternal.sharedInstance().L.e("[PushHandler, NotificationBroadcastReceiver] Received a null Intent bundle, stopping execution");
            return;
        }

        PushMessage message = bundle.getParcelable(EXTRA_MESSAGE);
        if (message == null) {
            CoreInternal.sharedInstance().L.e("[PushHandler, NotificationBroadcastReceiver] Received a null Intent bundle message, stopping execution");
            return;
        }

        message.recordAction(context, index);

        if (manager != null) {
            manager.cancel(message.hashCode());
        }

        try {
            //try/catch required due to Android 12
            if (android.os.Build.VERSION.SDK_INT < 31) {
                //this needs to be called before Android 12
                Intent closeNotificationsPanel = new Intent(Intent.ACTION_CLOSE_SYSTEM_DIALOGS);
                context.sendBroadcast(closeNotificationsPanel);
            }
        } catch (Exception ex) {
            CoreInternal.sharedInstance().L.e("[PushHandler, NotificationBroadcastReceiver] Encountered issue while trying to send the on click broadcast. [" + ex.toString() + "]");
        }

        if (index == 0) {
            if (message.getLink() != null) {
                CoreInternal.sharedInstance().L.d("[PushHandler, NotificationBroadcastReceiver] Starting activity with given link. Push body. [" + message.getLink() + "]");
                Uri uri = message.getLink();
                Intent i = new Intent(Intent.ACTION_VIEW, uri);
                i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                i.putExtra(EXTRA_MESSAGE, bundle);
                i.putExtra(EXTRA_ACTION_INDEX, index);

                ResolveInfo info = context.getPackageManager().resolveActivity(i, 0);
                if (info != null || isHttpOrHttpsUri(uri)) {
                    context.startActivity(i);
                } else {
                    CoreInternal.sharedInstance().L.d("Device cannot resolve intent for: " + Intent.ACTION_VIEW);
                    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(intent);
                }
            } else {
                CoreInternal.sharedInstance().L.d("[PushHandler, NotificationBroadcastReceiver] Starting activity without a link. Push body");
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            }
        } else {
            CoreInternal.sharedInstance().L.d("[PushHandler, NotificationBroadcastReceiver] Starting activity with given button link. [" + (index - 1) + "] [" + message.getButtons().get(index - 1).getLink() + "]");
            Uri uri = message.getButtons().get(index - 1).getLink();
            Intent i = new Intent(Intent.ACTION_VIEW, uri);
            i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            i.putExtra(EXTRA_MESSAGE, bundle);
            i.putExtra(EXTRA_ACTION_INDEX, index);

            ResolveInfo info = context.getPackageManager().resolveActivity(i, 0);
            if (info != null || isHttpOrHttpsUri(uri)) {
                context.startActivity(i);
            } else {
                CoreInternal.sharedInstance().L.d("Device cannot resolve intent for: " + Intent.ACTION_VIEW);
                intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            }
        }
    }
}
