package sdk.main.core;

class ModuleDeviceId extends ModuleBase {
    boolean exitTempIdAfterInit = false;

    ModuleLog L;

    ModuleDeviceId(CoreInternal coreInternal, Config config) {
        super(coreInternal);

        L = coreInternal.L;
        L.v("[ModuleDeviceId] Initialising");

        boolean customIDWasProvided = (config.deviceID != null);
        if (config.temporaryDeviceIdEnabled && !customIDWasProvided) {
            //if we want to use temporary ID mode and no developer custom ID is provided
            //then we override that custom ID to set the temporary mode
            config.deviceID = DeviceId.temporaryInTrackDeviceId;
        }

        //choose what kind of device ID will be used
        if (config.deviceID != null) {
            //if the developer provided a ID
            //or it's a temporary ID
            config.deviceIdInstance = new DeviceId(config.sharedPref, config.deviceID, _int.L);
        } else {
            //the dev provided only a type and the SDK should generate a appropriate ID
            config.deviceIdInstance = new DeviceId(config.sharedPref, config.idMode, _int.L);
        }

        //initialise the set device ID value
        config.deviceIdInstance.init(config.context, config.sharedPref, true);

        boolean temporaryDeviceIdIsCurrentlyEnabled = config.deviceIdInstance.temporaryIdModeEnabled();
        L.d("[ModuleDeviceId] [TemporaryDeviceId] Temp ID should be enabled[" + config.temporaryDeviceIdEnabled + "] Currently enabled: [" + temporaryDeviceIdIsCurrentlyEnabled + "]");

        if (temporaryDeviceIdIsCurrentlyEnabled && customIDWasProvided) {
            //if a custom ID was provided and we are still in temporary ID mode
            //it means the we had tempID mode at the previous app end
            //exit tempID after init finished
            L.d("[ModuleDeviceId] [TemporaryDeviceId] Decided we have to exit temporary device ID mode, mode enabled: [" + config.temporaryDeviceIdEnabled + "], custom Device ID Set: [" + customIDWasProvided + "]");

            exitTempIdAfterInit = true;
        }
    }

    void exitTemporaryIdMode(DeviceIdType type, String deviceId) {
        L.d("[ModuleDeviceId] Calling exitTemporaryIdMode");

        if (!_int.isInitialized()) {
            throw new IllegalStateException("init must be called before exitTemporaryIdMode");
        }

        //start by changing stored ID
        _int.connectionQueue_.getDeviceId().changeToId(_int.context_, _int.connectionQueue_.getSharedPref(), type, deviceId);

        //update stored request for ID change to use this new ID
        String[] storedRequests = _int.connectionQueue_.getSharedPref().connections();
        String temporaryIdTag = "&deviceId=" + DeviceId.temporaryInTrackDeviceId;
        String newIdTag = "&deviceId=" + deviceId;

        boolean foundOne = false;
        for (int a = 0; a < storedRequests.length; a++) {
            if (storedRequests[a].contains(temporaryIdTag)) {
                L.d("[ModuleDeviceId] [exitTemporaryIdMode] Found a tag to replace in: [" + storedRequests[a] + "]");
                storedRequests[a] = storedRequests[a].replace(temporaryIdTag, newIdTag);
                foundOne = true;
            }
        }

        if (foundOne) {
            _int.connectionQueue_.getSharedPref().replaceConnections(storedRequests);
        }

        _int.doStoredRequests();
    }

    /**
     * Changes current device id type to the one specified in parameter. Closes current session and
     * reopens new one with new id. Doesn't merge user profiles on the server
     *
     * @param type     Device ID type to change to
     * @param deviceId Optional device ID for a case when type = DEVELOPER_SPECIFIED
     */
    void changeDeviceIdWithoutMerge(DeviceIdType type, String deviceId) {
        if (type == null) {
            throw new IllegalStateException("type cannot be null");
        }

        if (type == DeviceIdType.DEVELOPER_SUPPLIED && deviceId == null) {
            throw new IllegalStateException("WHen type is 'DEVELOPER_SUPPLIED', provided deviceId cannot be null");
        }

        if (!_int.anyConsentGiven() && type != DeviceIdType.TEMPORARY_ID) {
            //if we are not trying to set a temporary id, consent has to be given
            L.e("[ModuleDeviceId] Can't change Device ID if no consent is given");
            return;
        }

        DeviceId currentDeviceId = _int.connectionQueue_.getDeviceId();

        if (currentDeviceId.temporaryIdModeEnabled() && (deviceId != null && deviceId.equals(DeviceId.temporaryInTrackDeviceId))) {
            // we already are in temporary mode and we want to set temporary mode
            // in this case we just ignore the request since nothing has to be done
            return;
        }

        if (currentDeviceId.temporaryIdModeEnabled() || _int.connectionQueue_.queueContainsTemporaryIdItems()) {
            // we are about to exit temporary ID mode
            // because of the previous check, we know that the new type is a different one
            // we just call our method for exiting it
            // we don't end the session, we just update the device ID and connection queue
            exitTemporaryIdMode(type, deviceId);
        }

        // we are either making a simple ID change or entering temporary mode
        // in both cases we act the same as the temporary ID requests will be updated with the final ID later

        //force flush events so that they are associated correctly
        _int.sendEventsForced();

        _int.moduleSessions.endSessionInternal(currentDeviceId.getId());
        currentDeviceId.changeToId(_int.context_, _int.connectionQueue_.getSharedPref(), type, deviceId);
        _int.moduleSessions.beginSessionInternal();

        //clear automated star rating session values because now we have a new user
        _int.moduleRatings.clearAutomaticStarRatingSessionCountInternal(_int.connectionQueue_.getSharedPref());
    }

    /**
     * Changes current device id to the one specified in parameter. Merges user profile with new id
     * (if any) with old profile.
     *
     * @param deviceId new device id
     */
    void changeDeviceIdWithMerge(String deviceId) {
        if (deviceId == null || "".equals(deviceId)) {
            throw new IllegalStateException("deviceId cannot be null or empty");
        }

        if (!_int.anyConsentGiven()) {
            L.e("[ModuleDeviceId] Can't change Device ID if no consent is given");
            return;
        }

        if (_int.connectionQueue_.getDeviceId().temporaryIdModeEnabled() || _int.connectionQueue_.queueContainsTemporaryIdItems()) {
            //if we are in temporary ID mode or
            //at some moment have enabled temporary mode

            if (deviceId.equals(DeviceId.temporaryInTrackDeviceId)) {
                //if we want to enter temporary ID mode
                //just exit, nothing to do

                L.w("[ModuleDeviceId, changeDeviceId] About to enter temporary ID mode when already in it");

                return;
            }

            // if a developer supplied ID is provided
            //we just exit this mode and set the id to the provided one
            exitTemporaryIdMode(DeviceIdType.DEVELOPER_SUPPLIED, deviceId);
        } else {
            //we are not in temporary mode, nothing special happens
            // we are either making a simple ID change or entering temporary mode
            // in both cases we act the same as the temporary ID requests will be updated with the final ID later

            _int.connectionQueue_.changeDeviceId(deviceId, _int.moduleSessions.roundedSecondsSinceLastSessionDurationUpdate());
        }
    }

    @Override
    public void initFinished(Config config) {
        if (exitTempIdAfterInit) {
            L.i("[ModuleDeviceId, initFinished] Exiting temp ID at the end of init");
            exitTemporaryIdMode(DeviceIdType.DEVELOPER_SUPPLIED, config.deviceID);
        }
    }

    @Override
    void halt() {

    }
}
