package sdk.main.core;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

class ModuleDynamicConfig extends ModuleBase {
    DynamicConfig dynamicConfigInterface = null;

    ModuleLog L;

    ModuleDynamicConfig(CoreInternal coreInternal, Config config) {
        super(coreInternal);

        L = coreInternal.L;

        L.v("[ModuleDynamicConfig] Initialising");

        dynamicConfigInterface = new DynamicConfig();
    }

    /**
     * Internal call for updating dynamic config keys
     */
    void updateDynamicConfigValues(JSONObject newValues, Boolean merge) {
        L.d("[ModuleDynamicConfig] Updating dynamic config values");

        //merge the new values into the current ones
        DynamicConfigValueStore store = loadConfig();
        if (!merge) {
            //in case of full updates, clear old values
            store.values = new JSONObject();
        }
        store.mergeValues(newValues);

        CoreInternal.sharedInstance().L.d("[ModuleDynamicConfig] Finished dynamic config processing, starting saving");

        saveConfig(store);

        CoreInternal.sharedInstance().L.d("[ModuleDynamicConfig] Finished dynamic config saving");
    }

    Object getValue(String key) {
        DynamicConfigValueStore values = loadConfig();
        return values.getValue(key);
    }

    void saveConfig(DynamicConfigValueStore store) {
        _int.connectionQueue_.getSharedPref().setDynamicConfigValues(store.dataToString());
    }

    DynamicConfigValueStore loadConfig() {
        SharedPref cs = _int.connectionQueue_.getSharedPref();
        String valuesString = cs.getDynamicConfigValues();
        //noinspection UnnecessaryLocalVariable
        DynamicConfigValueStore values = DynamicConfigValueStore.dataFromString(valuesString);
        return values;
    }

    void clearValueStore() {
        _int.connectionQueue_.getSharedPref().setDynamicConfigValues("");
    }

    Map<String, Object> getAllDynamicConfigValuesInternal() {
        DynamicConfigValueStore values = loadConfig();
        return values.getAllValues();
    }

    @Override
    public void initFinished(Config config) {
        L.d("[Init] Automatically updating dynamic config values");
        MqttController.INSTANCE.init(config.context.getApplicationContext());
    }

    @Override
    public void halt() {
        dynamicConfigInterface = null;
    }

    static class DynamicConfigValueStore {
        public JSONObject values = new JSONObject();

        private DynamicConfigValueStore(JSONObject values) {
            this.values = values;
        }

        public static DynamicConfigValueStore dataFromString(String storageString) {
            if (storageString == null || storageString.isEmpty()) {
                return new DynamicConfigValueStore(new JSONObject());
            }

            JSONObject values;
            try {
                values = new JSONObject(storageString);
            } catch (JSONException e) {
                CoreInternal.sharedInstance().L.e("[DynamicConfigValueStore] Couldn't decode DynamicConfigValueStore successfully: " + e.toString());
                values = new JSONObject();
            }
            return new DynamicConfigValueStore(values);
        }

        //add new values to the current storage
        public void mergeValues(JSONObject newValues) {
            if (newValues == null) {
                return;
            }

            Iterator<String> iter = newValues.keys();
            while (iter.hasNext()) {
                String key = iter.next();
                try {
                    Object value = newValues.get(key);
                    values.put(key, value);
                } catch (Exception e) {
                    CoreInternal.sharedInstance().L.e("[DynamicConfigValueStore] Failed merging new dynamic config values");
                }
            }
        }

        public Object getValue(String key) {
            return values.opt(key);
        }

        public Map<String, Object> getAllValues() {
            Map<String, Object> ret = new HashMap<>();

            Iterator<String> keys = values.keys();

            while (keys.hasNext()) {
                String key = keys.next();

                try {
                    ret.put(key, values.get(key));
                } catch (Exception ex) {
                    CoreInternal.sharedInstance().L.e("[DynamicConfigValueStore] Got JSON exception while calling 'getAllValues': " + ex.toString());
                }
            }

            return ret;
        }

        public String dataToString() {
            return values.toString();
        }
    }

    public class DynamicConfig {
        public Map<String, Object> getAllValues() {
            synchronized (_int) {
                L.i("[DynamicConfig] Calling 'getAllValues'");
                return getAllDynamicConfigValuesInternal();
            }
        }
    }
}
