package sdk.main.core;

import android.util.Log;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Date;
import java.util.Iterator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

class ModuleLog {
    private final Boolean enableInternalLog = true;
    LogCallback logListener = null;

    ModuleLog() {
    }

    void SetListener(LogCallback logListener) {
        this.logListener = logListener;
    }

    public void v(String msg) {
        sendLog(msg, LogLevel.Verbose);
        if (!enableInternalLog || !logEnabled()) {
            return;
        }
        if (CoreInternal.sharedInstance().isLoggingEnabled()) {
            Log.v(CoreInternal.TAG, msg);
        }
        informListener(msg, null, LogLevel.Verbose);
    }

    public void d(String msg) {
        sendLog(msg, LogLevel.Debug);
        if (!enableInternalLog || !logEnabled()) {
            return;
        }
        if (CoreInternal.sharedInstance().isLoggingEnabled()) {
            Log.d(CoreInternal.TAG, msg);
        }
        informListener(msg, null, LogLevel.Debug);
    }

    public void i(String msg) {
        sendLog(msg, LogLevel.Info);
        if (!enableInternalLog || !logEnabled()) {
            return;
        }
        if (CoreInternal.sharedInstance().isLoggingEnabled()) {
            Log.i(CoreInternal.TAG, msg);
        }
        informListener(msg, null, LogLevel.Info);
    }

    public void w(String msg) {
        w(msg, null);
    }

    public void w(String msg, Throwable t) {
        sendLog(msg, LogLevel.Warning);
        if (!logEnabled()) {
            return;
        }
        if (CoreInternal.sharedInstance().isLoggingEnabled()) {
            Log.w(CoreInternal.TAG, msg);
        }
        informListener(msg, null, LogLevel.Warning);
    }

    public void e(String msg) {
        e(msg, null);
    }

    public void e(String msg, Throwable t) {
        sendLog(msg, LogLevel.Error);
        if (!logEnabled()) {
            return;
        }
        if (CoreInternal.sharedInstance().isLoggingEnabled()) {
            Log.e(CoreInternal.TAG, msg, t);
        }
        informListener(msg, t, LogLevel.Error);
    }

    public boolean logEnabled() {
        return (logListener != null) || CoreInternal.sharedInstance().isLoggingEnabled();
    }

    private void sendLog(String msg, LogLevel level) {
        try {
            int serverLogLevel = (int) CoreInternal.sharedInstance().moduleDynamicConfig.getValue("logLevel");
            if (level.ordinal() >= serverLogLevel) {
                sendLog("[LEVEL " + level.toString() + "] " + msg);
            }
        } catch (Exception ignored) {
        }
    }

    public void sendLog(String msg) {
        try {
            Boolean serverLogIsEnabled = (Boolean) CoreInternal.sharedInstance().moduleDynamicConfig.getValue("logIsEnable");
            if (serverLogIsEnabled) {
                JSONObject log = new JSONObject();
                try {
                    log.put("message", msg);
                } catch (Exception ignored) {
                }

                sendLog(log);
            }
        } catch (Exception ignored) {
        }
    }

    public void sendLog(JSONObject log) {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.execute(() -> {
            HttpURLConnection urlConnection = null;
            try {
                URL url = new URL("https://kilo.intrack.ir/loki/api/v1/push");
                urlConnection = (HttpURLConnection) url.openConnection();
                urlConnection.setRequestMethod("POST");
                urlConnection.setRequestProperty("Content-Type", "application/json");
                urlConnection.setDoOutput(true);

                long now = (new Date()).getTime() * 1000000;

                JSONObject req = new JSONObject();
                JSONArray streams = new JSONArray();
                JSONObject stream = new JSONObject();
                JSONObject streamData = new JSONObject();
                JSONArray streamValues = new JSONArray();
                JSONArray streamValue = new JSONArray();
                JSONObject logObject = new JSONObject();

                logObject.put("sdk_version", CoreInternal.sharedInstance().INTRACK_SDK_VERSION_STRING);
                String userId = CoreInternal.sharedInstance().getUserId();
                if (userId != null) {
                    logObject.put("user_id", userId);
                }
                String deviceId = CoreInternal.sharedInstance().getDeviceID();
                if (deviceId != null) {
                    logObject.put("device_id", deviceId);
                }
                String appKey = CoreInternal.sharedInstance().getConfig().appKey;
                if (appKey != null) {
                    logObject.put("app_key", appKey);
                }

                Iterator<String> logKeys = log.keys();
                while (logKeys.hasNext()) {
                    String key = logKeys.next();
                    logObject.put(key, log.get(key));
                }

                streamValue.put(Long.toString(now));
                streamValue.put(logObject.toString());
                streamValues.put(streamValue);

                streamData.put("sdk", "InTrack_" + CoreInternal.sharedInstance().INTRACK_SDK_NAME);

                stream.put("stream", streamData);
                stream.put("values", streamValues);

                streams.put(stream);
                req.put("streams", streams);

                String jsonInputString = req.toString();

                try (OutputStream os = urlConnection.getOutputStream()) {
                    byte[] input = jsonInputString.getBytes("utf-8");
                    os.write(input, 0, input.length);
                }

                int code = urlConnection.getResponseCode();

            } catch (Exception ignored) {
                ignored.printStackTrace();
            } finally {
                if (urlConnection != null) {
                    urlConnection.disconnect();
                }
            }
        });
    }

    private void informListener(String msg, final Throwable t, final LogLevel level) {
        try {
            if (msg == null) {
                msg = "";
            }
            if (t != null) {
                msg += Log.getStackTraceString(t);
            }

            if (logListener != null) {
                logListener.LogHappened(msg, level.toString());
            }
        } catch (Exception ex) {
            Log.e(CoreInternal.TAG, "[ModuleLog] Failed to inform listener [" + ex.toString() + "]");
        }
    }

    public enum LogLevel {Verbose, Debug, Info, Warning, Error}
}
