package sdk.main.core;

import static sdk.main.core.Utils.extractNestedMap;

import android.content.Context;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONObject;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ir.intrack.android.sdk.InTrackPush;
import sdk.main.core.pushmessaging.CountDownType;
import sdk.main.core.pushmessaging.MessageType;
import sdk.main.core.pushmessaging.PushMessage;
import sdk.main.core.pushmessaging.PushMessageButton;
import sdk.main.core.pushmessaging.TimerTemplate;
import sdk.main.core.pushmessaging.TimerType;

class ModulePush {

    public static final String FCM_REGISTER_EVENT_KEY = "fcm_registered";
    public static final String PUSH_DELIVERY_EVENT_KEY = "push_notification_received";
    public static final String PUSH_CLICK_EVENT_KEY = "push_notification_click";
    public static final String PUSH_REJECTED_EVENT_KEY = "push_notification_rejected";
    public static final String PUSH_DISMISSED_EVENT_KEY = "push_notification_dismissed";
    public static final String PUSH_EVENT_ACTION_ID_KEY = "i";
    public static final String PUSH_EVENT_ACTION_INDEX_KEY = "b";
    static final String KEY_ID = "id";
    static final String KEY_TRACKING_ID = "trackingId";
    static final String KEY_TITLE = "title";
    static final String KEY_TITLE_COLOR = "titleColor";
    static final String KEY_MESSAGE = "message";
    static final String KEY_MESSAGE_COLOR = "messageColor";
    static final String KEY_SOUND = "sound";
    static final String KEY_BADGE = "badge";
    static final String KEY_LINK = "link";
    static final String KEY_MEDIA = "media";
    static final String KEY_STICKY = "sticky";
    static final String KEY_ANDROID_DISMISS_BUTTON_LABEL = "dismissLabel";
    static final String KEY_AUTO_DISMISS_DURATION = "autoDismissDuration";
    static final String KEY_TYPE = "type";
    static final String KEY_TIMER_TYPE = "type";
    static final String KEY_COUNTDOWN_TYPE = "countDownType";
    static final String KEY_DURATION = "duration";
    static final String KEY_TIME_COLOR = "timeColor";
    static final String KEY_PROGRESS_BAR_COLOR = "progressBarColor";
    static final String KEY_END_TIME = "endTime";
    static final String KEY_BUTTONS = "buttons";
    static final String KEY_BUTTONS_TITLE = "t";
    static final String KEY_BUTTONS_LINK = "l";
    static final String KEY_CUSTOM_DATA = "customData";

    static class MessageImpl implements InTrackPush.Message, Parcelable {

        private final String id;
        private final String title;
        private final String titleColor;
        private final String message;
        private final String messageColor;
        private final String sound;
        private final String dismissLabel;
        private final Integer badge;
        private final Uri link;
        private final URL media;
        private final boolean sticky;
        private final Long autoDismissDuration;
        private final MessageType type;
        @Nullable
        private final TimerTemplate timerTemplate;
        private final List<PushMessageButton> buttons;
        private final Map<String, String> data;
        private final Map<String, Object> customData;

        public MessageImpl(Map<String, String> data) {
            this.data = data;

            this.id = data.get(KEY_ID);
            this.title = data.get(KEY_TITLE);
            this.titleColor = data.get(KEY_TITLE_COLOR);
            this.message = data.get(KEY_MESSAGE);
            this.messageColor = data.get(KEY_MESSAGE_COLOR);
            this.sound = data.get(KEY_SOUND);
            this.dismissLabel = data.get(KEY_ANDROID_DISMISS_BUTTON_LABEL);
            this.sticky = parseBoolean(data.get(KEY_STICKY));
            this.autoDismissDuration = parseLong(data.get(KEY_AUTO_DISMISS_DURATION));
            this.type = MessageType.Companion.fromString(data.get(KEY_TYPE));
            Map<String, String> timerData = extractNestedMap(data.get("timerTemplate"));
            this.timerTemplate = parseTimerTemplate(timerData);
            this.badge = parseInteger(data.get(KEY_BADGE));
            this.link = parseUri(data.get(KEY_LINK));
            this.media = parseUrl(data.get(KEY_MEDIA));
            this.buttons = parseButtons(data.get(KEY_BUTTONS));
            this.customData = parseCustomData(data.get(KEY_CUSTOM_DATA));

            CoreInternal.sharedInstance().L.d("[MessageImpl] constructed: " + id);
        }

        @Override
        public int hashCode() {
            return id != null ? id.hashCode() : super.hashCode();
        }

        // Parsing Helpers
        private boolean parseBoolean(String value) {
            return value != null && Boolean.parseBoolean(value);
        }

        private Long parseLong(String value) {
            try {
                return value != null ? Long.parseLong(value) : null;
            } catch (NumberFormatException e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse long: " + value, e);
                return null;
            }
        }

        private Integer parseInteger(String value) {
            try {
                return value != null ? Integer.parseInt(value) : null;
            } catch (NumberFormatException e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse integer: " + value, e);
                return null;
            }
        }

        private Uri parseUri(String value) {
            try {
                return value != null ? Uri.parse(value) : null;
            } catch (Exception e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse URI: " + value, e);
                return null;
            }
        }

        private URL parseUrl(String value) {
            try {
                return value != null ? new URL(value) : null;
            } catch (MalformedURLException e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse URL: " + value, e);
                return null;
            }
        }

        /**
         * Parses a TimerTemplate object from the provided data map.
         */
        private TimerTemplate parseTimerTemplate(Map<String, String> data) {
            try {
                TimerType type = TimerType.Companion.fromString(data.get(KEY_TIMER_TYPE));
                CountDownType countDownType = CountDownType.Companion.fromString(data.get(KEY_COUNTDOWN_TYPE));
                Long duration = parseLong(data.get(KEY_DURATION));
                String endTime = data.get(KEY_END_TIME) != null ? data.get(KEY_END_TIME) : null;
                String timeColor = data.get(KEY_TIME_COLOR);
                String progressBarColor = data.get(KEY_PROGRESS_BAR_COLOR);

                return new TimerTemplate(type, countDownType, duration, endTime, timeColor, progressBarColor);
            } catch (Exception e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse TimerTemplate", e);
                return null;
            }
        }

        private List<PushMessageButton> parseButtons(String json) {
            List<PushMessageButton> buttonList = new ArrayList<>();
            if (json == null || json.isEmpty()) return buttonList;

            try {
                JSONArray array = new JSONArray(json);
                for (int i = 0; i < array.length(); i++) {
                    JSONObject btn = array.getJSONObject(i);
                    String title = btn.optString(KEY_BUTTONS_TITLE, null);
                    Uri link = parseUri(btn.optString(KEY_BUTTONS_LINK, null));
                    if (title != null && link != null) {
                        buttonList.add(new Button(this, i + 1, title, link));
                    }
                }
            } catch (Exception e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse buttons JSON", e);
            }

            return buttonList;
        }


        private Map<String, Object> parseCustomData(String rawCustomData) {
            if (rawCustomData == null || rawCustomData.isEmpty()) return null;

            try {
                JSONObject jsonObject = new JSONObject(rawCustomData);
                return Utils.mapifyJsonObject(jsonObject);
            } catch (Exception e) {
                CoreInternal.sharedInstance().L.w("[MessageImpl] Failed to parse custom data JSON", e);
                return null;
            }
        }

        // PushMessage Interface Implementations
        @Override
        public String getId() {
            return id;
        }

        @Override
        public String getTitle() {
            return title;
        }

        @Override
        public String getMessage() {
            return message;
        }

        @Override
        public String getSound() {
            return sound;
        }

        @Override
        public Integer getBadge() {
            return badge;
        }

        @Override
        public Uri getLink() {
            return link;
        }

        @Override
        public URL getMedia() {
            return media;
        }

        @Override
        public List<PushMessageButton> getButtons() {
            return buttons;
        }

        @Override
        public Set<String> getDataKeys() {
            return data.keySet();
        }

        @Override
        public Map<String, Object> getCustomData() {
            return customData;
        }

        @Override
        public boolean has(String key) {
            return data.containsKey(key);
        }

        @Override
        public String data(String key) {
            return data.get(key);
        }

        @Override
        public boolean getSticky() {
            return sticky;
        }

        @Override
        public Long getAutoDismissDuration() {
            return autoDismissDuration;
        }

        @Override
        public MessageType getType() {
            return type;
        }

        @Override
        public TimerTemplate getTimerTemplate() {
            return timerTemplate;
        }

        @Override
        public String getDismissLabel() {
            return dismissLabel;
        }

        @Nullable
        @Override
        public String getTitleColor() {
            return titleColor;
        }

        @Nullable
        @Override
        public String getMessageColor() {
            return messageColor;
        }

        @Override
        public void recordAction(Context context) {
            recordAction(context, 0);
        }

        @Override
        public void recordAction(Context context, int buttonIndex) {
            if (CoreInternal.sharedInstance().isInitialized()) {
                Map<String, Object> eventData = new HashMap<>();
                eventData.put(PUSH_EVENT_ACTION_ID_KEY, id);
                eventData.put(PUSH_EVENT_ACTION_INDEX_KEY, String.valueOf(buttonIndex));
                CoreInternal.sharedInstance().events().recordSystemEvent(PUSH_CLICK_EVENT_KEY, eventData);
            } else {
                SharedPref.cacheEventClick(id, String.valueOf(buttonIndex), new Date().getTime(), context);
            }
        }

        // Button Inner Class
        static class Button implements PushMessageButton {
            private final PushMessage message;
            private final int index;
            private final int icon;
            private final String title;
            private final Uri link;


            Button(PushMessage message, int index, String title, Uri link) {
                this.message = message;
                this.index = index;
                this.title = title;
                this.link = link;
                this.icon = 0;
            }

            @Override
            public int getIndex() {
                return index;
            }

            @Override
            public String getTitle() {
                return title;
            }

            @Override
            public Uri getLink() {
                return link;
            }

            @Override
            public int getIcon() {

                return icon;
            }

            @Override
            public void recordAction(Context context) {
                message.recordAction(context, index);
            }
        }

        // Parcelable Implementation
        @Override
        public int describeContents() {
            return 0;
        }

        @Override
        public void writeToParcel(Parcel dest, int flags) {
            dest.writeMap(data);
            CoreInternal.sharedInstance().L.d("[MessageImpl] written: " + data.get(KEY_ID));
        }

        public static final Creator<MessageImpl> CREATOR = new Creator<MessageImpl>() {
            @Override
            public MessageImpl createFromParcel(Parcel source) {
                Map<String, String> map = new HashMap<>();
                source.readMap(map, Map.class.getClassLoader());
                return new MessageImpl(map);
            }

            @Override
            public MessageImpl[] newArray(int size) {
                return new MessageImpl[size];
            }
        };
    }
}
