package sdk.main.core;

import static java.util.Objects.isNull;

import android.app.*;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;
import androidx.work.*;

import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;

import java.net.HttpURLConnection;
import java.net.URL;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import ir.intrack.android.sdk.BuildConfig;
import sdk.main.core.pushmessaging.*;

/**
 * Just a public holder class for Messaging-related display logic, listeners, managers, etc.
 */
public class PushHandler {
    public static final String EXTRA_ACTION_INDEX = packagePrefixBasedOnFlavor() + ".android.sdk.PushHandler.Action";
    public static final String EXTRA_MESSAGE = packagePrefixBasedOnFlavor() + ".android.sdk.PushHandler.message";
    public static final String EXTRA_INTENT = packagePrefixBasedOnFlavor() + ".android.sdk.PushHandler.intent";
    public static final String INTRACK_BROADCAST_PERMISSION_POSTFIX = ".PushHandler.BROADCAST_PERMISSION";
    public static final String SECURE_NOTIFICATION_BROADCAST = MessageIntentHandler.SECURE_NOTIFICATION_BROADCAST;
    protected static final String PUSH_AMP_CHANNEL_ID = "INT_TMP_MARKETING_AMPLIFIER";
    protected static final String DEFAULT_PUSH_ICON_META = packagePrefixBasedOnFlavor() + ".sdk.default_notification_icon";
    private static Application.ActivityLifecycleCallbacks callbacks = null;
    private static Activity activity = null;

    private static CoreProxy.MessagingProvider provider = null;

    static Integer notificationAccentColor = null;

    public static boolean useAdditionalIntentRedirectionChecks = false;

    @SuppressWarnings("FieldCanBeLocal")
    private static BroadcastReceiver consentReceiver = null;


    private static String packagePrefixBasedOnFlavor() {
        return "ir.intrack";
    }

    /**
     * Listens for push consent given and sends existing token to the server if any.
     */
    protected static class ConsentBroadcastReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent broadcast) {
            if (provider != null && getPushConsent(context)) {
                String token = getToken(context, provider);
                if (token != null && !"".equals(token)) {
                    onTokenRefresh(token, provider);
                }
            }
        }
    }

    /**
     * Retrieves current FCM token from FIREBASE_INSTANCEID_CLASS.
     *
     * @return token string or null if no token is currently available.
     */
    private static String getToken(Context context, CoreProxy.MessagingProvider prov) {

        switch (provider) {
            case FCM:
                return getFcmToken();
            case HMS:
                return getHmsToken(context);
            default:
                CoreInternal.sharedInstance().L.e("[PushHandler, getToken] Unsupported messaging provider: " + provider);
                return null;
        }
    }

    private static String getFcmToken() {
        try {
            // Try the modern FirebaseMessaging API
            Object firebaseMessagingInstance = UtilsMessaging.reflectiveCall(FIREBASE_MESSAGING_CLASS, null, "getInstance");
            Task<String> tokenTask = (Task<String>) UtilsMessaging.reflectiveCall(FIREBASE_MESSAGING_CLASS, firebaseMessagingInstance, "getToken");
            return Tasks.await(tokenTask);
        } catch (Exception ignored) {
            // Fall back to the older FirebaseInstanceId API
            try {
                Object instance = UtilsMessaging.reflectiveCall(FIREBASE_INSTANCEID_CLASS, null, "getInstance");
                return (String) UtilsMessaging.reflectiveCall(FIREBASE_INSTANCEID_CLASS, instance, "getToken");
            } catch (Throwable logged) {
                CoreInternal.sharedInstance().L.e("[PushHandler, getToken] Couldn't get token for [" + BuildConfig.FLAVOR + "] FCM", logged);
                return null;
            }
        }
    }

    private static String getHmsToken(Context context) {
        try {
            // Get Huawei configuration
            Object config = UtilsMessaging.reflectiveCallStrict(HUAWEI_CONFIG_CLASS, null, "fromContext", context, Context.class);
            if (config == null) {
                CoreInternal.sharedInstance().L.e("No Huawei Config");
                return null;
            }

            // Retrieve Huawei app ID
            Object appId = UtilsMessaging.reflectiveCall(HUAWEI_CONFIG_CLASS, config, "getString", "client/app_id");
            if (appId == null || "".equals(appId)) {
                CoreInternal.sharedInstance().L.e("No Huawei app id in config");
                return null;
            }

            // Get instance and retrieve the token
            Object instanceId = UtilsMessaging.reflectiveCallStrict(HUAWEI_INSTANCEID_CLASS, null, "getInstance", context, Context.class);
            if (instanceId == null) {
                CoreInternal.sharedInstance().L.e("No Huawei instance id class");
                return null;
            }

            Object token = UtilsMessaging.reflectiveCall(HUAWEI_INSTANCEID_CLASS, instanceId, "getToken", appId, "HCM");
            return (String) token;
        } catch (Throwable logged) {
            CoreInternal.sharedInstance().L.e("[PushHandler, getToken] Couldn't get token for [" + BuildConfig.FLAVOR + "] Huawei Push Kit", logged);
            return null;
        }
    }

    /**
     * Standard logic for displaying a {@link PushMessage}
     *
     * @param context context to run in (supposed to be called from {@code FirebaseMessagingService})
     * @param data    {@code RemoteMessage#getData()} result
     * @return {@code Boolean.TRUE} if displayed successfully, {@code Boolean.FALSE} if cannot display now, {@code null} if no message is found in {@code data}
     */
    protected static Boolean displayMessage(Context context, final Map<String, String> data, final String channelId, final int notificationSmallIcon, final Intent notificationIntent) {
        return displayMessage(context, decodeMessage(data), channelId, notificationSmallIcon, notificationIntent);
    }

    /**
     * Standard logic for displaying a {@link PushMessage}
     *
     * @param context   context to run in (supposed to be called from {@code FirebaseMessagingService})
     * @param msg       {@link PushMessage} instance
     * @param channelId notification channel on which the notification should be displayed
     * @return {@code Boolean.TRUE} if displayed successfully, {@code Boolean.FALSE} if cannot display now, {@code null} if no message is found in {@code data}
     */
    protected static Boolean displayMessage(final Context context, final PushMessage msg, final String channelId, final int notificationSmallIcon, final Intent notificationIntent) {
        CoreInternal.sharedInstance().L.d("[PushHandler, displayMessage] Displaying push message" + msg);

        updatePushChannel(channelId, context);

        if (msg == null || msg.getMessage() == null) {
            CoreInternal.sharedInstance().L.d("[PushHandler, displayMessage] No message to display.");
            return null;
        }
        // Record push delivery event
        sendDeliveryEvent(msg.getId());

        recordPushReceivedEvent(msg, context);
        return displayNotification(context, msg, channelId, notificationSmallIcon, notificationIntent);
    }

    /**
     * Updates the latest push channel in shared preferences.
     */
    private static void updatePushChannel(String channelId, Context context) {
        if (!Objects.equals(channelId, PUSH_AMP_CHANNEL_ID)) {
            SharedPref.setLatestPushChannel(channelId, context);
        }
    }

    private static void sendDeliveryEvent(String messageId) {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.execute(() -> {
            HttpURLConnection urlConnection = null;
            try {
                URL url = new URL("https://api.intrack.ir/api/sdk/track/delivered?token=" + messageId);
                urlConnection = (HttpURLConnection) url.openConnection();
                int responseCode = urlConnection.getResponseCode();
                CoreInternal.sharedInstance().L.d("[PushHandler, sendDeliveryEvent]  Delivery Event sent. Response code: " + responseCode);
            } catch (Exception e) {
                CoreInternal.sharedInstance().L.e("[PushHandler, sendDeliveryEvent] Failed to send delivery event.", e);
            } finally {
                if (urlConnection != null) {
                    urlConnection.disconnect();
                }
            }
        });
    }

    private static void recordPushReceivedEvent(PushMessage msg, Context context) {
        if (CoreInternal.sharedInstance().isInitialized()) {
            Map<String, Object> pushDeliveryData = new HashMap<>();
            pushDeliveryData.put("i", msg.getId());
            CoreInternal.sharedInstance().events().recordSystemEvent(ModulePush.PUSH_DELIVERY_EVENT_KEY, pushDeliveryData);
            CoreInternal.sharedInstance().sendEventsForced();
        } else {
            SharedPref.cacheEventDelivery(msg.getId(), new Date().getTime(), context);
        }

    }

    public static void recordPushDismissedEvent(String msgId, Context context) {
        if (CoreInternal.sharedInstance().isInitialized()) {
            Map<String, Object> pushDismissData = new HashMap<>();
            pushDismissData.put("i", msgId);
            CoreInternal.sharedInstance().events().recordSystemEvent(ModulePush.PUSH_DISMISSED_EVENT_KEY, pushDismissData);
            CoreInternal.sharedInstance().sendEventsForced();
        } else {
            SharedPref.cacheEventDismiss(msgId, new Date().getTime(), context);
        }
    }

    /**
     * Standard logic for displaying a {@link Notification} based on the {@link PushMessage}
     *
     * @param context               context to run in
     * @param msg                   message to get information from
     * @param channelId             notification channel on which the notification should be displayed
     * @param notificationSmallIcon smallIcon for notification {@link Notification#getSmallIcon()}
     * @param notificationIntent    activity-starting intent to send when user taps on {@link Notification} or one of its {@link Notification.Action}s. Pass {@code null} to go with main activity.
     * @return {@code Boolean.TRUE} if displayed successfully, {@code Boolean.FALSE} if cannot display now, {@code null} if message is not displayable as {@link Notification}
     */
    protected static Boolean displayNotification(final Context context, final PushMessage msg, final String channelId, final int notificationSmallIcon, final Intent notificationIntent) {
        if (!getPushConsent(context)) {
            return null;
        }

        initializeConsentReceiver(context);
        isValidMessage(msg);

        CoreInternal.sharedInstance().L.d("[PushHandler, displayNotification] Displaying push notification, additional intent provided:[" + (notificationIntent != null) + "]");

        final NotificationManager manager = getNotificationManager(context);

        if (manager == null) {
            return Boolean.FALSE;
        }

        if (!areNotificationsEnabled(context, manager, msg)) {
            return Boolean.FALSE;
        }


        NotificationCompat.Builder builder = buildNotificationBuilder(context, channelId, notificationSmallIcon, msg, notificationIntent);
        if (builder != null) {
            if (msg.getSound() != null) {
                if (msg.getSound().equals("default")) {
                    builder.setDefaults(Notification.DEFAULT_SOUND);
                } else {
                    builder.setSound(Uri.parse(msg.getSound()));
                }
            }

            handleNotificationMedia(builder, msg);

            if (msg.getAutoDismissDuration() != null && msg.getAutoDismissDuration() != -1) {
                autoDismissStickyNotification(msg, manager);
            }

            manager.notify(msg.hashCode(), builder.build());
            return Boolean.TRUE;
        } else {

            return Boolean.FALSE;
        }
    }

    private static void autoDismissStickyNotification(PushMessage msg, NotificationManager manager) {
        // If autoDismissDuration is not null, set a delay to dismiss the notification
        if (msg.getAutoDismissDuration() != null && msg.getAutoDismissDuration() != -1) {
            long delayInMillis = msg.getAutoDismissDuration() * 1000;
            new Handler(Looper.getMainLooper()).postDelayed(() -> {
                manager.cancel(msg.hashCode()); // Dismiss the notification
            }, delayInMillis);
        }
    }

    private static void initializeConsentReceiver(Context context) {
        if (consentReceiver != null) return;

        try {
            IntentFilter filter = new IntentFilter();
            filter.addAction(CoreInternal.CONSENT_BROADCAST);
            consentReceiver = new ConsentBroadcastReceiver();

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                context.registerReceiver(
                    consentReceiver,
                    filter,
                    context.getPackageName() + INTRACK_BROADCAST_PERMISSION_POSTFIX,
                    null,
                    Context.RECEIVER_EXPORTED
                );
            } else {
                context.registerReceiver(
                    consentReceiver,
                    filter,
                    context.getPackageName() + INTRACK_BROADCAST_PERMISSION_POSTFIX,
                    null
                );
            }
        } catch (Exception ignored) {
            CoreInternal.sharedInstance().L.e("[PushHandler, displayNotification] initializeConsentReceiver" + ignored.getMessage());
        }

    }

    private static boolean isValidMessage(PushMessage msg) {
        if (msg == null) {
            CoreInternal.sharedInstance().L.w("[PushHandler, displayNotification] Message is 'null'");
            return false;
        }

        if (msg.getTitle() == null && msg.getMessage() == null) {
            CoreInternal.sharedInstance().L.w("[PushHandler, displayNotification] Message title and body are 'null'");
            return false;
        }

        return true;
    }

    private static NotificationManager getNotificationManager(Context context) {
        NotificationManager manager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        if (manager == null) {
            CoreInternal.sharedInstance().L.w("[PushHandler, displayNotification] Notification manager is 'null'");
        }
        return manager;
    }

    private static boolean areNotificationsEnabled(Context context, NotificationManager manager, PushMessage msg) {
        if (Build.VERSION.SDK_INT > 24 && !manager.areNotificationsEnabled()) {
            recordPushRejectionEvent(context, msg);
            return false;
        }
        return true;

    }

    private static void recordPushRejectionEvent(Context context, PushMessage msg) {
        if (CoreInternal.sharedInstance().isInitialized()) {
            Map<String, Object> pushDeliveryData = new HashMap<>();
            pushDeliveryData.put("i", msg.getId());
            CoreInternal.sharedInstance().events().recordSystemEvent(ModulePush.PUSH_REJECTED_EVENT_KEY, pushDeliveryData);
        } else {
            SharedPref.cacheEventRejection(msg.getId(), new Date().getTime(), context);
        }
    }

    /**
     * Configures the countdown duration or end time for the TimerNotificationHelper.
     *
     * @param timerTemplate The TimerTemplate containing configuration details
     */
    private static long configureTimerDuration(TimerTemplate timerTemplate) {
        long endTime = 0;
        if (CountDownType.DURATION.equals(timerTemplate.getCountDownType())) {
            endTime = timerTemplate.getDuration() != null ? (timerTemplate.getDuration() * 1000) : 0;
        } else if (CountDownType.END_TIME.equals(timerTemplate.getCountDownType())) {
            endTime = UtilsTime.calculateRemainingTimeInMillis(timerTemplate.getEndTime());
        }
        return endTime;
    }

    private static @Nullable NotificationCompat.Builder buildNotificationBuilder(
        Context context,
        String channelId,
        int smallIcon,
        PushMessage msg,
        Intent notificationIntent
    ) {
        // Builder for Notification based on SDK version
        NotificationCompat.Builder builder = (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
            ? new NotificationCompat.Builder(context, channelId)
            : new NotificationCompat.Builder(context));

        // Common properties
        builder.setAutoCancel(true)
            .setSmallIcon(smallIcon)
            .setTicker(msg.getMessage())
            .setContentTitle(msg.getTitle())
            .setContentText(msg.getMessage());

        if (msg.getSticky()) {
            builder.setOngoing(true);
        }

        // Conditional display logic based on message type
        switch (msg.getType()) {
            case TEXT:
                //For TEXT type, show basic notification without remoteViews
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    builder.setStyle(new NotificationCompat.BigTextStyle()
                        .bigText(msg.getMessage())
                        .setBigContentTitle(msg.getTitle()));
                }
                break;

            case TIMER:
                TimerTemplate timerTemplate = msg.getTimerTemplate();
                if (timerTemplate != null) {
                    long duration = configureTimerDuration(timerTemplate);
                    if (duration <= 0) {
                        CoreInternal.sharedInstance().L.e("[PushHandler] Timer notification skipped: duration is <= 0.");
                        return null;
                    }

                    buildTimerNotification(context, msg, channelId, smallIcon, notificationIntent);
                } else {
                    CoreInternal.sharedInstance().L.e("[PushHandler] Timer notification skipped: template is null.");
                    return null;
                }
                break;
            case BANNER:
                // For BANNER type, show BigPictureStyle with an image
                handleNotificationMedia(builder, msg);

                break;
            default:
                // Fallback for unsupported or null types
                builder.setContentText("Unsupported message type");
                break;
        }
        // Set accent color for supported API levels
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP && notificationAccentColor != null) {
            builder.setColor(notificationAccentColor);
        }

        NotificationUtils.INSTANCE.setNotificationClickAction(context, builder, msg, msg.hashCode(), notificationIntent);
        NotificationUtils.INSTANCE.setNotificationButtons(context, builder, msg, msg.hashCode(), notificationIntent);
        return builder;
    }

    private static void buildTimerNotification(Context context, PushMessage msg,
                                               String channelId,
                                               int smallIcon, Intent notificationIntent) {
        TimerTemplate timerTemplate = msg.getTimerTemplate();

        if (timerTemplate != null && timerTemplate.getType() != null) {
            long duration = configureTimerDuration(timerTemplate);
            Intent serviceIntent = new Intent(context, TimerNotificationService.class)
                .putExtra(TimerNotificationService.EXTRA_NOTIFICATION, msg)
                .putExtra(TimerNotificationService.EXTRA_NOTIFICATION_ID, msg.hashCode())
                .putExtra(TimerNotificationService.EXTRA_NOTIFICATION_INTENT, notificationIntent)
                .putExtra(TimerNotificationService.EXTRA_SMALL_ICON, smallIcon)
                .putExtra(TimerNotificationService.EXTRA_CHANNEL_ID, channelId)
                .putExtra(TimerNotificationService.EXTRA_TIMER_DURATION, duration)
                .putExtra(TimerNotificationService.EXTRA_PROGRESS_DURATION, duration);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                context.startForegroundService(serviceIntent);
            } else {
                context.startService(serviceIntent);
            }
        }
    }

    private static void handleNotificationMedia(NotificationCompat.Builder builder, PushMessage msg) {
        if (msg.getMedia() != null) {
            try {
                ImageDownload imageDownload = Utils.startImageDownloadInBackground(msg.getMedia().toString());
                Utils.waitForAndApplyImageDownload(builder, msg, imageDownload);
            } catch (Exception e) {
                e.printStackTrace();
                // Report event
            }
        }
    }

    /**
     * Decode message from {@code RemoteMessage#getData()} map into {@link PushMessage}.
     *
     * @param data map to decode
     * @return message instance or {@code null} if cannot decode
     */
    public static PushMessage decodeMessage(Map<String, String> data) {
        ModulePush.MessageImpl message = new ModulePush.MessageImpl(data);
        return message.getId() == null ? null : message;
    }

    /**
     * Token refresh callback to be called from {@code FirebaseInstanceIdService}.
     *
     * @param token String token to be sent to server
     */
    protected static void onTokenRefresh(String token) {
        onTokenRefresh(token, CoreProxy.MessagingProvider.FCM);
    }

    /**
     * Token refresh callback to be called from {@code FirebaseInstanceIdService}.
     *
     * @param token    String token to be sent to server
     * @param provider which provider the token belongs to
     */
    protected static void onTokenRefresh(String token, CoreProxy.MessagingProvider provider) {
        if (!CoreInternal.sharedInstance().isInitialized()) {
            //is some edge cases this might be called before the SDK is initialized
            CoreInternal.sharedInstance().L.i("[PushHandler, onTokenRefresh] SDK is not initialized, ignoring call");
            return;
        }

        if (!getPushConsent(null)) {
            CoreInternal.sharedInstance().L.i("[PushHandler, onTokenRefresh] Consent not given, ignoring call");
            return;
        }
        CoreInternal.sharedInstance().L.i("[PushHandler, onTokenRefresh] Refreshing FCM push token, for [" + provider + "]");
        CoreInternal.sharedInstance().onRegistrationId(token, provider);
    }

    static final String FIREBASE_MESSAGING_CLASS = "com.google.firebase.messaging.FirebaseMessaging";
    static final String FIREBASE_INSTANCEID_CLASS = "com.google.firebase.iid.FirebaseInstanceId";

    static final String HUAWEI_MESSAGING_CLASS = "com.huawei.hms.push.HmsMessageService";
    static final String HUAWEI_CONFIG_CLASS = "com.huawei.agconnect.config.AGConnectServicesConfig";
    static final String HUAWEI_INSTANCEID_CLASS = "com.huawei.hms.aaid.HmsInstanceId";

    /**
     * Initialize messaging functionality
     *
     * @param application application instance
     * @throws IllegalStateException
     */
    @SuppressWarnings("unchecked")
    protected static void init(final Application application) throws IllegalStateException {
        init(application, null);
    }

    /**
     * Initialize messaging functionality
     *
     * @param application       application instance
     * @param preferredProvider prefer specified push provider, {@code null} means use FCM first, then fallback to Huawei
     * @throws IllegalStateException
     */
    @SuppressWarnings("unchecked")
    protected static void init(final Application application, CoreProxy.MessagingProvider preferredProvider) throws IllegalStateException {
        CoreInternal.sharedInstance().L.i("[PushHandler, init] Initialising [" + BuildConfig.FLAVOR + "] Push, App:[" + (application != null) + "], provider:[" + preferredProvider + "]");

        if (application == null) {
            throw new IllegalStateException("Non null application must be provided!");
        }

        // set preferred push provider
        PushHandler.provider = preferredProvider;
        if (provider == null) {
            if (UtilsMessaging.reflectiveClassExists(FIREBASE_MESSAGING_CLASS)) {
                provider = CoreProxy.MessagingProvider.FCM;
            } else if (UtilsMessaging.reflectiveClassExists(HUAWEI_MESSAGING_CLASS)) {
                provider = CoreProxy.MessagingProvider.HMS;
            }
        } else if (provider == CoreProxy.MessagingProvider.FCM && !UtilsMessaging.reflectiveClassExists(FIREBASE_MESSAGING_CLASS)) {
            provider = CoreProxy.MessagingProvider.HMS;
        } else if (provider == CoreProxy.MessagingProvider.HMS && !UtilsMessaging.reflectiveClassExists(HUAWEI_MESSAGING_CLASS)) {
            provider = CoreProxy.MessagingProvider.FCM;
        }

        // print error in case preferred push provider is not available
        if (provider == CoreProxy.MessagingProvider.FCM && !UtilsMessaging.reflectiveClassExists(FIREBASE_MESSAGING_CLASS)) {
            CoreInternal.sharedInstance().L.e("No FirebaseMessaging class in the class path. Please either add it to your gradle config or don't use PushHandler.");
            return;
        } else if (provider == CoreProxy.MessagingProvider.HMS && !UtilsMessaging.reflectiveClassExists(HUAWEI_MESSAGING_CLASS)) {
            CoreInternal.sharedInstance().L.e("No HmsMessageService class in the class path. Please either add it to your gradle config or don't use PushHandler.");
            return;
        } else if (provider == null) {
            CoreInternal.sharedInstance().L.e("Neither FirebaseMessaging, nor HmsMessageService class in the class path. Please either add Firebase / Huawei dependencies or don't use PushHandler.");
            return;
        }

        SharedPref.cacheLastMessagingMode(1, application);
        SharedPref.storeMessagingProvider(provider == CoreProxy.MessagingProvider.FCM ? 1 : 2, application);

        if (callbacks == null) {
            callbacks = new Application.ActivityLifecycleCallbacks() {
                @Override
                public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle bundle) {
                }

                @Override
                public void onActivityStarted(@NonNull Activity activity) {
                    PushHandler.activity = activity;
                }

                @Override
                public void onActivityResumed(@NonNull Activity activity) {
                }

                @Override
                public void onActivityPaused(@NonNull Activity activity) {
                }

                @Override
                public void onActivityStopped(@NonNull Activity activity) {
                    if (activity.equals(PushHandler.activity)) {
                        PushHandler.activity = null;
                    }
                }

                @Override
                public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle bundle) {
                }

                @Override
                public void onActivityDestroyed(@NonNull Activity activity) {
                }
            };

            application.registerActivityLifecycleCallbacks(callbacks);
        }

        createWorker(application);

        new Thread(() -> {
            try {
                long dateTime = SharedPref.getNextPushTokenDate(application);
                Calendar now = Calendar.getInstance();
                now.setTime(new Date());

                Date d = new Date(dateTime);
                if (d.before(now.getTime())) {
                    String token = getToken(application, provider);
                    if (token != null && !"".equals(token)) {
                        onTokenRefresh(token, provider);
                        now.add(Calendar.DATE, 14);
                        SharedPref.setNextPushTokenDate(now.getTimeInMillis(), application);
                    }
                }
            } catch (Exception ignored) {
            }
        }).start();
    }

    private static void createWorker(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            try {
                String latestPushChannel = SharedPref.getLatestPushChannel(context);
                if (latestPushChannel == null) {
                    CharSequence name = "Marketing Amplifier";
                    String description = "this is a temporary channel for amplification push performance";
                    int importance = NotificationManager.IMPORTANCE_HIGH;
                    NotificationChannel channel = new NotificationChannel(PUSH_AMP_CHANNEL_ID, name, importance);
                    channel.setDescription(description);
                    NotificationManager notificationManager = context.getSystemService(NotificationManager.class);
                    notificationManager.createNotificationChannel(channel);
                } else {
                    NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
                    notificationManager.deleteNotificationChannel(PUSH_AMP_CHANNEL_ID);
                }

                Long frequency = Long.valueOf((Integer) CoreInternal.sharedInstance().moduleDynamicConfig.getValue("pushAmpWorkerFrequency"));
                Boolean ampIsEnable = (Boolean) CoreInternal.sharedInstance().moduleDynamicConfig.getValue("pushAmpIsEnable");
                WorkManager workManager = WorkManager.getInstance(context);
                if (!isNull(ampIsEnable) && ampIsEnable) {
                    Constraints constraints = new Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED)
                        .setRequiresBatteryNotLow(true)
                        .build();
                    workManager.enqueueUniquePeriodicWork(
                        "coreIntKPushAmpWorker",
                        ExistingPeriodicWorkPolicy.REPLACE,
                        new PeriodicWorkRequest.Builder(
                            PushAmpWorker.class,
                            Duration.ofMinutes(isNull(frequency) ? 180 : frequency)
                        ).setConstraints(constraints).build()
                    );
                } else {
                    workManager.cancelUniqueWork("coreIntKPushAmpWorker");
                }
            } catch (Exception ex) {
                CoreInternal.sharedInstance().L.e("couldn't run push amp work", ex);
            }
        }

    }

    static boolean getPushConsent(Context context) {
        if (CoreInternal.sharedInstance().isInitialized() || context == null) {
            //todo currently this is also used when context is null and might result in unintended consequences
            //if SDK is initialised, used the stored value
            return CoreInternal.sharedInstance().getConsent(CoreProxy.SdkFeatureNames.push);
        } else {
            //if the SDK is not initialised, use the cached value
            return SharedPref.getConsentPushNoInit(context);
        }

    }

    /**
     * Returns which messaging mode was used in the previous init
     * -1 - no data / no init has happened
     * 0 - test mode
     * 1 - production mode
     */
    protected static int getLastMessagingMethod(Context context) {
        return SharedPref.getLastMessagingMode(context);
    }

}
