/*
Copyright (c) 2012, 2013, 2014 ST.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package sdk.main.core;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ir.intrack.android.sdk.BuildConfig;
import sdk.main.core.inappmessaging.model.message.IAMMessage;

/**
 * This class provides a persistence layer for the local event &amp; connection queues.
 * <p>
 * The "read" methods in this class are not synchronized, because the underlying data store
 * provides thread-safe reads.  The "write" methods in this class are synchronized, because
 * 1) they often read a list of items, modify the list, and then commit it back to the underlying
 * data store, and 2) while the CoreProxy singleton is synchronized to ensure only a single writer
 * at a time from the public API side, the internal implementation has a background thread that
 * submits data to a server, and it writes to this store as well.
 * <p>
 * NOTE: This class is only public to facilitate unit testing, because
 * of this bug in dexmaker: https://code.google.com/p/dexmaker/issues/detail?id=34
 */
class SharedPref {
    private static final String PREFERENCES = "INTRACK_STORE";
    private static final String PREFERENCES_PUSH = "ir.[" + BuildConfig.FLAVOR + "].android.api.messaging";
    private static final String DELIMITER = ":::";
    private static final String CONNECTIONS_PREFERENCE = "CONNECTIONS";
    private static final String EVENTS_PREFERENCE = "EVENTS";
    private static final String STAR_RATING_PREFERENCE = "STAR_RATING";
    private static final String CACHED_ADVERTISING_ID = "ADVERTISING_ID";
    private static final String DYNAMIC_CONFIG_VALUES = "DYNAMIC_CONFIG";
    private static final String CACHED_PUSH_MESSAGING_MODE = "PUSH_MESSAGING_MODE";
    private static final String CACHED_PUSH_MESSAGING_PROVIDER = "PUSH_MESSAGING_PROVIDER";
    private static final String CACHED_DELIVERY_PUSH_EVENTS = "CACHED_DELIVERY_PUSH_EVENTS";
    private static final String CACHED_CLICKED_PUSH_EVENTS = "CACHED_CLICKED_PUSH_EVENTS";
    private static final String CACHED_DISMISS_PUSH_EVENTS = "CACHED_DISMISS_PUSH_EVENTS";
    private static final String CACHED_REJECTED_PUSH_EVENTS = "CACHED_REJECTED_PUSH_EVENTS";
    private static final String INSTALL_REFERRER = "referrer";

    private static final String CACHED_DELIVERY_CUSTOM_CHANNEL = "CACHED_DELIVERY_CUSTOM_CHANNEL";
    private static final String CACHED_CLICKED_CUSTOM_CHANNEL = "CACHED_CLICKED_CUSTOM_CHANNEL";
    private static final String IN_APP_MESSAGES = "IN_APP_MESSAGES";
    private static final int MAX_EVENTS = 100;
    static int MAX_REQUESTS = 1000;//value is configurable for tests
    private static final int MAX_IAMESSAGE_PER_USER = 10;

    private final SharedPreferences preferences_;
    private final SharedPreferences preferencesPush_;

    private static final String CONSENT_GCM_PREFERENCES = "ir.[" + BuildConfig.FLAVOR + "].android.api.messaging.consent.gcm";

    private static final String LATEST_PUSH_CHANNEL_ID = "LATEST_PUSH_CHANNEL_ID";
    private static final String PUSH_TOKEN_DATE = "PUSH_TOKEN_DATE";
    private static final String CACHED_PUSH_MESSAGES = "CACHED_PUSH_MESSAGES";
    private static final String APP_INSTALL_EVENT_RECORDED = "APP_INSTALL_EVENT_RECORDED";


    private static final Gson gson = new Gson();

    private final String appFileDirPath;

    ModuleLog L;

    /**
     * Constructs a SharedPref object.
     *
     * @param context used to retrieve storage meta data, must not be null.
     */
    SharedPref(final Context context, ModuleLog logModule) {
        if (context == null) {
            throw new IllegalArgumentException("must provide valid context");
        }
        preferences_ = context.getSharedPreferences(PREFERENCES, Context.MODE_PRIVATE);
        preferencesPush_ = createPreferencesPush(context);
        appFileDirPath = provideAppDirPathFrom(context);
        L = logModule;
    }

    static SharedPreferences createPreferencesPush(Context context) {
        return context.getSharedPreferences(PREFERENCES_PUSH, Context.MODE_PRIVATE);
    }

    /**
     * Returns an unsorted array of the current stored connections.
     */
    public String[] connections() {
        final String joinedConnStr = preferences_.getString(CONNECTIONS_PREFERENCE, "");
        return joinedConnStr.length() == 0 ? new String[0] : joinedConnStr.split(DELIMITER);
    }

    /**
     * Returns an unsorted array of the current stored event JSON strings.
     */
    public String[] events() {
        final String joinedEventsStr = preferences_.getString(EVENTS_PREFERENCE, "");
        return joinedEventsStr.length() == 0 ? new String[0] : joinedEventsStr.split(DELIMITER);
    }

    /**
     * Returns a list of the current stored events, sorted by timestamp from oldest to newest.
     */
    public List<Event> eventsList() {
        final String[] array = events();
        final List<Event> events = new ArrayList<>(array.length);
        for (String s : array) {
            try {
                final Event event = Event.fromJSON(new JSONObject(s));
                if (event != null) {
                    events.add(event);
                }
            } catch (JSONException ignored) {
                // should not happen since JSONObject is being constructed from previously stringified JSONObject
                // events -> json objects -> json strings -> storage -> json strings -> here
            }
        }
        // order the events from least to most recent
        Collections.sort(events, new Comparator<Event>() {
            @Override
            public int compare(final Event e1, final Event e2) {
                return (int) (e1.timestamp - e2.timestamp);
            }
        });
        return events;
    }

    /**
     * Returns true if no connections are current stored, false otherwise.
     */
    public boolean isEmptyConnections() {
        return preferences_.getString(CONNECTIONS_PREFERENCE, "").length() == 0;
    }

    /**
     * Adds a connection to the local store.
     *
     * @param str the connection to be added, ignored if null or empty
     */
    public synchronized void addConnection(final String str) {
        if (str != null && str.length() > 0) {
            final List<String> currentLocalConnections = new ArrayList<>(Arrays.asList(connections()));
            if (isConnectionEvictionNeeded(currentLocalConnections.size())) {
                currentLocalConnections.remove(0);
            }
            currentLocalConnections.add(str);
            preferences_.edit().putString(CONNECTIONS_PREFERENCE, join(currentLocalConnections, DELIMITER)).apply();
        }
    }

    /**
     * Removes a connection from the local store.
     *
     * @param str the connection to be removed, ignored if null or empty,
     *            or if a matching connection cannot be found
     */
    public synchronized void removeConnection(final String str) {
        if (str != null && str.length() > 0) {
            final List<String> connections = new ArrayList<>(Arrays.asList(connections()));
            if (connections.remove(str)) {
                preferences_.edit().putString(CONNECTIONS_PREFERENCE, join(connections, DELIMITER)).apply();
            }
        }
    }

    protected synchronized void replaceConnections(final String[] newConns) {
        if (newConns != null) {
            final List<String> connections = new ArrayList<>(Arrays.asList(newConns));
            replaceConnectionsList(connections);
        }
    }

    protected synchronized void replaceConnectionsList(final List<String> newConns) {
        if (newConns != null) {
            preferences_.edit().putString(CONNECTIONS_PREFERENCE, join(newConns, DELIMITER)).apply();
        }
    }

    /**
     * Adds a custom event to the local store.
     *
     * @param event event to be added to the local store, must not be null
     */
    void addEvent(final Event event) {
        List<Event> currentLocalEvents = eventsList();
        if (isEventEvictionNeeded(currentLocalEvents.size())) {
            currentLocalEvents.remove(0);
        }
        currentLocalEvents.add(event);
        preferences_.edit().putString(EVENTS_PREFERENCE, joinEvents(currentLocalEvents, DELIMITER)).apply();
    }

    /**
     * Set the preferences that are used for the star rating
     */
    void setStarRatingPreferences(String prefs) {
        preferences_.edit().putString(STAR_RATING_PREFERENCE, prefs).apply();
    }

    /**
     * Get the preferences that are used for the star rating
     */
    String getStarRatingPreferences() {
        return preferences_.getString(STAR_RATING_PREFERENCE, "");
    }

    void setDynamicConfigValues(String values) {
        preferences_.edit().putString(DYNAMIC_CONFIG_VALUES, values).apply();
    }

    String getDynamicConfigValues() {
        return preferences_.getString(DYNAMIC_CONFIG_VALUES, "");
    }

    void setCachedAdvertisingId(String advertisingId) {
        preferences_.edit().putString(CACHED_ADVERTISING_ID, advertisingId).apply();
    }

    String getCachedAdvertisingId() {
        return preferences_.getString(CACHED_ADVERTISING_ID, "");
    }

    void setConsentPush(boolean consentValue) {
        preferencesPush_.edit().putBoolean(CONSENT_GCM_PREFERENCES, consentValue).apply();
    }

    Boolean getConsentPush() {
        return preferencesPush_.getBoolean(CONSENT_GCM_PREFERENCES, false);
    }

    Boolean isAppInstallEventRecorded() {
        return preferencesPush_.getBoolean(APP_INSTALL_EVENT_RECORDED, false);

    }

    void appInstallEventRecorded(boolean isRecorded) {
        preferencesPush_.edit().putBoolean(APP_INSTALL_EVENT_RECORDED, isRecorded).apply();
    }


    public static Boolean getConsentPushNoInit(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        return sp.getBoolean(CONSENT_GCM_PREFERENCES, false);
    }

    /**
     * Adds a custom event to the local store.
     *
     * @param key          name of the custom event, required, must not be the empty string
     * @param segmentation segmentation values for the custom event, may be null
     * @param timestamp    timestamp (seconds since 1970) in GMT when the event occurred
     *                     NaN and infinity values will be quietly ignored.
     */
    public synchronized void addEvent(
            final String key,
            final Map<String, String> segmentation,
            final Map<String, Integer> segmentationInt,
            final Map<String, Double> segmentationDouble,
            final Map<String, Long> segmentationLong,
            final Map<String, Boolean> segmentationBoolean,
            final Map<String, JSONObject> segmentationObject,
            final Map<String, JSONArray> segmentationArray,
            final UserDetails userDetails,
            final long timestamp
    ) {
        final Event event = new Event(null);
        event.key = key;
        event.segmentation = segmentation;
        event.segmentationDouble = segmentationDouble;
        event.segmentationLong = segmentationLong;
        event.segmentationInt = segmentationInt;
        event.segmentationBoolean = segmentationBoolean;
        event.segmentationObject = segmentationObject;
        event.segmentationArray = segmentationArray;
        event.timestamp = timestamp;
        event.userDetails = userDetails;

        addEvent(event);
    }

    /**
     * Removes the specified events from the local store. Does nothing if the event collection
     * is null or empty.
     *
     * @param eventsToRemove collection containing the events to remove from the local store
     */
    public synchronized void removeEvents(final Collection<Event> eventsToRemove) {
        if (eventsToRemove != null && eventsToRemove.size() > 0) {
            final List<Event> events = eventsList();
            if (events.removeAll(eventsToRemove)) {
                preferences_.edit().putString(EVENTS_PREFERENCE, joinEvents(events, DELIMITER)).apply();
            }
        }
    }

    /**
     * Converts a collection of Event objects to URL-encoded JSON to a string, with each
     * event JSON string delimited by the specified delimiter.
     *
     * @param collection events to join into a delimited string
     * @param delimiter  delimiter to use, should not be something that can be found in URL-encoded JSON string
     */
    @SuppressWarnings("SameParameterValue")
    static String joinEvents(final Collection<Event> collection, final String delimiter) {
        final List<String> strings = new ArrayList<>();
        for (Event e : collection) {
            strings.add(e.toJSON().toString());
        }
        return join(strings, delimiter);
    }

    public static synchronized void cacheCustomChannelDelivery(String id, Long date, Context context) {
        addStringToPreferences(CACHED_DELIVERY_CUSTOM_CHANNEL, String.format("%s@%s", id, date), context);
    }

    public static synchronized void cacheCustomChannelClick(String id, String url, Long date, Context context) {
        addStringToPreferences(CACHED_CLICKED_CUSTOM_CHANNEL, String.format("%s@%s@%s", id, url, date), context);
    }

    static Set<String> getCachedCustomChannelDelivery(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_DELIVERY_CUSTOM_CHANNEL, Collections.<String>emptySet());
    }

    static Set<String> getCachedCustomChannelClick(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_CLICKED_CUSTOM_CHANNEL, Collections.<String>emptySet());
    }

    static void clearCachedCustomChannelDelivery(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_DELIVERY_CUSTOM_CHANNEL).apply();
    }

    static void clearCachedCustomChannelClick(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_CLICKED_CUSTOM_CHANNEL).apply();
    }

    public static synchronized void cacheEventDelivery(String id, Long date, Context context) {
        addStringToPreferences(CACHED_DELIVERY_PUSH_EVENTS, String.format("%s@%s", id, date), context);
    }

    public static synchronized void cacheEventRejection(String id, Long date, Context context) {
        addStringToPreferences(CACHED_REJECTED_PUSH_EVENTS, String.format("%s@%s", id, date), context);
    }

    public static synchronized void cacheEventClick(String id, String index, Long date, Context context) {
        addStringToPreferences(CACHED_CLICKED_PUSH_EVENTS, String.format("%s@%s@%s", id, index, date), context);
    }

    public static synchronized void cacheEventDismiss(String id, Long date, Context context) {
        addStringToPreferences(CACHED_DISMISS_PUSH_EVENTS, String.format("%s@%s", id, date), context);
    }

    private static void addStringToPreferences(String key, String value, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        Set<String> stringSet = sp.getStringSet(key, Collections.<String>emptySet());
        Set<String> newSet = new HashSet<>(stringSet);
        newSet.add(value);
        sp.edit().putStringSet(key, newSet).apply();
    }

    static Set<String> getCachedDeliveryEvents(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_DELIVERY_PUSH_EVENTS, Collections.<String>emptySet());
    }

    static Set<String> getCachedClickedEvents(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_CLICKED_PUSH_EVENTS, Collections.<String>emptySet());
    }

    static Set<String> getCachedDismissedEvents(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_DISMISS_PUSH_EVENTS, Collections.<String>emptySet());
    }

    static Set<String> getCachedRejectedEvents(Context context) {
        return createPreferencesPush(context).getStringSet(CACHED_REJECTED_PUSH_EVENTS, Collections.<String>emptySet());
    }

    static void clearCachedDeliveryEvents(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_DELIVERY_PUSH_EVENTS).apply();
    }

    static void clearCachedClickedEvents(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_CLICKED_PUSH_EVENTS).apply();
    }

    static void clearCachedDismissEvents(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_DISMISS_PUSH_EVENTS).apply();
    }

    static void clearCachedRejectedEvents(Context context) {
        createPreferencesPush(context).edit().remove(CACHED_REJECTED_PUSH_EVENTS).apply();
    }

    public static void cacheLastMessagingMode(int mode, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        sp.edit().putInt(CACHED_PUSH_MESSAGING_MODE, mode).apply();
    }

    public static int getLastMessagingMode(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        return sp.getInt(CACHED_PUSH_MESSAGING_MODE, -1);
    }

    public static void storeMessagingProvider(int provider, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        sp.edit().putInt(CACHED_PUSH_MESSAGING_PROVIDER, provider).apply();
    }

    public static int getMessagingProvider(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        return sp.getInt(CACHED_PUSH_MESSAGING_PROVIDER, 0);
    }

    public static synchronized void setLatestPushChannel(String channelId, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        sp.edit().putString(LATEST_PUSH_CHANNEL_ID, channelId).apply();
    }

    public static synchronized String getLatestPushChannel(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        return sp.getString(LATEST_PUSH_CHANNEL_ID, null);
    }

    public static synchronized void setNextPushTokenDate(Long date, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        sp.edit().putLong(PUSH_TOKEN_DATE, date).apply();
    }

    public static synchronized long getNextPushTokenDate(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        return sp.getLong(PUSH_TOKEN_DATE, 0);
    }

    public static synchronized HashMap<String, PushAmpCacheMessage> getCachedPushMessages(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            String json = sp.getString(CACHED_PUSH_MESSAGES, "");
            HashMap<String, PushAmpCacheMessage> messages = gson.fromJson(json, new TypeToken<HashMap<String, PushAmpCacheMessage>>() {
            }.getType());

            HashMap<String, PushAmpCacheMessage> filteredMessages = new HashMap<>();
            if (messages != null) {
                for (Map.Entry<String, PushAmpCacheMessage> entry : messages.entrySet()) {
                    if (!entry.getValue().isExpired()) {
                        filteredMessages.put(entry.getKey(), entry.getValue());
                    }
                }
            }

            return filteredMessages;

        } catch (Exception ignored) {
        }
        return null;
    }

    public static synchronized PushAmpCacheMessage getACachedPushMessage(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            String json = sp.getString(CACHED_PUSH_MESSAGES, "");
            HashMap<String, PushAmpCacheMessage> messages = gson.fromJson(json, new TypeToken<HashMap<String, PushAmpCacheMessage>>() {
            }.getType());

            HashMap<String, PushAmpCacheMessage> filteredMessages = new HashMap<>();
            if (messages != null) {
                for (Map.Entry<String, PushAmpCacheMessage> entry : messages.entrySet()) {
                    if (!entry.getValue().isExpired() && !entry.getValue().getShowed()) {
                        return entry.getValue();
                    }
                }
            }

        } catch (Exception ignored) {
        }
        return null;
    }

    public static synchronized void addCachedPushMessage(PushAmpCacheMessage message, Context context, Boolean updateIfExist) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            String json = sp.getString(CACHED_PUSH_MESSAGES, "");
            HashMap<String, PushAmpCacheMessage> messages = gson.fromJson(json, new TypeToken<HashMap<String, PushAmpCacheMessage>>() {
            }.getType());

            if (!updateIfExist && messages.get(message.getId()) != null) {
                return;
            }

            HashMap<String, PushAmpCacheMessage> filteredMessages = new HashMap<>();
            if (messages != null) {
                for (Map.Entry<String, PushAmpCacheMessage> entry : messages.entrySet()) {
                    if (!entry.getValue().isExpired()) {
                        filteredMessages.put(entry.getKey(), entry.getValue());
                    }
                }
            }

            filteredMessages.put(message.getId(), message);
            sp.edit().putString(CACHED_PUSH_MESSAGES, gson.toJson(filteredMessages)).apply();
        } catch (Exception ignored) {
            sp.edit().remove(CACHED_PUSH_MESSAGES).apply();
            CoreInternal.sharedInstance().L.e(ignored.toString());
        }
    }

    public static synchronized PushAmpCacheMessage getCachedPushMessage(String id, Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            String json = sp.getString(CACHED_PUSH_MESSAGES, "");
            HashMap<String, PushAmpCacheMessage> messages = gson.fromJson(json, new TypeToken<HashMap<String, PushAmpCacheMessage>>() {
            }.getType());

            PushAmpCacheMessage message = messages.get(id);
            if (message != null && !message.isExpired()) {
                return message;
            }
        } catch (Exception ignored) {
            CoreInternal.sharedInstance().L.e(ignored.toString());
        }

        return null;
    }

    public synchronized String getInstallReferrer() {
        try {
            if (preferences_ != null) {
                return preferences_.getString(INSTALL_REFERRER, null);
            }
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("could not read install referrer, error: " + e.getLocalizedMessage());
        }
        return null;
    }

    public static synchronized String getInstallReferrerStatic(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            return sp.getString(INSTALL_REFERRER, null);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("could not read install referrer, error: " + e.getLocalizedMessage());
        }
        return null;
    }

    public synchronized void deleteInstallReferrer() {
        try {
            if (preferences_ != null) {
                preferences_.edit().remove(INSTALL_REFERRER).apply();
            }
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("could not read install referrer, error: " + e.getLocalizedMessage());
        }
    }

    public static synchronized void deleteInstallReferrerStatic(Context context) {
        SharedPreferences sp = createPreferencesPush(context);
        try {
            sp.edit().remove(INSTALL_REFERRER).apply();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("could not read install referrer, error: " + e.getLocalizedMessage());
        }
    }

    void setIAMessages(LinkedHashMap<String, IAMMessage> iamMessage, String userORdeviceID) {
        try {
            if (isIAMMessagesEvictionNeeded(iamMessage.size())) {
                Iterator<String> idIterator = iamMessage.keySet().iterator();
                int removeCount = iamMessage.size() - MAX_IAMESSAGE_PER_USER;
                while (removeCount > 0 && idIterator.hasNext()) {
                    idIterator.next();
                    idIterator.remove();
                    removeCount--;
                }
            }
            preferences_.edit().putString(IN_APP_MESSAGES + userORdeviceID, gson.toJson(iamMessage)).apply();
        } catch (Exception ignored) {
            CoreInternal.sharedInstance().L.e("Error setting IAMessages");
        }
    }


    LinkedHashMap<String, IAMMessage> getIAMessages(String id) {
        try {
            String json = preferences_.getString(IN_APP_MESSAGES + id, "");
            if (json != null) {
                return gson.fromJson(json, new TypeToken<HashMap<String, IAMMessage>>() {
                }.getType());
            }
        } catch (Exception ignored) {
        }
        return null;
    }

    // calculate file size for specific shared preference file in KB
    private static synchronized double getSharedPrefFileSizeKB(@NonNull String preferencesName, String fileDirPath) {
        if (fileDirPath != null) {
            try {

                // getting fileSize
                String filePath = String.format("%s/shared_prefs/%s.xml", fileDirPath, preferencesName);
                File sharedPreferencesFile = new File(filePath);
                if (sharedPreferencesFile.exists()) {
                    return sharedPreferencesFile.length() / 1024.0; // to KiloBytes
                } else {
                    // if there was not a file
                    return -1.0;
                }
            } catch (Exception e) {
                // if reading file threw exception
                return -1.0;
            }
        }
        // if file path was null (not retrieved maybe!)
        return -1.0;
    }

    private static synchronized boolean isEvictionNeeded(String fileDirPath) {
        // in case config is not available
        if (CoreInternal.sharedInstance().config_ == null) {
            return false;
        }

        // only if it is value is greater than `0`
        if (CoreInternal.sharedInstance().config_.sdkLocalStorageMaxFileSize > 0) {

            double primaryPrefsFileSize = getSharedPrefFileSizeKB(PREFERENCES, fileDirPath);
            double pushPrefsFileSize = getSharedPrefFileSizeKB(PREFERENCES_PUSH, fileDirPath);
            double totalSize = primaryPrefsFileSize + (pushPrefsFileSize > 0 ? pushPrefsFileSize : 0);

            return totalSize > CoreInternal.sharedInstance().config_.sdkLocalStorageMaxFileSize;
        }

        // no eviction, no extra process when no sdkLocalStorageMaxFileSize set
        return false;
    }

    private synchronized boolean isEventEvictionNeeded(int currentEventCount) {
        return currentEventCount >= MAX_EVENTS || isEvictionNeeded(appFileDirPath);
    }

    private synchronized boolean isConnectionEvictionNeeded(int currentConnectionCount) {
        return currentConnectionCount >= MAX_REQUESTS || isEvictionNeeded(appFileDirPath);
    }

    // push message shared prefs is instantiated only in the method this needs to be static
    private synchronized boolean isIAMMessagesEvictionNeeded(int currentIAMessageCount) {
        return currentIAMessageCount >= MAX_IAMESSAGE_PER_USER;
    }

    private static String provideAppDirPathFrom(Context context) {
        return context.getFilesDir().getParent();
    }

    /**
     * Joins all the strings in the specified collection into a single string with the specified delimiter.
     */
    static String join(final Collection<String> collection, final String delimiter) {
        final StringBuilder builder = new StringBuilder();

        int i = 0;
        for (String s : collection) {
            builder.append(s);
            if (++i < collection.size()) {
                builder.append(delimiter);
            }
        }

        return builder.toString();
    }

    /**
     * Retrieves a preference from local store.
     *
     * @param key the preference key
     */
    public synchronized String getPreference(final String key) {
        return preferences_.getString(key, null);
    }

    /**
     * Adds a preference to local store.
     * "
     *
     * @param key   the preference key
     * @param value the preference value, supply null value to remove preference
     */
    public synchronized void setPreference(final String key, final String value) {
        if (value == null) {
            preferences_.edit().remove(key).apply();
        } else {
            preferences_.edit().putString(key, value).apply();
        }
    }

    // for unit testing
    synchronized void clear() {
        final SharedPreferences.Editor prefsEditor = preferences_.edit();
        prefsEditor.remove(EVENTS_PREFERENCE);
        prefsEditor.remove(CONNECTIONS_PREFERENCE);
        prefsEditor.clear();
        prefsEditor.apply();

        preferencesPush_.edit().clear().apply();
    }
}
