package sdk.main.core;

import android.os.Build;

import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;

class UtilsTime {

    private static final TimeUniquesEnsurer timeGenerator = new TimeUniquesEnsurer();

    /**
     * Get's a instant for the current moment
     *
     * @return
     */
    public static synchronized Instant getCurrentInstant() {
        long timestamp = currentTimestampMs();
        return Instant.get(timestamp);
    }

    /**
     * Get current timestamp in milliseconds
     *
     * @return
     */
    public static synchronized long currentTimestampMs() {
        return timeGenerator.uniqueTimestamp();
    }

    /**
     * Utility method to return a current timestamp in seconds.
     */
    public static int currentTimestampSeconds() {
        return ((int) (System.currentTimeMillis() / 1000L));
    }

    public static class Instant {
        public final long timestampMs;

        protected Instant(long timestampInMillis) {
            this.timestampMs = timestampInMillis;
        }

        public static Instant get(long timestampInMillis) {
            if (timestampInMillis < 0L) {
                throw new IllegalArgumentException("timestampInMillis must be greater than or equal to zero");
            }
            Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis(timestampInMillis);
            return new Instant(timestampInMillis);
        }

        @Override
        public String toString() {
            return "Instant{" +
                "timestampMs=" + timestampMs +
                "parsed=" + new Date(timestampMs) +
                '}';
        }
    }

    private static class TimeUniquesEnsurer {
        final List<Long> lastTsMs = new ArrayList<>(10);
        final long addition = 0;

        long currentTimeMillis() {
            return System.currentTimeMillis() + addition;
        }

        synchronized long uniqueTimestamp() {
            long ms = currentTimeMillis();

            // change time back case
            if (lastTsMs.size() > 2) {
                long min = Collections.min(lastTsMs);
                if (ms < min) {
                    lastTsMs.clear();
                    lastTsMs.add(ms);
                    return ms;
                }
            }
            // usual case
            while (lastTsMs.contains(ms)) {
                ms += 1;
            }
            while (lastTsMs.size() >= 10) {
                lastTsMs.remove(0);
            }
            lastTsMs.add(ms);
            return ms;
        }
    }

    public static long calculateRemainingTimeInMillis(String endTime) {
        try {
            long remainingMillis;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                OffsetDateTime endDateTime = OffsetDateTime.parse(endTime, DateTimeFormatter.ISO_OFFSET_DATE_TIME);
                OffsetDateTime now = OffsetDateTime.now(ZoneOffset.UTC);
                remainingMillis = ChronoUnit.MILLIS.between(now, endDateTime);
            } else {
                String[] parts = endTime.substring(0, 23).split("[-T:.]");
                String offsetPart = endTime.substring(23);

                Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
                calendar.set(
                    Integer.parseInt(parts[0]),
                    Integer.parseInt(parts[1]) - 1,
                    Integer.parseInt(parts[2]),
                    Integer.parseInt(parts[3]),
                    Integer.parseInt(parts[4]),
                    Integer.parseInt(parts[5])
                );
                calendar.set(Calendar.MILLISECOND, Integer.parseInt(parts[6]));

                // Extract and apply the offset part
                int sign = offsetPart.charAt(0) == '-' ? -1 : 1;
                int offsetHours = Integer.parseInt(offsetPart.substring(1, 3));
                int offsetMinutes = Integer.parseInt(offsetPart.substring(4, 6));
                int offset = sign * (offsetHours * 60 + offsetMinutes) * 60000;
                calendar.add(Calendar.MILLISECOND, offset);

                long serverTimeInMillis = calendar.getTimeInMillis();
                Calendar now = Calendar.getInstance(TimeZone.getTimeZone(TimeZone.getDefault().getID()));
                long currentTimeInMillis = now.getTimeInMillis();

                remainingMillis = serverTimeInMillis - currentTimeInMillis;
            }

            return Math.max(remainingMillis, 0);
        } catch (Exception ignored) {
            return 0;
        }
    }
}
