package sdk.main.core.inappmessaging.display

import android.util.DisplayMetrics
import com.google.gson.Gson

data class Configuration(
    var position: ViewportPosition? = ViewportPosition.CENTER,
    var fallbackMetric: ViewportMetrics? = ViewportMetrics(),
    var responsiveMetrics: MutableList<ViewportMetrics>? = mutableListOf(),
    var overlay: Overlay? = null,
) {
    companion object {
        fun fromString(jsonString: String): Configuration? {
            return try {
                if (jsonString.isBlank())
                    null
                else
                    Gson().fromJson(jsonString, Configuration::class.java)
            } catch (ignored: Exception) {
                null
            }
        }
    }

    fun getFinalViewportMetric(windowWidth: Dimension, metrics: DisplayMetrics): ViewportMetrics? {
        val responsiveMetrics = this.responsiveMetrics ?: return fallbackMetric
        for (responsiveMetric in responsiveMetrics) {
            val value = responsiveMetric.breakpointWidth
            if (value != null && windowWidth.compareTo(value, metrics, true) < 1) {
                return responsiveMetric
            }
        }

        return fallbackMetric
    }

    fun copy(): Configuration {
        return Configuration(
            position = this.position, // Enum is immutable, safe to copy directly
            fallbackMetric = this.fallbackMetric?.copy(),
            responsiveMetrics = this.responsiveMetrics?.toMutableList(),
            overlay = this.overlay?.let {
                Overlay(
                    it.getColor(),
                    it.getOpacity(),
                    it.isDismissible()
                )
            }
        )
    }

    fun isEqual(other: Any?): Boolean {
        return try {
            if (this === other) {
                true
            } else if (other !is Configuration) {
                false
            } else {
                (this.position == other.position) &&
                    (this.overlay?.equals(other.overlay) ?: (other.overlay == null)) &&
                    (this.fallbackMetric?.equals(other.fallbackMetric)
                        ?: (other.fallbackMetric == null)) &&
                    (this.responsiveMetrics?.equals(other.responsiveMetrics)
                        ?: (other.responsiveMetrics == null))
            }
        } catch (ignored: NullPointerException) {
            false
        }
    }

    override fun toString(): String = Gson().toJson(this)

    fun shouldShowOverlay(): Boolean = overlay != null
}
