package sdk.main.core.inappmessaging.display

import android.app.Activity
import android.content.res.Resources
import android.os.Build
import android.util.DisplayMetrics
import android.view.ViewTreeObserver
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import sdk.main.core.inappmessaging.display.mraid.Orientation

data class Dimension(val value: Float?, val unit: UnitType?) {
    fun toDp(metrics: DisplayMetrics, isWidth: Boolean = true): Dp {
        return if (value == null) {
            0.dp
        } else when (unit) {
            UnitType.PX -> value.pxToDp(metrics)
            UnitType.DP -> value.dp
            UnitType.PERCENT -> value.percentToPx(metrics, isWidth = isWidth).pxToDp(metrics)
            else -> 0.dp
        }
    }

    override fun equals(other: Any?): Boolean {
        return if (this === other) {
            true
        } else if (other !is Dimension) {
            false
        } else {
            (this.value == other.value) && (this.unit == other.unit)
        }
    }

    fun compareTo(other: Dimension, metrics: DisplayMetrics, isWidth: Boolean): Int {
        return if (this.value == null && other.value == null) {
            0
        } else if (this.value == null) {
            -1
        } else if (other.value == null) {
            1
        } else {
            this.toDp(metrics, isWidth).compareTo(other.toDp(metrics, isWidth))
        }
    }

    override fun hashCode(): Int {
        return super.hashCode()
    }
}

fun Float.percentToPx(
    metrics: DisplayMetrics,
    isWidth: Boolean,
): Float = if (isWidth) (this / 100) * metrics.widthPixels else (this / 100) * metrics.heightPixels

fun Float.pxToDp(metrics: DisplayMetrics): Dp = (this / metrics.density).dp

fun Resources.getSizePx(key: String): Int = getIdentifier(key, "dimen", "android").let { id ->
    if (id > 0) getDimensionPixelSize(id) else 0
}

fun Resources.getSizeDp(key: String): Dp = getSizePx(key).let { size ->
    Dimension(size.toFloat(), UnitType.PX).toDp(displayMetrics)
}

// Only phone between 0-599 DP has navigation bar can move
fun Resources.isSmartPhone(): Boolean = configuration.smallestScreenWidthDp < 600

fun Resources.getStatusBarSizePx(): Int = getSizePx("status_bar_height")
fun Resources.getStatusBarSizeDp(): Dp = getSizeDp("status_bar_height")

fun Resources.getNavigationBarSizePx(orientation: Orientation): Int {
    return getSizePx(
        if (isSmartPhone() && (orientation == Orientation.Landscape || orientation == Orientation.LandscapeReverse)) {
            "navigation_bar_width"
        } else {
            "navigation_bar_height"
        }
    )
}

fun Resources.getNavigationBarSizeDp(orientation: Orientation): Dp {
    return getSizeDp(
        if (isSmartPhone() && (orientation == Orientation.Landscape || orientation == Orientation.LandscapeReverse)) {
            "navigation_bar_width"
        } else {
            "navigation_bar_height"
        }
    )
}

@Composable
fun Modifier.safeInsetsPadding(activity: Activity, displayMetrics: DisplayMetrics): Modifier {
    return if (Build.VERSION.SDK_INT > Build.VERSION_CODES.R) {
        this.safeDrawingPadding()
    } else if (Build.VERSION.SDK_INT > Build.VERSION_CODES.M) {
        val insetsState = remember { mutableStateOf(activity.window.decorView.rootWindowInsets) }

        DisposableEffect(Unit) {
            val listener = ViewTreeObserver.OnGlobalLayoutListener {
                insetsState.value = activity.window.decorView.rootWindowInsets
            }
            activity.window.decorView.viewTreeObserver.addOnGlobalLayoutListener(listener)

            onDispose {
                activity.window.decorView.viewTreeObserver.removeOnGlobalLayoutListener(listener)
            }
        }
        val insets = insetsState.value

        this.padding(
            top = Dimension(
                (insets?.systemWindowInsetTop ?: 0).toFloat(),
                UnitType.PX
            ).toDp(displayMetrics),
            bottom = Dimension(
                (insets?.systemWindowInsetBottom ?: 0).toFloat(),
                UnitType.PX
            ).toDp(displayMetrics),
            start = Dimension(
                (insets?.systemWindowInsetLeft ?: 0).toFloat(),
                UnitType.PX
            ).toDp(displayMetrics),
            end = Dimension(
                (insets?.systemWindowInsetRight ?: 0).toFloat(),
                UnitType.PX
            ).toDp(displayMetrics),
        )
    } else {
        val orientation = Orientation.fromAndroidActivity(activity)
        val resources = activity.resources
        val statusBarHeight = resources.getStatusBarSizeDp()
        val navigationBarSize = resources.getNavigationBarSizeDp(orientation)

        when (orientation) {
            Orientation.Portrait,
            Orientation.PortraitReverse,
            Orientation.Unspecified -> this.padding(
                top = statusBarHeight,
                bottom = navigationBarSize,
                start = 0.dp,
                end = 0.dp,
            )

            Orientation.Landscape,
            Orientation.LandscapeReverse -> this.padding(
                top = statusBarHeight,
                bottom = if (resources.isSmartPhone()) 0.dp else navigationBarSize,
                start = 0.dp,
                end = if (resources.isSmartPhone()) navigationBarSize else 0.dp,
            )
        }
    }
}
