package sdk.main.core.inappmessaging.display.mraid.utils

import android.annotation.SuppressLint
import android.content.Intent
import android.provider.CalendarContract
import com.google.gson.Gson
import org.json.JSONException
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Locale
import kotlin.math.abs

class W3CEvent private constructor() {
    var id: String = ""
    var description: String = ""
    var location: String? = null
    var summary: String? = null
    var start: String? = null
    var end: String? = null
    var status: String? = null
    var transparency: String? = null
    var reminder: String? = null
    var recurrence: W3CRepeatRule? = null

    @get:SuppressLint("NewApi", "InlinedApi")
    val insertIntent: Intent
        get() {
            val i: Intent
            val nativeMethod = !useMIME
            i = if (nativeMethod) {
                Intent(Intent.ACTION_EDIT).setData(CalendarContract.Events.CONTENT_URI)
            } else {
                Intent(Intent.ACTION_EDIT).setType("vnd.android.cursor.item/event")
            }
            if (description.isNotBlank()) {
                if (nativeMethod) {
                    i.putExtra(CalendarContract.Events.TITLE, description)
                } else {
                    i.putExtra("title", description)
                }
            }
            if (!location.isNullOrBlank()) {
                if (nativeMethod) {
                    i.putExtra(
                        CalendarContract.Events.EVENT_LOCATION,
                        location
                    )
                } else {
                    i.putExtra("eventLocation", location)
                }
            }
            if (!summary.isNullOrBlank()) {
                if (nativeMethod) {
                    i.putExtra(CalendarContract.Events.DESCRIPTION, summary)
                } else {
                    i.putExtra("description", summary)
                }
            }
            if (!start.isNullOrBlank()) {
                val start = millisFromDateString(start!!)
                if (start > 0) {
                    if (nativeMethod) {
                        i.putExtra(CalendarContract.EXTRA_EVENT_BEGIN_TIME, start)
                    } else {
                        i.putExtra("beginTime", start)
                    }
                }
            }
            if (!end.isNullOrBlank()) {
                val end = millisFromDateString(end!!)
                if (end > 0) {
                    if (nativeMethod) {
                        i.putExtra(CalendarContract.EXTRA_EVENT_END_TIME, end)
                    } else {
                        i.putExtra("endTime", end)
                    }
                }
            }
            if (!status.isNullOrBlank()) {
                if (nativeMethod) {
                    i.putExtra(CalendarContract.Events.STATUS, status)
                }
            }
            if (!transparency.isNullOrBlank()) {
                if (nativeMethod) {
                    i.putExtra(
                        CalendarContract.Events.VISIBLE,
                        transparency != "opaque"
                    )
                }
            }
            if (!reminder.isNullOrBlank()) {
                val time = millisFromDateString(reminder!!)
                if (time < 0) {
                    if (nativeMethod) {
                        i.putExtra(
                            CalendarContract.Reminders.MINUTES,
                            abs((time / 60000).toDouble())
                        )
                    }
                } else if (!start.isNullOrBlank()) {
                    if (nativeMethod) {
                        val tstart = millisFromDateString(start!!)
                        if (tstart > 0) {
                            i.putExtra(
                                CalendarContract.Reminders.MINUTES,
                                abs(((tstart - time) / 60000).toDouble())
                            )
                        }
                    }
                }
            }

            val repeatRuleBuilder = StringBuilder("")
            recurrence?.let { recurrenceValue ->
                var freq = recurrenceValue.frequency
                if (!freq.isNullOrBlank()) {
                    if (W3C_DAILY == freq) {
                        repeatRuleBuilder.append("FREQ=DAILY;")
                    } else if (W3C_WEEKLY == freq) {
                        repeatRuleBuilder.append("FREQ=WEEKLY;")
                    } else if (W3C_MONTHLY == freq) {
                        repeatRuleBuilder.append("FREQ=MONTHLY;")
                    } else if (W3C_YEARLY == freq) {
                        repeatRuleBuilder.append("FREQ=YEARLY;")
                    } else {
                        freq = ""
                    }
                } else {
                    freq = ""
                }

                recurrenceValue.interval?.let {
                    if (it > 0) {
                        repeatRuleBuilder.append("INTERVAL=")
                        repeatRuleBuilder.append(it)
                        repeatRuleBuilder.append(";")
                    }
                }

                if (W3C_WEEKLY == freq) {
                    recurrenceValue.daysInWeek?.size?.let { sizeValue ->
                        if (sizeValue > 0) {

                            repeatRuleBuilder.append("BYDAY=")
                            for (j in recurrenceValue.daysInWeek!!) {
                                when (j) {
                                    0 -> repeatRuleBuilder.append("SU,")
                                    1 -> repeatRuleBuilder.append("MO,")
                                    2 -> repeatRuleBuilder.append("TU,")
                                    3 -> repeatRuleBuilder.append("WE,")
                                    4 -> repeatRuleBuilder.append("TH,")
                                    5 -> repeatRuleBuilder.append("FR,")
                                    6 -> repeatRuleBuilder.append("SA,")
                                }
                            }
                        }
                        repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                    }
                }
                if (W3C_MONTHLY == freq) {

                    recurrenceValue.daysInMonth?.size?.let { daysInMonthValue ->
                        repeatRuleBuilder.append("BYMONTHDAY=")
                        for (j in recurrenceValue.daysInMonth!!) {
                            repeatRuleBuilder.append(j)
                            repeatRuleBuilder.append(",")
                        }
                        repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                    }
                }
                if (W3C_YEARLY == freq) {
                    recurrenceValue.daysInYear?.size?.let {
                        repeatRuleBuilder.append("BYYEARDAY=")
                        for (j in recurrenceValue.daysInYear!!) {
                            repeatRuleBuilder.append(j)
                            repeatRuleBuilder.append(",")
                        }
                        repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                    }
                }
                if (W3C_YEARLY == freq) {
                    recurrenceValue.monthsInYear?.size?.let {
                        repeatRuleBuilder.append("BYMONTH=")
                        for (j in recurrenceValue.monthsInYear!!) {
                            repeatRuleBuilder.append(j)
                            repeatRuleBuilder.append(",")
                        }
                        repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                    }
                }
                if (W3C_MONTHLY == freq) {
                    recurrenceValue.weeksInMonth?.size?.let {
                        repeatRuleBuilder.append("BYWEEKNO=")
                        for (j in recurrenceValue.weeksInMonth!!) {
                            repeatRuleBuilder.append(j)
                            repeatRuleBuilder.append(",")
                        }
                        repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                    }
                }
                if (!recurrenceValue.expires.isNullOrBlank()) {
                    repeatRuleBuilder.append("UNTIL=")
                    repeatRuleBuilder.append(recurrenceValue.expires)
                    repeatRuleBuilder.append(";")
                }
                recurrenceValue.exceptionDates?.size?.let {
                    repeatRuleBuilder.append("EXDATE=")
                    for (s in recurrenceValue.exceptionDates!!) {
                        repeatRuleBuilder.append(s)
                        repeatRuleBuilder.append(",")
                    }
                    repeatRuleBuilder.setCharAt(repeatRuleBuilder.length - 1, ';')
                }

                if (nativeMethod) {
                    i.putExtra(
                        CalendarContract.Events.RRULE,
                        repeatRuleBuilder.toString()
                    )
                } else {
                    i.putExtra("rrule", repeatRuleBuilder.toString())
                }
            }

            return i
        }

    private fun millisFromDateString(date: String): Long {
        dateFormats.forEach { df ->
            return try {
                df.parse(date)!!.time
            } catch (parseException: ParseException) {
                try {
                    date.toLong()
                } catch (numberFormatException: NumberFormatException) {
                    -1
                }
            }
        }
        return -1
    }

    companion object {
        //Constants for Event frequency
        const val W3C_DAILY: String = "daily"
        const val W3C_WEEKLY: String = "weekly"
        const val W3C_MONTHLY: String = "monthly"
        const val W3C_YEARLY: String = "yearly"
        var useMIME: Boolean = false
        private val dateFormats = arrayListOf(
            SimpleDateFormat(
                "yyyy-MM-dd'T'HH:mm:ssZZZZZ",
                Locale.US
            ), SimpleDateFormat("yyyy-MM-dd'T'HH:mmZZZZZ", Locale.US)
        )

        @JvmStatic
        fun createFromJSON(s: String): W3CEvent {
            return try {
                Gson().fromJson(s, W3CEvent::class.java)
            } catch (ignored: JSONException) {
                W3CEvent()
            }
        }
    }
}
