package sdk.main.core.pushmessaging

import android.content.Context
import android.net.Uri
import android.os.Parcelable
import java.net.URL

interface PushMessage : Parcelable {

    /**
     * Internal message ID.
     *
     * @return ID string or `null` if no ID is specified in the message.
     */
    val id: String?

    /**
     * Title of the message.
     *
     * @return Title string or `null` if no title is specified in the message.
     */
    val title: String

    /**
     * Title color of the message.
     *
     * @return Title color hex string or `null` if no title color is specified in the message.
     */
    val titleColor: String?

    /**
     * Content of the message.
     *
     * @return Message string or `null` if no content is specified.
     */
    val message: String?

    /**
     * Content of the message color.
     *
     * @return Message color hex string or `null` if no color is specified.
     */
    val messageColor: String?

    /**
     * Message sound. Default is "default". Other sounds are specified as URIs
     * from app resources.
     *
     * @return Sound string or `null` if no sound is specified.
     */
    val sound: String?

    /**
     * Badge count for the message.
     *
     * @return Badge number or `null` if no badge is specified.
     */
    val badge: Int?

    /**
     * Default link to open for the message.
     *
     * @return Link as a [Uri] or `null` if no link is specified.
     */
    val link: Uri?

    /**
     * Media URL associated with the message (e.g., JPEG or PNG image).
     *
     * @return Media URL or `null` if no media is specified.
     */
    val media: URL?

    /**
     * List of buttons associated with the message.
     *
     * @return List of [PushMessageButton], or an empty list if no buttons are specified.
     */
    val buttons: List<PushMessageButton>

    /**
     * Set of all data keys in the message, including standard keys like "title" or "message".
     *
     * @return Set of data keys.
     */
    val dataKeys: Set<String>

    /**
     * Map of custom data key-value pairs sent with the message.
     *
     * @return Custom data map.
     */
    val customData: Map<String, Any>

    /**
     * Type of the message. Possible values: TEXT, CAROUSEL, RATING, BANNER, TIMER.
     */
    val type: MessageType

    /**
     * Timer template associated with the message, if any.
     */
    val timerTemplate: TimerTemplate?

    /**
     * Whether the message is sticky.
     */
    val sticky: Boolean

    /**
     * Duration after which the message will automatically dismiss.
     */
    val autoDismissDuration: Long?

    /**
     * Label for the dismiss button in Android notifications.
     */
    val dismissLabel: String?

    /**
     * Checks whether the message data contains the specified key.
     *
     * @param key Key to check for.
     * @return `true` if the key exists, `false` otherwise.
     */
    fun has(key: String): Boolean

    /**
     * Retrieves the data associated with the specified key.
     *
     * @param key Key to retrieve the value for.
     * @return Value for the key or `null` if it doesn't exist.
     */
    fun data(key: String): String?

    /**
     * Records an action event occurrence for the message.
     *
     * @param context Context to record the action in.
     */
    fun recordAction(context: Context)

    /**
     * Records an action event occurrence for a specific button index.
     *
     * @param context Context to record the action in.
     * @param buttonIndex Index of the button to record the action for.
     */
    fun recordAction(context: Context, buttonIndex: Int)
}
