/*
Copyright (c) 2012, 2013, 2014 ST.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package sdk.main.core;

import android.app.ActivityManager;
import android.content.Context;

import com.android.installreferrer.api.InstallReferrerClient;
import com.android.installreferrer.api.InstallReferrerStateListener;
import com.android.installreferrer.api.ReferrerDetails;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;

/**
 * ConnectionQueue queues session and event data and periodically sends that data to
 * a server on a background thread.
 * <p>
 * None of the methods in this class are synchronized because access to this class is
 * controlled by the CoreProxy singleton, which is synchronized.
 * <p>
 * NOTE: This class is only public to facilitate unit testing, because
 * of this bug in dexmaker: https://code.google.com/p/dexmaker/issues/detail?id=34
 */
class ConnectionQueue {
    public ModuleLog L;
    private SharedPref store_;
    private ExecutorService executor_;
    private String appKey_;
    private String authKey_;
    private Context context_;
    private String serverURL_;
    private Future<?> connectionProcessorFuture_;
    private DeviceId deviceId_;
    private SSLContext sslContext_;
    private InstallReferrerClient referrerClient;
    private Map<String, String> requestHeaderCustomValues;

    // Getters are for unit testing
    String getAppKey() {
        return appKey_;
    }

    void setAppKey(final String appKey) {
        appKey_ = appKey;
    }

    String getAuthKey() {
        return authKey_;
    }

    void setAuthKey(final String authKey) {
        authKey_ = authKey;
    }

    Context getContext() {
        return context_;
    }

    void setContext(final Context context) {
        context_ = context;
    }

    String getServerURL() {
        return serverURL_;
    }

    void setServerURL(final String serverURL) {
        serverURL_ = serverURL;

        if (CoreInternal.publicKeyPinCertificates == null && CoreInternal.certificatePinCertificates == null) {
            sslContext_ = null;
        } else {
            try {
                TrustManager[] tm = {new CertificateTrustManager(CoreInternal.publicKeyPinCertificates, CoreInternal.certificatePinCertificates)};
                sslContext_ = SSLContext.getInstance("TLS");
                sslContext_.init(null, tm, null);
            } catch (Throwable e) {
                throw new IllegalStateException(e);
            }
        }
    }

    SharedPref getSharedPref() {
        return store_;
    }

    void setSharedPref(final SharedPref sharedPref) {
        store_ = sharedPref;
    }

    DeviceId getDeviceId() {
        return deviceId_;
    }

    public void setDeviceId(DeviceId deviceId) {
        this.deviceId_ = deviceId;
        String didStr = null;
        if (deviceId != null) {
            didStr = deviceId.getId();
        }
    }

    protected void setRequestHeaderCustomValues(Map<String, String> headerCustomValues) {
        requestHeaderCustomValues = headerCustomValues;
    }

    /**
     * Checks internal state and throws IllegalStateException if state is invalid to begin use.
     *
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void checkInternalState() {
        if (context_ == null) {
            throw new IllegalStateException("context has not been set");
        }
        if (appKey_ == null || appKey_.length() == 0) {
            throw new IllegalStateException("app key has not been set");
        }
        if (store_ == null) {
            throw new IllegalStateException("shared pref has not been set");
        }
        if (serverURL_ == null || !UtilsNetworking.isValidURL(serverURL_)) {
            throw new IllegalStateException("server URL is not valid");
        }
        if (CoreInternal.publicKeyPinCertificates != null && !serverURL_.startsWith("https")) {
            throw new IllegalStateException("server must start with https once you specified public keys");
        }
    }

    /**
     * Records a session start event for the app and sends it to the server.
     *
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void beginSession() {
        checkInternalState();
        L.d("[Connection Queue] beginSession");

        boolean dataAvailable = false;
        Map<String, Object> metrics = new HashMap<>();

        if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.sessions)) {
            dataAvailable = true;
        }

//        if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.attribution)) {
//            //add attribution data if consent given
//            if (CoreInternal.sharedInstance().isAttributionEnabled) {
//                String cachedAdId = store_.getCachedAdvertisingId();
//
//                if (!cachedAdId.isEmpty()) {
//                    JSONObject adidObject = new JSONObject();
//                    try {
//                        adidObject.put("adid", cachedAdId);
//                    } catch (JSONException e) {
//                        e.printStackTrace();
//                    }
//                    metrics.put("aid", adidObject);
//
//                    dataAvailable = true;
//                }
//            }
//        }

        CoreInternal.sharedInstance().isBeginSessionSent = true;

        if (dataAvailable && foregrounded()) {
            CoreInternal.sharedInstance().events().recordSystemEvent(ModuleSessions.BEGIN_SESSION_EVENT_KEY, metrics);
        }
    }

    private boolean foregrounded() {
        ActivityManager.RunningAppProcessInfo appProcessInfo = new ActivityManager.RunningAppProcessInfo();
        ActivityManager.getMyMemoryState(appProcessInfo);
        return (appProcessInfo.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND || appProcessInfo.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_VISIBLE);
    }

    /**
     * Records a session duration event for the app and sends it to the server. This method does nothing
     * if passed a negative or zero duration.
     *
     * @param duration duration in seconds to extend the current app session, should be more than zero
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void updateSession(final int duration) {
        checkInternalState();
        L.d("[Connection Queue] updateSession");

        if (duration > 0) {
            boolean dataAvailable = false;//will only send data if there is something valuable to send

            Map<String, Object> data = new HashMap<>();

            if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.sessions)) {
                data.put("session_duration", duration);
                dataAvailable = true;
            }

//            if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.attribution)) {
//                if (CoreInternal.sharedInstance().isAttributionEnabled) {
//                    String cachedAdId = store_.getCachedAdvertisingId();
//
//                    if (!cachedAdId.isEmpty()) {
//                        JSONObject adidObject = new JSONObject();
//                        try {
//                            adidObject.put("adid", cachedAdId);
//                        } catch (JSONException e) {
//                            e.printStackTrace();
//                        }
//                        data.put("aid", adidObject);
//                        dataAvailable = true;
//                    }
//                }
//            }


            if (dataAvailable && foregrounded()) {
                CoreInternal.sharedInstance().events().recordSystemEvent(ModuleSessions.UPDATE_SESSION_EVENT_KEY, data);
            }
        }
    }

    public void changeDeviceId(String deviceId, final int duration) {
        checkInternalState();
        L.d("[Connection Queue] changeDeviceId");

        if (!CoreInternal.sharedInstance().anyConsentGiven()) {
            L.d("[Connection Queue] request ignored, consent not given");
            //no consent set, aborting
            return;
        }

        String data = prepareCommonRequestData();

        if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.sessions)) {
            data += "&session_duration=" + duration;
        }

        // !!!!! THIS SHOULD ALWAYS BE ADDED AS THE LAST FIELD, OTHERWISE MERGING BREAKS !!!!!
        data += "&deviceId=" + UtilsNetworking.urlEncodeString(deviceId);

        store_.addConnection(data);
        tick();
    }

    public void tokenSession(String token, CoreProxy.MessagingProvider provider) {
        checkInternalState();
        L.d("[Connection Queue] tokenSession");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.push)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        DeviceInfo.setPushKey(token);
        DeviceInfo.setPushProvider(provider);

        final Map<String, Object> registerEventData = new HashMap<>();
        registerEventData.put("pushKey", token);
        registerEventData.put("pushProvider", provider.name());

        L.d("[Connection Queue] Waiting for 10 seconds before adding token request to queue");

        // To ensure begin_session will be fully processed by the server before token_session
        final ScheduledExecutorService worker = Executors.newSingleThreadScheduledExecutor();
        worker.schedule(new Runnable() {
            @Override
            public void run() {
                L.d("[Connection Queue] Finished waiting 10 seconds adding token request");
                CoreInternal.sharedInstance().events().recordSystemEvent(ModulePush.FCM_REGISTER_EVENT_KEY, registerEventData);
            }
        }, 10, TimeUnit.SECONDS);
    }

    /**
     * Records a session end event for the app and sends it to the server. Duration is only included in
     * the session end event if it is more than zero.
     *
     * @param duration duration in seconds to extend the current app session
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void endSession(final int duration) {
        endSession(duration, null);
    }

    void endSession(final int duration, String deviceIdOverride) {
        checkInternalState();
        L.d("[Connection Queue] endSession");

        boolean dataAvailable = false;//will only send data if there is something valuable to send
        Map<String, Object> data = new HashMap<>();

        if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.sessions)) {
            if (duration > 0) {
                data.put("session_duration", duration);
            }
            dataAvailable = true;
        }

        if (deviceIdOverride != null && CoreInternal.sharedInstance().anyConsentGiven()) {
            //if no consent is given, device ID override is not sent
            data.put("override_id", deviceIdOverride);
            dataAvailable = true;
        }

        if (dataAvailable) {
            CoreInternal.sharedInstance().events().recordSystemEvent(ModuleSessions.END_SESSION_EVENT_KEY, data);
        }
    }

    /**
     * Send user location
     */
    void sendLocation(boolean locationDisabled, String locationCountryCode, String locationCity, String locationGpsCoordinates, String locationIpAddress) {
        checkInternalState();
        L.d("[Connection Queue] sendLocation");

        String data = prepareCommonRequestData();

        data += prepareLocationData(locationDisabled, locationCountryCode, locationCity, locationGpsCoordinates, locationIpAddress);

        store_.addConnection(data);

        tick();
    }

    /**
     * Send user data to the server.
     *
     * @throws java.lang.IllegalStateException if context, app key, store, or server URL have not been set
     */
    void sendUserData() {
        checkInternalState();
        L.d("[Connection Queue] sendUserData");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.users)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        String userdata = UserData.getDataForRequest();

        if (!userdata.equals("")) {
            String data = prepareCommonRequestData()
                    + userdata;
            store_.addConnection(data);

            tick();
        }
    }

    /**
     * Attribute installation to CoreProxy server.
     *
     * @param referrer query parameters
     * @throws java.lang.IllegalStateException if context, app key, store, or server URL have not been set
     */
    void sendReferrerData(String referrer) {
        checkInternalState();
        L.d("[Connection Queue] checkInternalState");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.attribution)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        if (referrer != null) {
            String data = prepareCommonRequestData()
                    + referrer;
            store_.addConnection(data);

            tick();
        }
    }

    void sendInstallData() {
        if (!getSharedPref().isAppInstallEventRecorded()) {
            CoreInternal.sharedInstance().events().recordSystemEvent(ModuleSessions.APP_INSTALL_EVENT_KEY);
            getSharedPref().appInstallEventRecorded(true);
        }

        referrerClient = InstallReferrerClient.newBuilder(context_).build();
        referrerClient.startConnection(new InstallReferrerStateListener() {
            @Override
            public void onInstallReferrerSetupFinished(int responseCode) {
                switch (responseCode) {
                    case InstallReferrerClient.InstallReferrerResponse.OK:
                        try {
                            ReferrerDetails response = referrerClient.getInstallReferrer();

                            String referrerUrl = response.getInstallReferrer();
                            for (String keyValuePair : referrerUrl.split("&")) {
                                String[] keyValueArray = keyValuePair.split("=");
                                String key = UtilsNetworking.urlDecodeString(keyValueArray[0]);
                                String value = UtilsNetworking.urlDecodeString(keyValueArray[1]);
                                if (key.equals("utm_source")) {
                                    DeviceInfo.setCampaignSource(value);
                                }
                                if (key.equals("utm_medium")) {
                                    DeviceInfo.setCampaignMedium(value);
                                }
                                if (key.equals("utm_term")) {
                                    DeviceInfo.setCampaignTerm(value);
                                }
                                if (key.equals("utm_content")) {
                                    DeviceInfo.setCampaignContent(value);
                                }
                                if (key.equals("utm_campaign")) {
                                    DeviceInfo.setCampaignName(value);
                                }
                            }
                            referrerClient.endConnection();
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                        break;
                    case InstallReferrerClient.InstallReferrerResponse.FEATURE_NOT_SUPPORTED:
                        // API not available on the current Play Store app.
                        break;
                    case InstallReferrerClient.InstallReferrerResponse.SERVICE_UNAVAILABLE:
                        // Connection couldn't be established.
                        break;
                }
            }

            @Override
            public void onInstallReferrerServiceDisconnected() {
            }
        });
    }

    /**
     * Reports a crash with device data to the server.
     *
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void sendCrashReport(String error, boolean nonfatal, boolean isNativeCrash, final Map<String, Object> customSegmentation) {
        checkInternalState();
        L.d("[Connection Queue] sendCrashReport");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.crashes)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        //limit the size of the crash report to 20k characters
        if (!isNativeCrash) {
            error = error.substring(0, Math.min(20000, error.length()));
        }

        final String data = prepareCommonRequestData()
                + "&events=" + UtilsNetworking.urlEncodeString(CrashDetails.getCrashData(context_, error, nonfatal, isNativeCrash, customSegmentation));

        store_.addConnection(data);

        tick();
    }

    /**
     * Records the specified events and sends them to the server.
     *
     * @param events URL-encoded JSON string of event data
     * @throws IllegalStateException if context, app key, store, or server URL have not been set
     */
    void recordEvents(final String events) {
        checkInternalState();
        L.d("[Connection Queue] sendConsentChanges");

        ////////////////////////////////////////////////////
        ///CONSENT FOR EVENTS IS CHECKED ON EVENT CREATION//
        ////////////////////////////////////////////////////

        final String data = prepareCommonRequestData(
                events.contains(ModulePush.FCM_REGISTER_EVENT_KEY)
        ) + "&events=" + events;

        store_.addConnection(data);
        tick();
    }

    void sendConsentChanges(String formattedConsentChanges) {
        checkInternalState();
        L.d("[Connection Queue] sendConsentChanges");

        final String data = prepareCommonRequestData()
                + "&consent=" + UtilsNetworking.urlEncodeString(formattedConsentChanges);

        store_.addConnection(data);

        tick();
    }

    void sendAPMCustomTrace(String key, Long durationMs, Long startMs, Long endMs, String customMetrics) {
        checkInternalState();

        L.d("[Connection Queue] sendAPMCustomTrace");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.apm)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        String apmData = "{\"type\":\"device\",\"name\":\"" + key + "\", \"apm_metrics\":{\"duration\": " + durationMs + customMetrics + "}, \"stz\": " + startMs + ", \"etz\": " + endMs + "}";

        final String data = prepareCommonRequestData()
                + "&count=1"
                + "&apm=" + UtilsNetworking.urlEncodeString(apmData);

        store_.addConnection(data);

        tick();
    }

    void sendAPMNetworkTrace(String networkTraceKey, Long responseTimeMs, int responseCode, int requestPayloadSize, int responsePayloadSize, Long startMs, Long endMs) {
        checkInternalState();

        L.d("[Connection Queue] sendAPMNetworkTrace");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.apm)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        String apmMetrics = "{\"response_time\": " + responseTimeMs + ", \"response_payload_size\":" + responsePayloadSize + ", \"response_code\":" + responseCode + ", \"request_payload_size\":" + requestPayloadSize + "}";
        String apmData = "{\"type\":\"network\",\"name\":\"" + networkTraceKey + "\", \"apm_metrics\":" + apmMetrics + ", \"stz\": " + startMs + ", \"etz\": " + endMs + "}";

        final String data = prepareCommonRequestData()
                + "&count=1"
                + "&apm=" + UtilsNetworking.urlEncodeString(apmData);

        store_.addConnection(data);

        tick();
    }

    void sendAPMAppStart(long durationMs, Long startMs, Long endMs) {
        checkInternalState();

        L.d("[Connection Queue] sendAPMAppStart");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.apm)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        String apmData = "{\"type\":\"device\",\"name\":\"app_start\", \"apm_metrics\":{\"duration\": " + durationMs + "}, \"stz\": " + startMs + ", \"etz\": " + endMs + "}";

        final String data = prepareCommonRequestData()
                + "&count=1"
                + "&apm=" + UtilsNetworking.urlEncodeString(apmData);

        store_.addConnection(data);

        tick();
    }

    void sendAPMScreenTime(boolean recordForegroundTime, long durationMs, Long startMs, Long endMs) {
        checkInternalState();

        L.d("[Connection Queue] sendAPMScreenTime, recording foreground time: [" + recordForegroundTime + "]");

        if (!CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.apm)) {
            L.d("[Connection Queue] request ignored, consent not given");
            return;
        }

        final String eventName = recordForegroundTime ? "app_in_foreground" : "app_in_background";

        String apmData = "{\"type\":\"device\",\"name\":\"" + eventName + "\", \"apm_metrics\":{\"duration\": " + durationMs + "}, \"stz\": " + startMs + ", \"etz\": " + endMs + "}";

        final String data = prepareCommonRequestData()
                + "&count=1"
                + "&apm=" + UtilsNetworking.urlEncodeString(apmData);

        store_.addConnection(data);

        tick();
    }

    String prepareCommonRequestData() {
        return prepareCommonRequestData(false);
    }

    String prepareCommonRequestData(Boolean includePushMetrics) {
        UtilsTime.Instant instant = UtilsTime.getCurrentInstant();

        StringBuilder baseData = new StringBuilder(
                "appKey=" + UtilsNetworking.urlEncodeString(appKey_) +
                        "&authKey=" + UtilsNetworking.urlEncodeString(authKey_) +
                        "&date=" + UtilsNetworking.urlEncodeString(Utils.formatDate(instant.timestampMs)) +
                        "&timezone=" + DeviceInfo.getTimezone()
        );

        Map<String, Object> deviceMetrics = new HashMap<>(DeviceInfo.getDeviceMetrics(context_));
        if (includePushMetrics) {
            deviceMetrics.putAll(DeviceInfo.getPushMetrics());
        }
        JSONObject deviceMetricsObject = new JSONObject(deviceMetrics);
        baseData.append("&device=").append(UtilsNetworking.urlEncodeString(deviceMetricsObject.toString()));

        return baseData.toString();
    }

    private String prepareLocationData(boolean locationDisabled, String locationCountryCode, String locationCity, String locationGpsCoordinates, String locationIpAddress) {
        String data = "";

        if (locationDisabled || !CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.location)) {
            //if location is disabled or consent not given, send empty location info
            //this way it is cleared server side and geoip is not used
            //do this only if allowed
            data += "&location=";
        } else {
            //if we get here, location consent was given
            //location should be sent, add all the fields we have

            if (locationGpsCoordinates != null && !locationGpsCoordinates.isEmpty()) {
                data += "&location=" + UtilsNetworking.urlEncodeString(locationGpsCoordinates);
            }

            if (locationCity != null && !locationCity.isEmpty()) {
                data += "&city=" + UtilsNetworking.urlEncodeString(locationCity);
            }

            if (locationCountryCode != null && !locationCountryCode.isEmpty()) {
                data += "&country_code=" + UtilsNetworking.urlEncodeString(locationCountryCode);
            }

            if (locationIpAddress != null && !locationIpAddress.isEmpty()) {
                data += "&ip=" + UtilsNetworking.urlEncodeString(locationIpAddress);
            }
        }
        return data;
    }

    String prepareRatingWidgetRequest(String widgetId) {
        String data = prepareCommonRequestData()
                + "&widget_id=" + UtilsNetworking.urlEncodeString(widgetId)
                + "&deviceId=" + UtilsNetworking.urlEncodeString(deviceId_.getId());
        return data;
    }

    String prepareFeedbackListRequest() {
        String data = prepareCommonRequestData()
                + "&method=feedback"
                + "&deviceId=" + UtilsNetworking.urlEncodeString(deviceId_.getId());

        return data;
    }

    String prepareIAMListRequest(String date) {
        String data = "&userId=" + CoreInternal.sharedInstance().getUserId()
                + "&mobileDevice=ANDROID"
                + "&anonymousId=" + UtilsNetworking.urlEncodeString(deviceId_.getId())
                + "&packageName=" + DeviceInfo.getAppId(context_)
                + "&date=" + UtilsNetworking.urlEncodeString(date);

        return data;
    }

    /**
     * Ensures that an executor has been created for ConnectionProcessor instances to be submitted to.
     */
    void ensureExecutor() {
        if (executor_ == null) {
            executor_ = Executors.newSingleThreadExecutor();
        }
    }

    /**
     * Starts ConnectionProcessor instances running in the background to
     * process the local connection queue data.
     * Does nothing if there is connection queue data or if a ConnectionProcessor
     * is already running.
     */
    void tick() {
        L.d("[Connection Queue] ticked , is foreground? " + foregrounded());
        L.v("[Connection Queue] tick, Not empty:[" + !store_.isEmptyConnections() + "], Has processor:[" + (connectionProcessorFuture_ == null) + "], Done or null:[" + (connectionProcessorFuture_ == null
                || connectionProcessorFuture_.isDone()) + "]");

        if (!store_.isEmptyConnections() && (connectionProcessorFuture_ == null || connectionProcessorFuture_.isDone())) {
            ensureExecutor();
            connectionProcessorFuture_ = executor_.submit(createConnectionProcessor());
        }
    }

    public ConnectionProcessor createConnectionProcessor() {
        return new ConnectionProcessor(getServerURL(), store_, deviceId_, sslContext_, requestHeaderCustomValues, L);
    }

    public boolean queueContainsTemporaryIdItems() {
        String[] storedRequests = getSharedPref().connections();
        String temporaryIdTag = "&deviceId=" + DeviceId.temporaryInTrackDeviceId;

        for (String storedRequest : storedRequests) {
            if (storedRequest.contains(temporaryIdTag)) {
                return true;
            }
        }

        return false;
    }

    // for unit testing
    ExecutorService getExecutor() {
        return executor_;
    }

    void setExecutor(final ExecutorService executor) {
        executor_ = executor;
    }

    Future<?> getConnectionProcessorFuture() {
        return connectionProcessorFuture_;
    }

    void setConnectionProcessorFuture(final Future<?> connectionProcessorFuture) {
        connectionProcessorFuture_ = connectionProcessorFuture;
    }
}
