/*
Copyright (c) 2012, 2013, 2014 ST.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package sdk.main.core;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.net.Uri;
import android.os.Bundle;

import androidx.annotation.NonNull;

import com.lyft.kronos.AndroidClockFactory;
import com.lyft.kronos.KronosClock;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import ir.intrack.android.sdk.BuildConfig;

@SuppressWarnings("JavadocReference")
class CoreInternal {

    /**
     * Tag used in all logging in the SDK.
     */
    public static final String TAG = sharedInstance().setTagName();
    /**
     * Broadcast sent when consent set is changed
     */
    public static final String CONSENT_BROADCAST = "ir.intrack.android.sdk.CoreProxy.CONSENT_BROADCAST";
    /**
     * Default string used in the begin session metrics if the
     * app version cannot be found.
     */
    protected static final String DEFAULT_APP_VERSION = "1.0";
    static final String DEFAULT_INTRACK_SDK_VERSION_STRING = BuildConfig.SDK_VERSION_NAME;
    static final int DEFAULT_INTRACK_SDK_VERSION_CODE_STRING = BuildConfig.SDK_VERSION_CODE;
    /**
     * Used as request meta data on every request
     */
    static final String DEFAULT_INTRACK_SDK_NAME = BuildConfig.SDK_NAME;
    /**
     * How often onTimer() is called.
     */
    private static final long TIMER_DELAY_IN_SECONDS = 60;
    //user data access
    public static UserData userData;
    protected static List<String> publicKeyPinCertificates;
    protected static List<String> certificatePinCertificates;
    static long applicationStart = System.currentTimeMillis();
    /**
     * Determines how many custom events can be queued locally before
     * an attempt is made to submit them to a server.
     */
    private static int EVENT_QUEUE_SIZE_THRESHOLD = 10;
    //a list of valid feature names that are used for checking
    protected final String[] validFeatureNames = new String[]{
            CoreProxy.SdkFeatureNames.sessions,
            CoreProxy.SdkFeatureNames.events,
            CoreProxy.SdkFeatureNames.push,
            CoreProxy.SdkFeatureNames.crashes,
            CoreProxy.SdkFeatureNames.iam,
    };
    final Map<String, Boolean> featureConsentValues = new HashMap<>();
    final List<String> collectedConsentChanges = new ArrayList<>();
    private final ScheduledExecutorService timerService_;
    @SuppressWarnings("ArraysAsListWithZeroOrOneArgument")
    private final List<String> appCrawlerNames = new ArrayList<>(Arrays.asList("Calypso AppCrawler"));//List against which device name is checked to determine if device is app crawler
    private final Map<String, String[]> groupedFeatures = new HashMap<>();
    /**
     * Current version of Android SDK as a displayable string.
     */
    public String INTRACK_SDK_VERSION_STRING = DEFAULT_INTRACK_SDK_VERSION_STRING;
    /**
     * Used as request meta data on every request
     */
    public String INTRACK_SDK_NAME = DEFAULT_INTRACK_SDK_NAME;
    /**
     * Internal logger
     * Should not be used outside of the SDK
     * No guarantees of not breaking functionality
     * Exposed only for the SDK push implementation
     */
    public ModuleLog L = new ModuleLog();
    //attribution
    protected boolean isAttributionEnabled = true;
    protected boolean isBeginSessionSent = false;
    //GDPR
    protected boolean requiresConsent = false;
    ConnectionQueue connectionQueue_;
    EventQueue eventQueue_;
    boolean disableUpdateSessionRequests_ = false;//todo, move to module after 'setDisableUpdateSessionRequests' is removed
    Context context_;
    //Internal modules for functionality grouping
    List<ModuleBase> modules = new ArrayList<>();
    ModuleCrash moduleCrash = null;
    ModuleEvents moduleEvents = null;
    ModuleRatings moduleRatings = null;
    ModuleSessions moduleSessions = null;
    ModuleDynamicConfig moduleDynamicConfig = null;
    ModuleAPM moduleAPM = null;
    ModuleConsent moduleConsent = null;
    ModuleDeviceId moduleDeviceId = null;
    ModuleLocation moduleLocation = null;
    ModuleFeedback moduleFeedback = null;
    ModuleIAM moduleIAM = null;
    //custom request header fields
    Map<String, String> requestHeaderCustomValues;
    Boolean delayedPushConsent = null;//if this is set, consent for push has to be set before finishing init and sending push changes
    boolean delayedLocationErasure = false;//if location needs to be cleared at the end of init
    String[] locationFallback;//temporary used until location can't be set before init
    Config config_ = null;
    private ScheduledFuture<?> timerFuture = null;
    private int activityCount_;
    //w - warnings
    //e - errors
    //i - user accessible calls and important SDK internals
    //d - regular SDK internals
    //v - spammy SDK internals
    private boolean enableLogging_;
    //app crawlers
    private boolean shouldIgnoreCrawlers = true;//ignore app crawlers by default
    private boolean deviceIsAppCrawler = false;//by default assume that device is not a app crawler
    //push related
    private boolean addMetadataToPushIntents = false;// a flag that indicates if metadata should be added to push notification intents
    //internal flags
    private boolean calledAtLeastOnceOnStart = false;//flag for if the onStart function has been called at least once
    private boolean appLaunchDeepLink = true;
    /**
     * Initializes the SDK. Call from your main Activity's onCreate() method.
     * Must be called before other SDK methods can be used.
     * To initialise the SDK, you must pass a Config object that contains
     * all the necessary information for setting up the SDK
     *
     * @param config contains all needed information to init SDK
     */
    private KronosClock kronosClock;

    /**
     * Constructs a CoreProxy object.displayMessage
     * Creates a new ConnectionQueue and initializes the session timer.
     */
    CoreInternal() {
        timerService_ = Executors.newSingleThreadScheduledExecutor();
        staticInit();
    }

    /**
     * Returns the CoreProxy singleton.
     */
    public static CoreInternal sharedInstance() {
        return SingletonHolder.instance;
    }

    public static void onCreate(Activity activity) {
        Intent launchIntent = activity.getPackageManager().getLaunchIntentForPackage(activity.getPackageName());

        if (sharedInstance().L.logEnabled()) {

            String mainClassName = "[VALUE NULL]";
            if (launchIntent != null && launchIntent.getComponent() != null) {
                mainClassName = launchIntent.getComponent().getClassName();
            }

            sharedInstance().L.d("Activity created: " + activity.getClass().getName() + " ( main is " + mainClassName + ")");
        }

        Intent intent = activity.getIntent();
        if (intent != null) {
            Uri data = intent.getData();
            if (data != null) {
                if (sharedInstance().L.logEnabled()) {
                    sharedInstance().L.d("Data in activity created intent: " + data + " (appLaunchDeepLink " + sharedInstance().appLaunchDeepLink + ") ");
                }
                if (sharedInstance().appLaunchDeepLink) {
                    DeviceInfo.deepLink = data.toString();
                }
            }
        }
    }

    /**
     * Allows public key pinning.
     * Supply list of SSL certificates (base64-encoded strings between "-----BEGIN CERTIFICATE-----" and "-----END CERTIFICATE-----" without end-of-line)
     * along with server URL starting with "https://".SdkName will only accept connections to the server
     * if public key of SSL certificate provided by the server matches one provided to this method or by {@link #enableCertificatePinning(List)}.
     *
     * @param certificates List of SSL public keys
     */
    private static void enablePublicKeyPinning(List<String> certificates) {
        sharedInstance().L.i("Enabling public key pinning");
        publicKeyPinCertificates = certificates;
    }

    /**
     * Allows certificate pinning.
     * Supply list of SSL certificates (base64-encoded strings between "-----BEGIN CERTIFICATE-----" and "-----END CERTIFICATE-----" without end-of-line)
     * along with server URL starting with "https://". SdkName will only accept connections to the server
     * if certificate provided by the server matches one provided to this method or by {@link #enablePublicKeyPinning(List)}.
     *
     * @param certificates List of SSL certificates
     */
    private static void enableCertificatePinning(List<String> certificates) {
        CoreInternal.sharedInstance().L.i("Enabling certificate pinning");
        certificatePinCertificates = certificates;
    }

    public static void applicationOnCreate() {
    }

    public Config getConfig() {
        return config_;
    }

    private String setTagName() {
        return "Intrack";
    }

    private void staticInit() {
        connectionQueue_ = new ConnectionQueue();
        CoreInternal.userData = new UserData(connectionQueue_);
        startTimerService(timerService_, timerFuture, TIMER_DELAY_IN_SECONDS);
    }

    private void startTimerService(ScheduledExecutorService service, ScheduledFuture<?> previousTimer, long timerDelay) {
        if (previousTimer != null && !previousTimer.isCancelled()) {
            previousTimer.cancel(false);
        }

        //minimum delay of 1 second
        //maximum delay if 10 minutes
        if (timerDelay < 1) {
            timerDelay = 1;
        } else if (timerDelay > 600) {
            timerDelay = 600;
        }

        timerFuture = service.scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                onTimer();
            }
        }, timerDelay, timerDelay, TimeUnit.SECONDS);
    }

    public long getCurrentTimeMs() {
        if (kronosClock != null) {
            return kronosClock.getCurrentTimeMs();
        }
        return System.currentTimeMillis();
    }

    public String getCurrentTimeFormatted() {
        long now = System.currentTimeMillis();
        if (kronosClock != null) {
            now = kronosClock.getCurrentTimeMs();
        }
        return Utils.formatDate(now);
    }

    public synchronized CoreInternal init(Config config) {
        try {
            kronosClock = AndroidClockFactory.createKronosClock(config.application);
            kronosClock.syncInBackground();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (config == null) {
            throw new IllegalArgumentException("Can't init SDK with 'null' config");
        }

        //enable logging
        if (config.loggingEnabled) {
            //enable logging before any potential logging calls
            setLoggingEnabled(true);
        }

        if (config.overrideSDKVersion != null) {
            INTRACK_SDK_VERSION_STRING = config.overrideSDKVersion;
        }
        if (config.overrideSDKName != null) {
            INTRACK_SDK_NAME = config.overrideSDKName;
        }

        L.SetListener(config.providedLogCallback);

        L.d("[Init] Initializing[" + setTagName() + "] [" + INTRACK_SDK_NAME + "] SDK version [" + INTRACK_SDK_VERSION_STRING + "]");

        if (config.context == null) {
            if (config.application != null) {
                L.d("[Init] No explicit context provided. Using context from the provided application class");
                config.context = config.application;
            } else {
                throw new IllegalArgumentException("valid context is required in [ " + setTagName() + "] init, but was provided 'null'");
            }
        } else {
            L.d("[Init] Using explicitly provided context");
        }

        if (!UtilsNetworking.isValidURL(config.serverURL)) {
            throw new IllegalArgumentException("valid serverURL is required");
        }

        //enable unhandled crash reporting
        if (config.enableUnhandledCrashReporting) {
            enableCrashReporting();
        }

        //react to given consent
        if (config.shouldRequireConsent) {
            setRequiresConsent(true);
            if (config.enabledFeatureNames == null) {
                L.i("[Init] Consent has been required but no consent was given during init");
            } else {
                setConsentInternal(config.enabledFeatureNames, true);
            }
        }

        if (config.serverURL.charAt(config.serverURL.length() - 1) == '/') {
            L.v("[Init] Removing trailing '/' from provided server url");
            config.serverURL = config.serverURL.substring(0, config.serverURL.length() - 1);//removing trailing '/' from server url
        }

        if (config.appKey == null || config.appKey.length() == 0) {
            throw new IllegalArgumentException("valid appKey is required, but was provided either 'null' or empty String");
        }

        if (config.authKey == null || config.authKey.length() == 0) {
            throw new IllegalArgumentException("valid authKey is required, but was provided either 'null' or empty String");
        }

        if (config.application == null) {
            L.w("[Init] Initialising the SDK without providing the application class is deprecated");
        }

        if (config.deviceID != null && config.deviceID.length() == 0) {
            //device ID is provided but it's a empty string
            throw new IllegalArgumentException("valid deviceID is required, but was provided as empty String");
        }
        if (config.idMode == DeviceIdType.TEMPORARY_ID) {
            throw new IllegalArgumentException("Temporary_ID type can't be provided during init");
        }
        if (config.deviceID == null && config.idMode == null) {
            //device ID was not provided and no preferred mode specified. Choosing default
            config.idMode = DeviceIdType.OPEN_UDID;
        }
        if (config.idMode == DeviceIdType.DEVELOPER_SUPPLIED && config.deviceID == null) {
            throw new IllegalArgumentException("Valid device ID has to be provided with the Developer_Supplied device ID type");
        }
        if (config.deviceID == null && config.idMode == DeviceIdType.ADVERTISING_ID && !AdvertisingIdAdapter.isAdvertisingIdAvailable()) {
            //choosing advertising ID as type, but it's available on this device
            L.e("valid deviceID is required because Advertising ID is not available (you need to include Google Play services 4.0+ into your project)");
            return this;
        }
        if (eventQueue_ != null && (!connectionQueue_.getServerURL().equals(config.serverURL) ||
                !connectionQueue_.getAppKey().equals(config.appKey) ||
                !DeviceId.deviceIDEqualsNullSafe(config.deviceID, config.idMode, connectionQueue_.getDeviceId()))) {
            //not sure if this needed
            L.e("[" + setTagName() + "] cannot be reinitialized with different values");
            return this;
        }

        if (L.logEnabled()) {
            L.i("[Init] Checking init parameters");
            L.i("[Init] Is consent required? [" + requiresConsent + "]");

            // Context class hierarchy
            // Context
            //|- ContextWrapper
            //|- - Application
            //|- - ContextThemeWrapper
            //|- - - - Activity
            //|- - Service
            //|- - - IntentService

            Class contextClass = config.context.getClass();
            Class contextSuperClass = contextClass.getSuperclass();

            String contextText = "[Init] Provided Context [" + config.context.getClass().getSimpleName() + "]";
            if (contextSuperClass != null) {
                contextText += ", it's superclass: [" + contextSuperClass.getSimpleName() + "]";
            }

            L.i(contextText);
        }

        //set internal context, it's allowed to be changed on the second init call
        context_ = config.context.getApplicationContext();

        // if we get here and eventQueue_ != null, init is being called again with the same values,
        // so there is nothing to do, because we are already initialized with those values
        if (eventQueue_ == null) {
            L.d("[Init] About to init internal systems");

            config_ = config;

            if (config.sessionUpdateTimerDelay != null) {
                //if we need to change the timer delay, do that first
                startTimerService(timerService_, timerFuture, config.sessionUpdateTimerDelay);
            }

            final SharedPref sharedPref;
            if (config.sharedPref != null) {
                //we are running a test and using a mock object
                sharedPref = config.sharedPref;
            } else {
                sharedPref = new SharedPref(config.context, L);
                config.setSharedPref(sharedPref);
            }

            config_.appInfoPrefs = new AppInfoPrefs(context_);

            //check legacy access methods
            if (locationFallback != null && config.locationCountyCode == null && config.locationCity == null && config.locationLocation == null && config.locationIpAddress == null) {
                //if the fallback was set and config did not contain any location, use the fallback info
                // { country_code, city, gpsCoordinates, ipAddress };
                config.locationCountyCode = locationFallback[0];
                config.locationCity = locationFallback[1];
                config.locationLocation = locationFallback[2];
                config.locationIpAddress = locationFallback[3];
            }

            //initialise modules
            moduleConsent = new ModuleConsent(this, config);
            moduleDeviceId = new ModuleDeviceId(this, config);
            moduleCrash = new ModuleCrash(this, config);
            moduleEvents = new ModuleEvents(this, config);
            moduleRatings = new ModuleRatings(this, config);
            moduleSessions = new ModuleSessions(this, config);
            moduleDynamicConfig = new ModuleDynamicConfig(this, config);
            moduleAPM = new ModuleAPM(this, config);
            moduleLocation = new ModuleLocation(this, config);
            moduleFeedback = new ModuleFeedback(this, config);
            moduleIAM = new ModuleIAM(this, config);
            moduleEvents.registerEventListener(moduleIAM);

            modules.clear();
            modules.add(moduleConsent);
            modules.add(moduleDeviceId);
            modules.add(moduleCrash);
            modules.add(moduleEvents);
            modules.add(moduleRatings);
            modules.add(moduleSessions);
            modules.add(moduleDynamicConfig);
            modules.add(moduleAPM);
            modules.add(moduleLocation);
            modules.add(moduleFeedback);
            modules.add(moduleIAM);

            L.i("[Init] Finished initialising modules");

            //init other things
//            L.d("[Init] Currently cached advertising ID [" + sharedPref.getCachedAdvertisingId() + "]");
            AdvertisingIdAdapter.cacheAdvertisingID(config.context, sharedPref);

            addCustomNetworkRequestHeaders(config.customNetworkRequestHeaders);

            setPushIntentAddMetadata(config.pushIntentAddMetadata);

            if (config.eventQueueSizeThreshold != null) {
                setEventQueueSizeToSend(config.eventQueueSizeThreshold);
            }

            if (config.publicKeyPinningCertificates != null) {
                enablePublicKeyPinning(Arrays.asList(config.publicKeyPinningCertificates));
            }

            if (config.certificatePinningCertificates != null) {
                enableCertificatePinning(Arrays.asList(config.certificatePinningCertificates));
            }

            if (config.enableAttribution != null) {
                setEnableAttribution(config.enableAttribution);
            }

            //app crawler check
            shouldIgnoreCrawlers = config.shouldIgnoreAppCrawlers;
            if (config.appCrawlerNames != null) {
                Collections.addAll(Arrays.asList(config.appCrawlerNames));
            }

            checkIfDeviceIsAppCrawler();

            //initialize networking queues
            connectionQueue_.L = L;
            connectionQueue_.setServerURL(config.serverURL);
            connectionQueue_.setAppKey(config.appKey);
            connectionQueue_.setAuthKey(config.authKey);
            connectionQueue_.setSharedPref(sharedPref);
            connectionQueue_.setDeviceId(config.deviceIdInstance);
            connectionQueue_.setRequestHeaderCustomValues(requestHeaderCustomValues);
            connectionQueue_.setContext(context_);

            eventQueue_ = new EventQueue(sharedPref);
            //AFTER THIS POINT THE SDK IS COUNTED AS INITIALISED

            //set global application listeners
            if (config.application != null) {
                config.application.registerActivityLifecycleCallbacks(new Application.ActivityLifecycleCallbacks() {
                    @Override
                    public void onActivityCreated(@NonNull Activity activity, Bundle bundle) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivityCreated, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityCreated(activity, bundle);
                        }
                    }

                    @Override
                    public void onActivityStarted(@NonNull Activity activity) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivityStarted, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityStarted(activity);
                        }
                    }

                    @Override
                    public void onActivityResumed(@NonNull Activity activity) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ] onActivityResumed, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityResumed(activity);
                        }
                    }

                    @Override
                    public void onActivityPaused(@NonNull Activity activity) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivityPaused, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityPaused(activity);
                        }
                    }

                    @Override
                    public void onActivityStopped(@NonNull Activity activity) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivityStopped, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityStopped(activity);
                        }
                    }

                    @Override
                    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle bundle) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivitySaveInstanceState, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivitySaveInstanceState(activity, bundle);
                        }
                    }

                    @Override
                    public void onActivityDestroyed(@NonNull Activity activity) {
                        if (L.logEnabled()) {
                            L.d("[[ " + setTagName() + " ]] onActivityDestroyed, " + activity.getClass().getSimpleName());
                        }
                        for (ModuleBase module : modules) {
                            module.callbackOnActivityDestroyed(activity);
                        }
                    }
                });
/*
                config.application.registerComponentCallbacks(new ComponentCallbacks() {
                    @Override
                    public void onConfigurationChanged(Configuration configuration) {

                    }

                    @Override
                    public void onLowMemory() {

                    }
                });
 */

                for (ModuleBase module : modules) {
                    module.initFinished(config);
                }

                L.i("[Init] Finished initialising SDK");

                Map<String, Object> appUpdatedData = DeviceInfo.checkAppUpdated(config.context);
                if (appUpdatedData != null) {
                    events().recordSystemEvent(DeviceInfo.APP_UPDATED_EVENT_KEY, appUpdatedData);
                }
            }
        } else {
            //if this is not the first time we are calling init

            // context is allowed to be changed on the second init call
            connectionQueue_.setContext(context_);
        }

        connectionQueue_.sendInstallData();


        return this;
    }

    /**
     * Checks whether Sdk init has been already called.
     *
     * @return true if Sdk is ready to use
     */
    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    public synchronized boolean isInitialized() {
        return eventQueue_ != null;
    }

    /**
     * Immediately disables session &amp; event tracking and clears any stored session &amp; event data.
     * This API is useful if your app has a tracking opt-out switch, and you want to immediately
     * disable tracking when a user opts out.
     */
    public synchronized void halt() {
        L.i("Halting [[ " + setTagName() + " ]]!");
        eventQueue_ = null;
        L.SetListener(null);

        if (connectionQueue_ != null) {
            final SharedPref sharedPref = connectionQueue_.getSharedPref();
            if (sharedPref != null) {
                sharedPref.clear();
            }
            connectionQueue_.setContext(null);
            connectionQueue_.setServerURL(null);
            connectionQueue_.setAppKey(null);
            connectionQueue_.setSharedPref(null);
            connectionQueue_ = null;
        }

        activityCount_ = 0;

        for (ModuleBase module : modules) {
            module.halt();
        }
        modules.clear();

        if (moduleEvents != null && moduleIAM != null)
            moduleEvents.unregisterEventListener(moduleIAM);

        moduleCrash = null;
        moduleEvents = null;
        moduleRatings = null;
        moduleSessions = null;
        moduleDynamicConfig = null;
        moduleConsent = null;
        moduleAPM = null;
        moduleDeviceId = null;
        moduleLocation = null;
        moduleFeedback = null;
        moduleIAM = null;

        INTRACK_SDK_VERSION_STRING = DEFAULT_INTRACK_SDK_VERSION_STRING;
        INTRACK_SDK_NAME = DEFAULT_INTRACK_SDK_NAME;

        staticInit();
    }

    synchronized void notifyDeviceIdChange() {
        L.d("Notifying modules that device ID changed");

        for (ModuleBase module : modules) {
            module.deviceIdChanged();
        }
    }

    /**
     * Tells the SDK that an Activity has started. Since Android does not have an
     * easy way to determine when an application instance starts and stops, you must call this
     * method from every one of your Activity's onStart methods for accurate application
     * session tracking.
     */
    public synchronized void onStart(Activity activity) {
        if (L.logEnabled()) {
            String activityName = "NULL ACTIVITY PROVIDED";
            if (activity != null) {
                activityName = activity.getClass().getSimpleName();
            }
            L.d("[[ " + setTagName() + " ]] onStart called, name:[" + activityName + "], [" + activityCount_ + "] -> [" + (activityCount_ + 1) + "] activities now open");
        }

        appLaunchDeepLink = false;
        if (!isInitialized()) {
            L.e("init must be called before onStart");
            return;
        }

        ++activityCount_;
        if (activityCount_ == 1 && !moduleSessions.manualSessionControlEnabled) {
            //if we open the first activity
            //and we are not using manual session control,
            //begin a session

            moduleSessions.beginSessionInternal();
        }

        //check if there is an install referrer data
        String referrer = ReferrerReceiver.getReferrer(context_);
        L.d("Checking referrer: " + referrer);
        if (referrer != null) {
            connectionQueue_.sendReferrerData(referrer);
            ReferrerReceiver.deleteReferrer(context_);
        }

        CrashDetails.inForeground();

        for (ModuleBase module : modules) {
            module.onActivityStarted(activity);
        }

        calledAtLeastOnceOnStart = true;
    }

    /**
     * Tells the  SDK that an Activity has stopped. Since Android does not have an
     * easy way to determine when an application instance starts and stops, you must call this
     * method from every one of your Activity's onStop methods for accurate application
     * session tracking.
     * unbalanced calls to onStart/onStop are detected
     */
    public synchronized void onStop() {
        L.d("[[ " + setTagName() + " ]] onStop called, [" + activityCount_ + "] -> [" + (activityCount_ - 1) + "] activities now open");

        if (!isInitialized()) {
            L.e("init must be called before onStop");
            return;
        }
        if (activityCount_ == 0) {
            L.e("must call onStart before onStop");
            return;
        }

        --activityCount_;
        if (activityCount_ == 0 && !moduleSessions.manualSessionControlEnabled) {
            // if we don't use manual session control
            // Called when final Activity is stopped.
            // Sends an end session event to the server, also sends any unsent custom events.
            moduleSessions.endSessionInternal(null);
        }

        CrashDetails.inBackground();

        for (ModuleBase module : modules) {
            module.onActivityStopped();
        }
    }

    public synchronized void onConfigurationChanged(Configuration newConfig) {
        L.d("Calling [onConfigurationChanged]");
        if (!isInitialized()) {
            L.e("init must be called before onConfigurationChanged");
            return;
        }

        for (ModuleBase module : modules) {
            module.onConfigurationChanged(newConfig);
        }
    }

    /**
     * DON'T USE THIS!!!!
     */
    public void onRegistrationId(String registrationId) {
        onRegistrationId(registrationId, CoreProxy.MessagingProvider.FCM);
    }

    /**
     * DON'T USE THIS!!!!
     */
    public void onRegistrationId(String registrationId, CoreProxy.MessagingProvider provider) {
        if (!getConsent(CoreProxy.SdkFeatureNames.push)) {
            return;
        }

        connectionQueue_.tokenSession(registrationId, provider);
    }

    /**
     * Changes current device id type to the one specified in parameter. Closes current session and
     * reopens new one with new id. Doesn't merge user profiles on the server
     *
     * @param type     Device ID type to change to
     * @param deviceId Optional device ID for a case when type = DEVELOPER_SPECIFIED
     */
    public void changeDeviceIdWithoutMerge(DeviceIdType type, String deviceId) {
        L.d("Calling [changeDeviceIdWithoutMerge] with type and ID");

        if (!isInitialized()) {
            L.e("init must be called before changeDeviceIdWithoutMerge");
            return;
        }

        moduleDeviceId.changeDeviceIdWithoutMerge(type, deviceId);
    }

    /**
     * Changes current device id to the one specified in parameter. Merges user profile with new id
     * (if any) with old profile.
     *
     * @param deviceId new device id
     */
    public void changeDeviceIdWithMerge(String deviceId) {
        L.d("Calling [changeDeviceIdWithMerge] only with ID");
        if (!isInitialized()) {
            L.e("init must be called before changeDeviceIdWithMerge");
            return;
        }

        moduleDeviceId.changeDeviceIdWithMerge(deviceId);
    }

    /**
     * Sets custom segments to be reported with crash reports
     * In custom segments you can provide any string key values to segments crashes by
     *
     * @param segments Map&lt;String, Object&gt; key segments and their values
     * todo move to module after 'setCustomCrashSegments' is removed
     */
    synchronized void setCustomCrashSegmentsInternal(Map<String, Object> segments) {
        L.d("[ModuleCrash] Calling setCustomCrashSegmentsInternal");

        if (!getConsent(CoreProxy.SdkFeatureNames.crashes)) {
            return;
        }

        if (segments != null) {
            Utils.removeKeysFromMap(segments, ModuleEvents.reservedSegmentationKeys);
            Utils.removeUnsupportedDataTypes(segments);
            CrashDetails.setCustomSegments(segments);
        }
    }

    /**
     * Enable crash reporting to send unhandled crash reports to server
     *
     * @return Returns link  for call chaining
     */
    private synchronized CoreInternal enableCrashReporting() {
        L.d("Enabling unhandled crash reporting");
        //get default handler
        final Thread.UncaughtExceptionHandler oldHandler = Thread.getDefaultUncaughtExceptionHandler();

        Thread.UncaughtExceptionHandler handler = new Thread.UncaughtExceptionHandler() {

            @Override
            public void uncaughtException(Thread t, Throwable e) {
                L.d("Uncaught crash handler triggered");
                if (getConsent(CoreProxy.SdkFeatureNames.crashes)) {

                    StringWriter sw = new StringWriter();
                    PrintWriter pw = new PrintWriter(sw);
                    e.printStackTrace(pw);

                    //add other threads
                    if (moduleCrash.recordAllThreads) {
                        moduleCrash.addAllThreadInformationToCrash(pw);
                    }

                    String exceptionString = sw.toString();

                    //check if it passes the crash filter
                    if (!moduleCrash.crashFilterCheck(exceptionString)) {
                        CoreInternal.sharedInstance().connectionQueue_.sendCrashReport(exceptionString, false, false, null);
                    }
                }

                //if there was another handler before
                if (oldHandler != null) {
                    //notify it also
                    oldHandler.uncaughtException(t, e);
                }
            }
        };

        Thread.setDefaultUncaughtExceptionHandler(handler);
        return this;
    }

    /**
     * Check if logging has been enabled internally in the SDK
     *
     * @return true means "yes"
     */
    public synchronized boolean isLoggingEnabled() {
        return enableLogging_;
    }

    /**
     * Sets whether debug logging is turned on or off. Logging is disabled by default.
     *
     * @param enableLogging true to enable logging, false to disable logging
     */
    private synchronized void setLoggingEnabled(final boolean enableLogging) {
        enableLogging_ = enableLogging;
        L.d("Enabling logging");
    }

    /**
     * Returns if the sdk onStart function has been called at least once
     *
     * @return true - yes, it has, false - no it has not
     */
    public synchronized boolean hasBeenCalledOnStart() {
        return calledAtLeastOnceOnStart;
    }

    /**
     * @param size
     */
    private synchronized void setEventQueueSizeToSend(int size) {
        L.d("Setting event queue size: [" + size + "]");

        if (size < 1) {
            L.d("[setEventQueueSizeToSend] queue size can't be less than zero");
            size = 1;
        }

        EVENT_QUEUE_SIZE_THRESHOLD = size;
    }

    /**
     * Send events if any of them are stored
     */
    protected void sendEventsIfExist() {
        if (eventQueue_.size() > 0) {
            connectionQueue_.recordEvents(eventQueue_.events());
        }
    }

    /**
     * Submits all of the locally queued events to the server if there are more than 10 of them.
     */
    protected void sendEventsIfNeeded() {
        if (eventQueue_.size() >= EVENT_QUEUE_SIZE_THRESHOLD) {
            connectionQueue_.recordEvents(eventQueue_.events());
        }
    }

    /**
     * Immediately sends all stored events
     */
    protected void sendEventsForced() {
        if (eventQueue_.size() > 0) {
            //only send events if there is anything to send
            connectionQueue_.recordEvents(eventQueue_.events());
        }
    }

    /**
     * Called every 60 seconds to send a session heartbeat to the server. Does nothing if there
     * is not an active application session.
     */
    synchronized void onTimer() {
        L.v("[onTimer] Calling heartbeat, Activity count:[" + activityCount_ + "]");

        if (isInitialized()) {
            final boolean hasActiveSession = activityCount_ > 0;
            if (hasActiveSession) {
                if (!moduleSessions.manualSessionControlEnabled) {
                    moduleSessions.updateSessionInternal();
                }

                if (eventQueue_.size() > 0) {
                    connectionQueue_.recordEvents(eventQueue_.events());
                }
            }

            connectionQueue_.tick();
        }
    }

    private void checkIfDeviceIsAppCrawler() {
        String deviceName = DeviceInfo.getDeviceModel();

        for (int a = 0; a < appCrawlerNames.size(); a++) {
            if (deviceName.equals(appCrawlerNames.get(a))) {
                deviceIsAppCrawler = true;
                return;
            }
        }
    }

    /**
     * Return if current device is detected as a app crawler
     *
     * @return returns if devices is detected as a app crawler
     */
    public boolean isDeviceAppCrawler() {
        return deviceIsAppCrawler;
    }

    /**
     * Return if the sdk should ignore app crawlers
     */
    public boolean ifShouldIgnoreCrawlers() {
        if (!isInitialized()) {
            L.e("init must be called before ifShouldIgnoreCrawlers");
            return false;
        }
        return shouldIgnoreCrawlers;
    }

    /**
     * Returns the device id used by for this device
     *
     * @return device ID
     */
    public synchronized String getDeviceID() {
        if (!isInitialized()) {
            L.e("init must be called before getDeviceID");
            return null;
        }

//        L.d("[[ " + setTagName() + " ]] Calling 'getDeviceID'");

        return connectionQueue_.getDeviceId().getId();
    }

    /**
     * Returns the type of the device ID used by for this device.
     *
     * @return device ID type
     */
    public synchronized DeviceIdType getDeviceIDType() {
        if (!isInitialized()) {
            L.e("init must be called before getDeviceID");
            return null;
        }

        L.d("[[ " + setTagName() + " ]] Calling 'getDeviceIDType'");

        return connectionQueue_.getDeviceId().getType();
    }

    /**
     * @param shouldAddMetadata
     */
    private synchronized void setPushIntentAddMetadata(boolean shouldAddMetadata) {
        L.d("[[ " + setTagName() + " ]] Setting if adding metadata to push intents: [" + shouldAddMetadata + "]");
        addMetadataToPushIntents = shouldAddMetadata;
    }

    /**
     * Set if attribution should be enabled
     *
     * @param shouldEnableAttribution set true if you want to enable it, set false if you want to disable it
     */
    private synchronized void setEnableAttribution(boolean shouldEnableAttribution) {
        L.d("[[ " + setTagName() + " ]] Setting if attribution should be enabled");
        isAttributionEnabled = shouldEnableAttribution;
    }

    /**
     * @param shouldRequireConsent
     */
    private synchronized void setRequiresConsent(boolean shouldRequireConsent) {
        L.d("[[ " + setTagName() + " ]] Setting if consent should be required, [" + shouldRequireConsent + "]");
        requiresConsent = shouldRequireConsent;
    }

    /**
     * Special things needed to be done during setting push consent
     *
     * @param consentValue The value of push consent
     */
    void doPushConsentSpecialAction(boolean consentValue) {
        L.d("[[ " + setTagName() + " ]] Doing push consent special action: [" + consentValue + "]");
        connectionQueue_.getSharedPref().setConsentPush(consentValue);
    }

    /**
     * Actions needed to be done for the consent related location erasure
     */
    void doLocationConsentSpecialErasure() {
        moduleLocation.resetLocationValues();
        connectionQueue_.sendLocation(true, null, null, null, null);
    }

    /**
     * Check if the given name is a valid feature name
     *
     * @param name the name of the feature to be tested if it is valid
     * @return returns true if value is contained in feature name array
     */
    private boolean isValidFeatureName(String name) {
        for (String fName : validFeatureNames) {
            if (fName.equals(name)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Prepare features into json format
     *
     * @param features     the names of features that are about to be changed
     * @param consentValue the value for the new consent
     * @return provided consent changes in json format
     */
    private String formatConsentChanges(String[] features, boolean consentValue) {
        StringBuilder preparedConsent = new StringBuilder();
        preparedConsent.append("{");

        for (int a = 0; a < features.length; a++) {
            if (a != 0) {
                preparedConsent.append(",");
            }
            preparedConsent.append('"');
            preparedConsent.append(features[a]);
            preparedConsent.append('"');
            preparedConsent.append(':');
            preparedConsent.append(consentValue);
        }

        preparedConsent.append("}");

        return preparedConsent.toString();
    }

    /**
     * Group multiple features into a feature group
     *
     * @param groupName name of the consent group
     * @param features  array of feature to be added to the consent group
     */
    public synchronized void createFeatureGroup(String groupName, String[] features) {
        L.d("[[ " + setTagName() + " ]] Creating a feature group with the name: [" + groupName + "]");

        if (!isInitialized()) {
            L.w("[[ " + setTagName() + " ]] Calling 'createFeatureGroup' before initialising the SDK is deprecated!");
        }

        groupedFeatures.put(groupName, features);
    }

    CoreInternal setConsentInternal(String[] featureNames, boolean isConsentGiven) {
        final boolean isInit = isInitialized();//is the SDK initialized

        if (!requiresConsent) {
            //if consent is not required, ignore all calls to it
            return this;
        }

        if (featureNames == null) {
            L.w("[[ " + setTagName() + " ]] Calling setConsent with null featureNames!");
            return this;
        }

        boolean previousSessionsConsent = false;
        if (featureConsentValues.containsKey(CoreProxy.SdkFeatureNames.sessions)) {
            previousSessionsConsent = featureConsentValues.get(CoreProxy.SdkFeatureNames.sessions);
        }

        boolean previousLocationConsent = false;
        if (featureConsentValues.containsKey(CoreProxy.SdkFeatureNames.location)) {
            previousLocationConsent = featureConsentValues.get(CoreProxy.SdkFeatureNames.location);
        }

        boolean currentSessionConsent = previousSessionsConsent;

        for (String featureName : featureNames) {
            L.d("[[ " + setTagName() + " ]] Setting consent for feature: [" + featureName + "] with value: [" + isConsentGiven + "]");

            if (!isValidFeatureName(featureName)) {
                L.w("[[ " + setTagName() + " ]] Given feature: [" + featureName + "] is not a valid name, ignoring it");
                continue;
            }

            featureConsentValues.put(featureName, isConsentGiven);

            //special actions for each feature
            switch (featureName) {
                case CoreProxy.SdkFeatureNames.push:
                    if (isInit) {
                        //if the SDK is already initialized, do the special action now
                        doPushConsentSpecialAction(isConsentGiven);
                    } else {
                        //do the special action later
                        delayedPushConsent = isConsentGiven;
                    }
                    break;
                case CoreProxy.SdkFeatureNames.sessions:
                    currentSessionConsent = isConsentGiven;
                    break;
                case CoreProxy.SdkFeatureNames.location:
                    if (previousLocationConsent && !isConsentGiven) {
                        //if consent is about to be removed
                        if (isInit) {
                            doLocationConsentSpecialErasure();
                        } else {
                            delayedLocationErasure = true;
                        }
                    }
                    break;
                case CoreProxy.SdkFeatureNames.apm:
                    if (!isConsentGiven) {
                        //in case APM consent is removed, clear custom and network traces
                        moduleAPM.clearNetworkTraces();
                        moduleAPM.cancelAllTracesInternal();
                    }
            }
        }

        String formattedChanges = formatConsentChanges(featureNames, isConsentGiven);

        if (isInit && (collectedConsentChanges.size() == 0)) {
            //if sdk is initialized and collected changes are already sent, send consent now
            connectionQueue_.sendConsentChanges(formattedChanges);

            context_.sendBroadcast(new Intent(CONSENT_BROADCAST));

            //if consent has changed and it was set to true
            if ((previousSessionsConsent != currentSessionConsent) && currentSessionConsent) {
                //if consent was given, we need to begin the session
                if (isBeginSessionSent) {
                    //if the first timing for a beginSession call was missed, send it again
                    if (!moduleSessions.manualSessionControlEnabled) {
                        moduleSessions.beginSessionInternal();
                    }
                }
            }

            //if consent was changed and set to false
            if ((previousSessionsConsent != currentSessionConsent) && !currentSessionConsent) {
                if (!isBeginSessionSent) {
                    //if session consent was removed and first begins session was not sent
                    //that means that we might not have sent the initially given location information

                    if (moduleLocation.anyValidLocation()) {
                        moduleLocation.sendCurrentLocation();
                    }
                }
            }
        } else {
            // if sdk is not initialized, collect and send it after it is

            collectedConsentChanges.add(formattedChanges);
        }

        return this;
    }

    /**
     * Remove the consent of a feature
     *
     * @param featureNames the names of features for which consent should be removed
     * @return Returns link  for call chaining
     * @deprecated use 'CoreProxy.sharedInstance().consent().removeConsent(featureNames)'
     */
    public synchronized CoreInternal removeConsent(String[] featureNames) {
        L.d("[[ " + setTagName() + " ]] Removing consent for features named: [" + Arrays.toString(featureNames) + "]");

        if (!isInitialized()) {
            L.w("Calling 'removeConsent' before initialising the SDK is deprecated!");
        }

        setConsentInternal(featureNames, false);

        return this;
    }

    /**
     * Remove consent for all features
     *
     * @return Returns link for call chaining
     * @deprecated use 'CoreProxy.sharedInstance().consent().removeConsentAll()'
     */
    public synchronized CoreInternal removeConsentAll() {
        L.d("[[ " + setTagName() + " ]] Removing consent for all features");

        if (!isInitialized()) {
            L.w("Calling 'removeConsentAll' before initialising the SDK is deprecated!");
        }

        removeConsent(validFeatureNames);

        return this;
    }

    /**
     * Get the current consent state of a feature
     *
     * @param featureName the name of a feature for which consent should be checked
     * @return the consent value
     * @deprecated use 'CoreProxy.sharedInstance().consent().getConsent(featureName)'
     */
    public synchronized boolean getConsent(String featureName) {
        if (!requiresConsent) {
            //return true silently
            return true;
        }

        Boolean returnValue = featureConsentValues.get(featureName);

        if (returnValue == null) {
            returnValue = false;
        }

        L.v("[[ " + setTagName() + " ]] Returning consent for feature named: [" + featureName + "] [" + returnValue + "]");

        return returnValue;
    }

    /**
     * Print the consent values of all features
     *
     * @return Returns link  for call chaining
     * @deprecated use 'CoreProxy.sharedInstance().consent().checkAllConsent()'
     */
    public synchronized CoreInternal checkAllConsent() {
        L.d("[[ " + setTagName() + " ]] Checking and printing consent for All features");
        L.d("[[ " + setTagName() + " ]] Is consent required? [" + requiresConsent + "]");

        //make sure push consent has been added to the feature map
        getConsent(CoreProxy.SdkFeatureNames.push);

        StringBuilder sb = new StringBuilder();

        for (String key : featureConsentValues.keySet()) {
            sb.append("Feature named [").append(key).append("], consent value: [").append(featureConsentValues.get(key)).append("]\n");
        }

        L.d(sb.toString());

        return this;
    }

    /**
     * Returns true if any consent has been given
     *
     * @return true - any consent has been given, false - no consent has been given
     * todo move to module
     */
    protected boolean anyConsentGiven() {
        if (!requiresConsent) {
            //no consent required - all consent given
            return true;
        }

        for (String key : featureConsentValues.keySet()) {
            if (featureConsentValues.get(key)) {
                return true;
            }
        }
        return false;
    }


    /**
     * Allows you to add custom header key/value pairs to each request
     */
    private void addCustomNetworkRequestHeaders(Map<String, String> headerValues) {
        L.i("[[ " + setTagName() + " ]] Calling addCustomNetworkRequestHeaders");
        requestHeaderCustomValues = headerValues;
        if (connectionQueue_ != null) {
            connectionQueue_.setRequestHeaderCustomValues(requestHeaderCustomValues);
        }
    }

    /**
     * Deletes all stored requests to server.
     * This includes events, crashes, views, sessions, etc
     * Call only if you don't need that information
     */
    public void flushRequestQueues() {
        L.i("[[ " + setTagName() + " ]] Calling flushRequestQueues");

        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before flushRequestQueues");
            return;
        }

        SharedPref store = connectionQueue_.getSharedPref();

        int count = 0;

        while (true) {
            final String[] storedEvents = store.connections();
            if (storedEvents == null || storedEvents.length == 0) {
                // currently no data to send, we are done for now
                break;
            }
            //remove stored data
            store.removeConnection(storedEvents[0]);
            count++;
        }

        L.d("[[ " + setTagName() + " ]] flushRequestQueues removed [" + count + "] requests");
    }

    /**
     * CoreProxy will attempt to fulfill all stored requests on demand
     */
    public void doStoredRequests() {
        L.i("[[ " + setTagName() + " ]] Calling doStoredRequests");

        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before doStoredRequests");
            return;
        }

        connectionQueue_.tick();
    }

    /**
     * Go through the request queue and replace the appKey of all requests with the current appKey
     */
    synchronized public void requestQueueOverwriteAppKeys() {
        L.i("[[ " + setTagName() + " ]] Calling requestQueueOverwriteAppKeys");

        if (!isInitialized()) {
            L.e("[[ " + setTagName() + " ]] CoreProxy.sharedInstance().init must be called before requestQueueOverwriteAppKeys");
            return;
        }

        List<String> filteredRequests = requestQueueReplaceWithAppKey(connectionQueue_.getSharedPref().connections(), connectionQueue_.getAppKey());
        if (filteredRequests != null) {
            connectionQueue_.getSharedPref().replaceConnectionsList(filteredRequests);
            doStoredRequests();
        }
    }

    /**
     * Go through the request queue and delete all requests that don't have the current application key
     */
    synchronized public void requestQueueEraseAppKeysRequests() {
        L.i("[[ " + setTagName() + " ]] Calling requestQueueEraseAppKeysRequests");

        if (!isInitialized()) {
            L.e("[[ " + setTagName() + " ]] CoreProxy.sharedInstance().init must be called before requestQueueEraseAppKeysRequests");
            return;
        }

        List<String> filteredRequests = requestQueueRemoveWithoutAppKey(connectionQueue_.getSharedPref().connections(), connectionQueue_.getAppKey());
        connectionQueue_.getSharedPref().replaceConnectionsList(filteredRequests);
        doStoredRequests();
    }

    synchronized List<String> requestQueueReplaceWithAppKey(String[] storedRequests, String targetAppKey) {
        try {
            List<String> filteredRequests = new ArrayList<>();

            if (storedRequests == null || targetAppKey == null) {
                //early abort
                return filteredRequests;
            }

            String replacementPart = "appKey=" + UtilsNetworking.urlEncodeString(targetAppKey);

            for (int a = 0; a < storedRequests.length; a++) {
                if (storedRequests[a] == null) {
                    continue;
                }

                boolean found = false;
                String[] parts = storedRequests[a].split("&");

                for (int b = 0; b < parts.length; b++) {
                    if (parts[b].contains("appKey=")) {
                        parts[b] = replacementPart;
                        found = true;
                        break;
                    }
                }

                if (found) {
                    //recombine and add
                    StringBuilder stringBuilder = new StringBuilder(storedRequests[a].length());

                    for (int c = 0; c < parts.length; c++) {
                        if (c != 0) {
                            stringBuilder.append("&");
                        }
                        stringBuilder.append(parts[c]);
                    }
                    filteredRequests.add(stringBuilder.toString());
                } else {
                    //pass through the old one
                    filteredRequests.add(storedRequests[a]);
                }
            }

            return filteredRequests;
        } catch (Exception ex) {
            //in case of failure, abort
            L.e("[[ " + setTagName() + " ]] Failed while overwriting appKeys, " + ex.toString());

            return null;
        }
    }

    synchronized List<String> requestQueueRemoveWithoutAppKey(String[] storedRequests, String targetAppKey) {
        List<String> filteredRequests = new ArrayList<>();

        if (storedRequests == null || targetAppKey == null) {
            //early abort
            return filteredRequests;
        }

        String searchablePart = "appKey=" + targetAppKey;

        for (int a = 0; a < storedRequests.length; a++) {
            if (storedRequests[a] == null) {
                continue;
            }

            if (!storedRequests[a].contains(searchablePart)) {
                L.d("[requestQueueEraseAppKeysRequests] Found a entry to remove: [" + storedRequests[a] + "]");
            } else {
                filteredRequests.add(storedRequests[a]);
            }
        }

        return filteredRequests;
    }

    /**
     * Go into temporary device ID mode
     *
     * @return
     */
    public CoreInternal enableTemporaryIdMode() {
        L.i("[[ " + setTagName() + " ]] Calling enableTemporaryIdMode");

        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before enableTemporaryIdMode");
            return this;
        }

        moduleDeviceId.changeDeviceIdWithoutMerge(DeviceIdType.TEMPORARY_ID, DeviceId.temporaryInTrackDeviceId);

        return this;
    }

    public String getUserId() {
        return config_.sharedPref.getPreference(ModuleSessions.USER_ID_PREFERENCE_KEY);
    }

    public ModuleCrash.Crashes crashes() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing crashes");
            return null;
        }

        return moduleCrash.crashesInterface;
    }

    public ModuleEvents.Events events() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing events");
            return null;
        }

        return moduleEvents.eventsInterface;
    }

    public ModuleRatings.Ratings ratings() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing ratings");
            return null;
        }

        return moduleRatings.ratingsInterface;
    }

    public ModuleSessions.Sessions sessions() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing sessions");
            return null;
        }

        return moduleSessions.sessionInterface;
    }

    public ModuleDynamicConfig.DynamicConfig dynamicConfig() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing remote config");
            return null;
        }

        return moduleDynamicConfig.dynamicConfigInterface;
    }

    public ModuleAPM.Apm apm() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing apm");
            return null;
        }

        return moduleAPM.apmInterface;
    }

    public ModuleConsent.Consent consent() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing consent");
            return null;
        }

        return moduleConsent.consentInterface;
    }

    public ModuleLocation.Location location() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing location");
            return null;
        }

        return moduleLocation.locationInterface;
    }

    public ModuleFeedback.Feedback feedback() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing feedback");
            return null;
        }

        return moduleFeedback.feedbackInterface;
    }

    public ModuleIAM.IAM iam() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing feedback");
            return null;
        }

        return moduleIAM.iamInterface;
    }

    public ModuleIAM iamModule() {
        if (!isInitialized()) {
            L.e("CoreProxy.sharedInstance().init must be called before accessing feedback");
            return null;
        }

        return moduleIAM;
    }

    // for unit testing
    ConnectionQueue getConnectionQueue() {
        return connectionQueue_;
    }

    void setConnectionQueue(final ConnectionQueue connectionQueue) {
        connectionQueue_ = connectionQueue;
    }

    ExecutorService getTimerService() {
        return timerService_;
    }

    EventQueue getEventQueue() {
        return eventQueue_;
    }

    void setEventQueue(final EventQueue eventQueue) {
        eventQueue_ = eventQueue;
    }

    long getPrevSessionDurationStartTime() {
        return moduleSessions.prevSessionDurationStartTime_;
    }

    void setPrevSessionDurationStartTime(final long prevSessionDurationStartTime) {
        moduleSessions.prevSessionDurationStartTime_ = prevSessionDurationStartTime;
    }

    int getActivityCount() {
        return activityCount_;
    }

    synchronized boolean getDisableUpdateSessionRequests() {
        return disableUpdateSessionRequests_;
    }

    // see http://stackoverflow.com/questions/7048198/thread-safe-singletons-in-java
    private static class SingletonHolder {
        @SuppressLint("StaticFieldLeak")
        static final CoreInternal instance = new CoreInternal();
    }
}
