package sdk.main.core;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.util.Base64;

import androidx.fragment.app.Fragment;

import org.json.JSONException;
import org.json.JSONObject;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import ir.intrack.android.sdk.BuildConfig;
import sdk.main.core.pushmessaging.PushMessage;

public class CoreProxy {

    public static void init(Config config) {
        try {
            CoreInternal.sharedInstance().init(config);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in initialization, " + e.getMessage());
        }
    }

    public static long getCurrentTimeMs() {
        return CoreInternal.sharedInstance().getCurrentTimeMs();
    }

    public static boolean isInitialized() {
        try {
            return CoreInternal.sharedInstance().isInitialized();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in getting initialization status " + e.getMessage());
        }
        return false;
    }

    public static void initNotificationService(final Application application, MessagingProvider preferredProvider) {
        try {
            PushHandler.useAdditionalIntentRedirectionChecks = false;
            PushHandler.init(application, preferredProvider);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in notification service initialization, " + e.getMessage());
        }
    }

    public static void recordEvent(String key) {
        try {
            CoreInternal.sharedInstance().events().recordEvent(key, new HashMap<String, Object>());
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording event, " + e.getMessage());
        }
    }

    public static void recordEvent(String key, Map<String, Object> segmentation) {
        try {
            CoreInternal.sharedInstance().events().recordEvent(key, segmentation);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording event, " + e.getMessage());
        }
    }

    public static void recordLogin(UserDetails userDetails) {
        try {
            CoreInternal.sharedInstance().sessions().recordLogin(userDetails);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording login, " + e.getMessage());
        }
    }

    public static void recordLogout() {
        try {
            CoreInternal.sharedInstance().sessions().recordLogout();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording logout, " + e.getMessage());
        }
    }

    public static void recordCustomChannelDelivery(final Context context, final String token) {
        try {
            CustomChannelHandler.recordEvent(context, token, CustomChannelHandler.EventType.DELIVERY);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording CustomChannelDelivery, " + e.getMessage());
        }
    }

    public static void recordCustomChannelClick(final Context context, final String token) {
        try {
            CustomChannelHandler.recordEvent(context, token, CustomChannelHandler.EventType.CLICK);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recording CustomChannelClick, " + e.getMessage());
        }
    }

    public static void recordHandledException(Exception exception) {
        try {
            CoreInternal.sharedInstance().crashes().recordHandledException(exception);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recordHandledException, " + e.getMessage());
        }
    }

    public static void recordHandledException(Throwable exception) {
        try {
            CoreInternal.sharedInstance().crashes().recordHandledException(exception);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recordHandledException, " + e.getMessage());
        }
    }

    public static void recordUnhandledException(Exception exception) {
        try {
            CoreInternal.sharedInstance().crashes().recordUnhandledException(exception);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recordUnhandledException, " + e.getMessage());
        }
    }

    public static void recordUnhandledException(Throwable exception) {
        try {
            CoreInternal.sharedInstance().crashes().recordUnhandledException(exception);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in recordUnhandledException, " + e.getMessage());
        }
    }

    public static void customNavigatedTo(String customScreen) {
        CoreInternal.sharedInstance().iamModule().customNavigatedTo(customScreen);
    }

    public static void customNavigatedTo(String customScreen, Map<String, Object> screenData) {
        CoreInternal.sharedInstance().iamModule().customNavigatedTo(customScreen, screenData);
    }

    public static void updateScreenData(String customScreen, Map<String, Object> screenData) {
        CoreInternal.sharedInstance().iamModule().updateScreenData(customScreen, screenData);
    }

    public static void updateScreenData(Activity activity, Map<String, Object> screenData) {
        CoreInternal.sharedInstance().iamModule().updateScreenData(activity, screenData);
    }

    public static void updateScreenData(Fragment fragment, Map<String, Object> screenData) {
        CoreInternal.sharedInstance().iamModule().updateScreenData(fragment, screenData);
    }

    public static void updateProfile(UserDetails userDetails) {
        try {
            CoreInternal.sharedInstance().sessions().updateProfile(userDetails);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in updating profile, " + e.getMessage());
        }
    }

    public static String getUserId() {
        try {
            return CoreInternal.sharedInstance().getUserId();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in getting user id, " + e.getMessage());
        }
        return "";
    }

    public static String getDeviceId() {
        try {
            return CoreInternal.sharedInstance().getDeviceID();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in getting device id, " + e.getMessage());
        }
        return "";
    }

    public static void onTokenRefresh(String token) {
        try {
            PushHandler.onTokenRefresh(token);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in refreshing token, " + e.getMessage());
        }
    }

    public static void onTokenRefresh(String token, MessagingProvider provider) {
        try {
            PushHandler.onTokenRefresh(token, provider);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in refreshing token, " + e.getMessage());
        }
    }

    public static Boolean displayMessage(final Context context, final Map<String, String> data, final String channelId, final int notificationSmallIcon, final Intent notificationIntent) {
        try {
            String messageId = data.get(ModulePush.KEY_ID);
            String trackingId = data.get(ModulePush.KEY_TRACKING_ID);

            // send log when push notification or silent received
            sendPushDeliveryLog(data);

            if (messageId == null && trackingId != null) {
                SharedPref store;
                if (CoreInternal.sharedInstance().isInitialized()) {
                    store = CoreInternal.sharedInstance().config_.sharedPref;
                } else {
                    ModuleLog L = new ModuleLog();
                    store = new SharedPref(context, L);
                }

                //bulk message
                JSONObject jsonId = new JSONObject();
                try {
                    jsonId.put("id", trackingId);
                    String userId = store.getPreference(ModuleSessions.USER_ID_PREFERENCE_KEY);

                    String deviceId = store.getPreference("ir.[" + BuildConfig.FLAVOR + "].android.api.DeviceId.id");
                    byte[] pidBytes = Base64.decode(store.getPreference("ir.[" + BuildConfig.FLAVOR + "].android.api.appkey"), Base64.DEFAULT);
                    int productId = 0;
                    for (byte b : pidBytes) {
                        productId = (productId << 8) + ((int) b & 0xFF);
                    }
                    jsonId.put("productId", productId);
                    jsonId.put("timestamp", UtilsTime.currentTimestampMs());
                    if (userId != null) {
                        jsonId.put("userId", userId);
                    } else {
                        jsonId.put("anonymousId", deviceId);
                    }
                    byte[] a;
                    if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.KITKAT) {

                        a = jsonId.toString().getBytes(StandardCharsets.UTF_8);//TODO: Check sdk version then set utf 8.
                    } else {
                        a = jsonId.toString().getBytes(Charset.forName("UTF-8"));
                    }

                    messageId = Base64.encodeToString(a, Base64.DEFAULT);
                } catch (JSONException e) {
                    e.printStackTrace();
                }

                data.put(ModulePush.KEY_ID, messageId);
            }

            //TODO: here we could check if this message is an inAppMessage, send it to it's module
            // also check for source that should be sdk;
            if (data.containsKey("pushType") && Objects.equals(data.get("pushType"), "IN_APP")) {
                CoreInternal.sharedInstance().moduleIAM.updateMessages(data.get("messages"));
                return true;
            } else {
                if (messageId != null) {
                    PushAmpCacheMessage msg = SharedPref.getCachedPushMessage(messageId, context);
                    if (msg != null && msg.getShowed()) {
                        return false;
                    } else {
                        SharedPref.addCachedPushMessage(new PushAmpCacheMessage(messageId, true), context, true);
                    }
                }

                PushMessage message = PushHandler.decodeMessage(data);
                return PushHandler.displayMessage(context, message, channelId, notificationSmallIcon, notificationIntent);
            }
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in displaying message, " + e.getMessage());
            return false;
        }
    }

    public static void initPush(Application application) {
        try {
            PushHandler.init(application);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in initializing push, " + e.getMessage());
        }
    }

    public static int getLastMessagingMethod(Context context) {
        try {
            return PushHandler.getLastMessagingMethod(context);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in getting last messaging method, " + e.getMessage());
        }
        return 0;
    }

    public static void onStart(Activity activity) {
        try {
            CoreInternal.sharedInstance().onStart(activity);
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in onStart, " + e.getMessage());
        }
    }

    public static void onStop() {
        try {
            CoreInternal.sharedInstance().onStop();
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.e("Exception occurred in onStop, " + e.getMessage());
        }
    }

    // handles delivery log sending if "inTrack" is source and "debug" is "true"
    private static void sendPushDeliveryLog(final Map<String, String> pushData) {
        try {
            CoreInternal.sharedInstance().L.v("[Delivery Log] checking source for delivery log sending.");
            sendLogOnSilentPushInTrack(pushData);
            sendLogOnDebugFieldInBody(pushData);
        } catch (Exception ignored) {
        }
    }

    private static void sendLogOnSilentPushInTrack(final Map<String, String> pushData) {

        // check if Silent Push From InTrack
        if (Utils.isSilentPushFromInTrack(pushData)) {

            CoreInternal.sharedInstance().L.v("[Delivery Log] Silent Push From InTrack Detected");

            // send log if source is "inTrack"
            CoreInternal.sharedInstance().L.sendLog("Silent Push message received");

            CoreInternal.sharedInstance().L.v("[Delivery Log] delivery Log sent");

        } else {
            CoreInternal.sharedInstance().L.v("[Delivery Log] message is not Silent Push From InTrack");
        }

    }

    private static void sendLogOnDebugFieldInBody(final Map<String, String> pushData) {

        if (Utils.isMessageDebugActive(pushData)) {

            CoreInternal.sharedInstance().L.v("[Delivery Log] debug key found and is true");

            // getting cached push IDs
            Map<String, PushAmpCacheMessage> cachedPushIds = SharedPref.getCachedPushMessages(CoreInternal.sharedInstance().context_);

            String joinedIDs = Utils.adaptCacheMessage4Log(cachedPushIds);

            CoreInternal.sharedInstance().L.v("[Delivery Log] cached push IDs: " + joinedIDs);

            CoreInternal.sharedInstance().L.sendLog("cached push IDs: " + joinedIDs);

        } else {
            CoreInternal.sharedInstance().L.v("[Delivery Log] data does not contain `debug` or `debug: true`");
        }

    }

    public static String getAuthKey() {
        return CoreInternal.sharedInstance().config_.authKey;
    }

    public static String getAppKey() {
        return CoreInternal.sharedInstance().config_.appKey;
    }

    public static String getSdkName() {
        return CoreInternal.DEFAULT_INTRACK_SDK_NAME;
    }

    public static String getPackageName() {
        try {
            return DeviceInfo.getAppId(CoreInternal.sharedInstance().context_);
        } catch (Exception ignored) {
        }
        return "";
    }

    public static long getSDKVersionCode() {
        try {
            return DeviceInfo.getSdkVersionCode(CoreInternal.sharedInstance().context_);
        } catch (Exception ignored) {
        }
        return -1;
    }

    public static String getOS() {
        return DeviceInfo.getOS();
    }

    public static String getOSVersion() {
        return DeviceInfo.getOSVersion();
    }

    public static long getOSVersionCode() {
        return DeviceInfo.getOSVersionCode();
    }

    public static String getSdkVersion() {
        return CoreInternal.DEFAULT_INTRACK_SDK_VERSION_STRING;
    }

    public static String getIFA() {
        return CoreInternal.sharedInstance().getDeviceID();
    }

    public static boolean isTrackingLimitedEnabled() {
        return false; //todo: implement a mechanism for tracking limitation detection
    }

    public static boolean isCoppaEnabled() {
        return false; // todo: implement a mechanism for coppa
    }

    public enum MessagingProvider {
        FCM,    // Firebase
        HMS,    // Huawei
    }

    public static class SdkFeatureNames {
        public static final String sessions = "sessions";
        public static final String events = "events";
        public static final String location = "location";
        public static final String crashes = "crashes";
        public static final String attribution = "attribution";
        public static final String users = "users";
        public static final String push = "push";
        public static final String starRating = "star-rating";
        public static final String apm = "apm";
        public static final String feedback = "feedback";
        public static final String iam = "iam";
    }

}
