package sdk.main.core;

import android.content.Context;

import ir.intrack.android.sdk.BuildConfig;


@SuppressWarnings("DefaultFileTemplate")
class DeviceId {
    protected final static String temporaryInTrackDeviceId = "INTTemporaryDeviceID";
    /**
     * Enum used throughout Sdk which controls what kind of ID  should use.
     */

    private static final String PREFERENCE_KEY_ID_ID = "ir.[" + BuildConfig.FLAVOR + "].android.api.DeviceId.id";
    private static final String PREFERENCE_KEY_ID_TYPE = "ir.[" + BuildConfig.FLAVOR + "].android.api.DeviceId.type";
    ModuleLog L;
    private String id;
    private DeviceIdType type;

    /**
     * Initialize DeviceId with Type of OPEN_UDID or ADVERTISING_ID
     *
     * @param type type of ID generation strategy
     */
    protected DeviceId(SharedPref store, DeviceIdType type, ModuleLog moduleLog) {
        if (type == null) {
            throw new IllegalStateException("Please specify DeviceId.Type, that is which type of device ID generation you want to use");
        } else if (type == DeviceIdType.DEVELOPER_SUPPLIED) {
            throw new IllegalStateException("Please use another DeviceId constructor for device IDs supplied by developer");
        }
        //setup the preferred device ID type
        this.type = type;
        L = moduleLog;

        //check if there wasn't a value set before
        retrieveId(store);
    }

    /**
     * Initialize DeviceId with Developer-supplied id string
     *
     * @param developerSuppliedId Device ID string supplied by developer
     */
    protected DeviceId(SharedPref store, String developerSuppliedId, ModuleLog moduleLog) {
        if (developerSuppliedId == null || "".equals(developerSuppliedId)) {
            throw new IllegalStateException("Please make sure that device ID is not null or empty");
        }
        //setup the preferred device ID type
        this.type = DeviceIdType.DEVELOPER_SUPPLIED;
        this.id = developerSuppliedId;
        L = moduleLog;

        //check if there wasn't a value set before
        retrieveId(store);
    }

    /**
     * Helper method for null safe comparison of current device ID and the one supplied to SDK.init
     *
     * @return true if supplied device ID equal to the one registered before
     */
    static boolean deviceIDEqualsNullSafe(final String id, DeviceIdType type, final DeviceId deviceId) {
        if (type == null || type == DeviceIdType.DEVELOPER_SUPPLIED) {
            final String deviceIdId = deviceId == null ? null : deviceId.getId();
            return (deviceIdId == null && id == null) || (deviceIdId != null && deviceIdId.equals(id));
        } else {
            return true;
        }
    }

    /**
     * Used during setup to retrieve the previously saved value
     *
     * @param store
     */
    private void retrieveId(SharedPref store) {
        //check if there is some stored value
        String storedId = store.getPreference(PREFERENCE_KEY_ID_ID);
        if (storedId != null && !storedId.isEmpty()) {
            //if there was a value saved previously, set it and it's type
            this.id = storedId;
            this.type = retrieveType(store, PREFERENCE_KEY_ID_TYPE);
        }
    }

    /**
     * Initialize device ID generation, that is start up required services and send requests.
     * Device ID is expected to be available after some time.
     * In some cases, Sdk can override ID generation strategy to other one, for example when
     * Google Play Services are not available and user chose Advertising ID strategy, it will fall
     * back to OpenUDID
     *
     * @param context         Context to use
     * @param store           SharedPref to store configuration in
     * @param raiseExceptions whether to raise exceptions in case of illegal state or not
     */
    protected void init(Context context, SharedPref store, boolean raiseExceptions) {
        DeviceIdType overriddenType = retrieveOverriddenType(store);

        // Some time ago some ID generation strategy was not available and SDK fell back to
        // some other strategy. We still have to use that strategy.
        if (overriddenType != null && overriddenType != type) {
            L.i("[DeviceId] Overridden device ID generation strategy detected: " + overriddenType + ", using it instead of " + this.type);
            type = overriddenType;
        }

        switch (type) {
            case DEVELOPER_SUPPLIED:
                // no initialization for developer id
                // we just store the provided value so that it's
                setAndStoreId(store, DeviceIdType.DEVELOPER_SUPPLIED, id);
                break;
            case OPEN_UDID:
                L.i("[DeviceId] Using OpenUDID");
                OpenUDIDAdapter.sync(context);
                setAndStoreId(store, DeviceIdType.OPEN_UDID, OpenUDIDAdapter.OpenUDID);
                break;
            case ADVERTISING_ID:
                if (AdvertisingIdAdapter.isAdvertisingIdAvailable()) {
                    L.i("[DeviceId] Using Advertising ID");
                    AdvertisingIdAdapter.setAdvertisingId(context, store, this);
                } else {
                    // Fall back to OpenUDID on devices without google play services set up
                    L.i("[DeviceId] Advertising ID is not available, falling back to OpenUDID");
                    OpenUDIDAdapter.sync(context);
                    setAndStoreId(store, DeviceIdType.OPEN_UDID, OpenUDIDAdapter.OpenUDID);
                }
                break;
        }
    }

    private void storeOverriddenType(SharedPref store, DeviceIdType type) {
        // Using strings is safer when it comes to extending Enum values list
        store.setPreference(PREFERENCE_KEY_ID_TYPE, type == null ? null : type.toString());
    }

    private DeviceIdType retrieveOverriddenType(SharedPref store) {
        return retrieveType(store, PREFERENCE_KEY_ID_TYPE);
    }

    private DeviceIdType retrieveType(SharedPref store, String preferenceName) {
        // Using strings is safer when it comes to extending Enum values list
        String typeString = store.getPreference(preferenceName);
        if (typeString == null) {
            return null;
        } else if (typeString.equals(DeviceIdType.DEVELOPER_SUPPLIED.toString())) {
            return DeviceIdType.DEVELOPER_SUPPLIED;
        } else if (typeString.equals(DeviceIdType.OPEN_UDID.toString())) {
            return DeviceIdType.OPEN_UDID;
        } else if (typeString.equals(DeviceIdType.ADVERTISING_ID.toString())) {
            return DeviceIdType.ADVERTISING_ID;
        } else if (typeString.equals(DeviceIdType.TEMPORARY_ID.toString())) {
            return DeviceIdType.TEMPORARY_ID;
        } else {
            return null;
        }
    }

    protected String getId() {
        if ((id == null || id.isEmpty()) && type == DeviceIdType.OPEN_UDID) {
            id = OpenUDIDAdapter.OpenUDID;
        }
        return id;
    }

    @SuppressWarnings("SameParameterValue")
    protected void setId(DeviceIdType type, String id) {
        L.w("[DeviceId] Device ID is " + id + " (type " + type + ")");
        this.type = type;
        this.id = id;
    }

    @SuppressWarnings("SameParameterValue")
    protected void switchToIdType(DeviceIdType type, Context context, SharedPref store) {
        L.w("[DeviceId] Switching to device ID generation strategy " + type + " from " + this.type);
        this.type = type;
        storeOverriddenType(store, type);
        init(context, store, false);
    }

    protected void changeToDeveloperProvidedId(SharedPref store, String newId) {
        setAndStoreId(store, DeviceIdType.DEVELOPER_SUPPLIED, newId);
    }

    protected void changeToId(Context context, SharedPref store, DeviceIdType type, String deviceId) {
        setAndStoreId(store, type, deviceId);
        init(context, store, false);
    }

    void setAndStoreId(SharedPref store, DeviceIdType type, String deviceId) {
        this.id = deviceId;
        this.type = type;

        store.setPreference(PREFERENCE_KEY_ID_ID, deviceId);
        store.setPreference(PREFERENCE_KEY_ID_TYPE, type.toString());
    }

    /**
     * Returns the current type which would be returned to the developer
     *
     * @return
     */
    protected DeviceIdType getType() {
        if (temporaryIdModeEnabled()) {
            return DeviceIdType.TEMPORARY_ID;
        }
        return type;
    }

    /**
     * Checks if temporary device ID mode is enabled by checking the currently set ID
     *
     * @return
     */
    protected boolean temporaryIdModeEnabled() {
        String id = getId();
        if (id == null) {
            return false;
        }

        return id.equals(temporaryInTrackDeviceId);
    }
}
