package sdk.main.core;

import android.app.Activity;
import android.os.Build;
import android.os.Bundle;

import androidx.fragment.app.Fragment;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.*;

import io.reactivex.annotations.NonNull;
import io.reactivex.annotations.Nullable;
import ir.intrack.android.sdk.BuildConfig;
import sdk.main.core.inappmessaging.InAppMessagingDisplayCallbacks;
import sdk.main.core.inappmessaging.display.InAppMessagingDisplay;
import sdk.main.core.inappmessaging.model.message.IAMMessage;

public class ModuleIAM extends ModuleBase implements ModuleEvents.InTrackEventListener {
    final static String IMPRESSION_EVENT_KEY = "IN_APP_DELIVERED";
    final static String CLICK_EVENT_KEY = "IN_APP_CLICK";
    final static String DISMISS_EVENT_KEY = "IN_APP_CLOSED";
    final static String FAILED_EVENT_KEY = "IN_APP_VIEW_FAILED";
    final static String RECEIVED_EVENT_KEY = "IN_APP_RECEIVED";
    //interface for SDK users
    final IAM iamInterface;
    private final Gson gson = new Gson();
    boolean isEnabled = false;
    LinkedHashMap<String, IAMMessage> parsedMessages = new LinkedHashMap<>();
    ModuleLog L;
    private Activity currentlyBoundActivity;
    private InAppMessagingDisplay inAppMessagingDisplay;
    private ModuleIAMFlows flows;

    ModuleIAM(CoreInternal coreInternal, Config config) {
        super(coreInternal);
        L = coreInternal.L;
        iamInterface = new IAM();
    }

    @NonNull
    public Map<String, IAMMessage> getParsedMessages() {
        return parsedMessages;
    }

    private void processStoredMessages() {
        try {
            String uid = _int.getUserId() == null ? _int.getDeviceID() : _int.getUserId();
            assert _int.config_.sharedPref != null;
            LinkedHashMap<String, IAMMessage> messagesList = _int.config_.sharedPref.getIAMessages(uid);
            parsedMessages = new LinkedHashMap<>();
            if (messagesList != null) {
                L.d("[ModuleIAM] we have totally" + messagesList.size() + "messages");

                long now = CoreProxy.getCurrentTimeMs();
                for (Map.Entry<String, IAMMessage> entry : messagesList.entrySet()) {
                    String msgId = entry.getKey();
                    IAMMessage msg = entry.getValue();


                    if (msg.getEndDate() != null && msg.getEndDate().getTime() > now) {
                        parsedMessages.put(msgId, msg);
                    }
                }

                processMessages();
                L.d("[ModuleIAM] we have " + parsedMessages.size() + " stored message");
            }
        } catch (Exception e) {
            L.d("[ModuleIAM] processStoredMessages faced with error : " + e.getMessage());
        }
    }


    public void customNavigatedTo(String customScreen) {
        if (isEnabled) {
            flows.customNavigatedTo(customScreen);
        }
    }

    public void customNavigatedTo(String customScreen, Map<String, Object> screenData) {
        if (isEnabled) {
            flows.customNavigatedTo(customScreen, screenData);
        }
    }

    public void updateScreenData(String customScreen, Map<String, Object> screenData) {
        if (isEnabled) {
            flows.updateScreenData(customScreen, screenData);
        }
    }

    public void updateScreenData(Activity activity, Map<String, Object> screenData) {
        if (isEnabled) {
            flows.updateScreenData(activity, screenData);
        }
    }

    public void updateScreenData(Fragment fragment, Map<String, Object> screenData) {
        if (isEnabled) {
            flows.updateScreenData(fragment, screenData);
        }
    }

    synchronized void updateUserId(String userId) {
        MqttController.INSTANCE.updateUserId(userId);

        //TODO:update flow
        processStoredMessages();
    }

    synchronized void updateMessages(String _messages) {
        if (!_int.isInitialized()) {
            L.e("[" + BuildConfig.FLAVOR + "].init must be called before updateMessages");
            return;
        }

        if (_messages == null || _messages.isEmpty()) {
            L.e("[ModuleIAM] Trying to update messages with empty or null value, ignoring request");
            return;
        }

        try {
            ArrayList<IAMMessage> newMessages = gson.fromJson(_messages, new TypeToken<ArrayList<IAMMessage>>() {
            }.getType());
            updateMessages(newMessages);
        } catch (Exception e) {
            L.e("[ModuleIAM] updating inApp Messages failed " + e.getMessage());
        }
    }

    synchronized void updateMessages(List<IAMMessage> newMessages) {
        if (!_int.isInitialized()) {
            L.e(BuildConfig.FLAVOR + "].init must be called before updateMessages");
            return;
        }

        if (newMessages == null || newMessages.isEmpty()) {
            L.e("[ModuleIAM] Trying to update messages with empty or null value, ignoring request");
            return;
        }

        try {
            if (L.logEnabled()) {
                L.d("[ModuleIAM] receive [" + newMessages.size() + "] new messages");
            }
            long now = CoreProxy.getCurrentTimeMs();
            for (IAMMessage msg : newMessages) {
                if (msg == null) continue;
                String msgId = msg.getId();
                if (msgId == null) {
                    msgId = String.valueOf(CoreProxy.getCurrentTimeMs());
                    msg.setId(msgId);
                }
                if ((msg.getEndDate() != null && msg.getEndDate().getTime() < now) || msg.getShowCount() == 0) {
                    parsedMessages.remove(msgId);
                } else if (!parsedMessages.containsKey(msgId)) {
                    Map<String, Object> eventData = new HashMap<>();
                    eventData.put("id", msgId);
                    _int.events().recordSystemEvent(RECEIVED_EVENT_KEY, eventData);
                    parsedMessages.put(msgId, msg);
                }
            }

            L.d("[ModuleIAM] we have totally " + parsedMessages.size() + " message");

            storeMessages();
            processMessages();
        } catch (Exception e) {
            L.e("[ModuleIAM] updating inApp Messages failed " + e.getMessage());
        }
    }

    protected void storeMessages() {
        String id = _int.getUserId() == null ? _int.getDeviceID() : _int.getUserId();
        assert _int.config_.sharedPref != null;
        _int.config_.sharedPref.setIAMessages(parsedMessages, id);
    }

    private void processMessages() {
        flows.processMessages();
    }

    @Override
    void deviceIdChanged() {
        MqttController.INSTANCE.updateDeviceId(_int.getDeviceID());
    }

    @Override
    public void initFinished(Config config) {
        if (!_int.connectionQueue_.getDeviceId().temporaryIdModeEnabled()) {

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && config.enableInAppMessaging) {
                L.v("[ModuleIAM] Initialising");

                if (config.autoTrackingUseShortName) {
                    L.d("[ModuleIAM] Enabling automatic view tracking short names");
                }

                flows = new ModuleIAMFlows(this);

                if (config.application != null && config.enableInAppMessaging) {
                    inAppMessagingDisplay = new InAppMessagingDisplay();
                    if (config.inAppListener != null) {
                        inAppMessagingDisplay.setListener(config.inAppListener);
                    }

                    processStoredMessages();

                    assert config.context != null;
                    MqttController.INSTANCE.init(config.context.getApplicationContext());
                    MqttController.INSTANCE.initIAM();


                    isEnabled = true;

                    if (!config.inAppMessagingActivityAndFragmentAutoTracking) {
                        L.d("[ModuleIAMFlows] screenTracking is off, so we add a starting page");
                        this.customNavigatedTo("[INT]_START_PAGE", null);
                    }

                }
            }
        }
    }

    @Override
    public void callbackOnActivityCreated(Activity activity, Bundle bundle) {
        if (isEnabled) {
            flows.onActivityCreated(activity, bundle);
        }
    }

    @Override
    public void callbackOnActivityStarted(Activity activity) {
        if (isEnabled) {
            flows.onActivityStarted(activity);
        }
    }

    @Override
    public void callbackOnActivityResumed(Activity activity) {
        if (isEnabled && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (activity != null) {
                currentlyBoundActivity = activity;
                flows.onActivityResumed(activity);
            }
            if (inAppMessagingDisplay != null) {
                inAppMessagingDisplay.onActivityResumed(activity);
            }
            MqttController.INSTANCE.onActivityResume();
        }
    }

    @Override
    public void callbackOnActivityPaused(Activity activity) {
        if (isEnabled && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (activity != null) {
                currentlyBoundActivity = activity;
                flows.onActivityPaused(activity);
            }
            if (inAppMessagingDisplay != null) {
                inAppMessagingDisplay.onActivityPaused(activity);
            }
            MqttController.INSTANCE.onActivityPaused();
        }
    }

    @Override
    public void callbackOnActivityStopped(Activity activity) {
        if (isEnabled) {
            flows.onActivityStopped(activity);
        }
    }

    @Override
    public void callbackOnActivitySaveInstanceState(Activity activity, Bundle bundle) {
        if (isEnabled) {
            flows.onActivitySaveInstanceState(activity, bundle);
        }
    }

    @Override
    public void callbackOnActivityDestroyed(Activity activity) {
        if (isEnabled) {
            flows.onActivityDestroyed(activity);
        }
    }

    @Override
    void onActivityStarted(Activity activity) {
        if (isEnabled && activity != null) {
            currentlyBoundActivity = activity;
        }
    }

    void displayProperMessage(final IAMMessage message) {
        try {
            if (message != null && currentlyBoundActivity != null) {
                inAppMessagingDisplay.displayMessage(currentlyBoundActivity, message, new InAppMessagingDisplayCallbacks() {
                    @Override
                    public void impressionDetected() {
                        assert message.getId() != null;
                        String msgId = message.getId();

                        Map<String, Object> eventData = new HashMap<>();
                        eventData.put("id", msgId);

                        _int.events().recordSystemEvent(IMPRESSION_EVENT_KEY, eventData);

                        IAMMessage msg = parsedMessages.get(msgId);
                        if (msg != null) {
                            try {
                                msg.decreaseShowCount();
                                int showCount = msg.getShowCount();
                                L.i("[ModuleIAM] impressionDetected on message with message data:" + msg);

                                if (showCount <= 0) {
                                    parsedMessages.remove(msgId);
                                } else {
                                    parsedMessages.put(msgId, msg);
                                }
                                storeMessages();
                            } catch (Exception e) {
                                L.e("[ModuleIAM] impressionDetected couldn't remove from parsedMessage" + e.getMessage());
                            }
                        } else {
                            L.w("[ModuleIAM] impressionDetected with id" + msgId + " but not found in parsedMessage (" + parsedMessages.size() + " message)");
                        }
                    }

                    @Override
                    public void messageDismissed(InAppMessagingDismissType dismissType) {
                        assert message.getId() != null;
                        String msgId = message.getId();

                        Map<String, Object> eventData = new HashMap<>();
                        eventData.put("id", msgId);
                        eventData.put("dismissType", dismissType);

                        _int.events().recordSystemEvent(DISMISS_EVENT_KEY, eventData);

                        IAMMessage msg = parsedMessages.get(msgId);
                        if (msg != null) {
                            try {
                                msg.setShowCount(0);
                                L.i("[ModuleIAM] try to remove message on messageDismissed event with message data:" + msg + "from parsedMessages");

                                parsedMessages.remove(msgId);
                                storeMessages();
                            } catch (Exception e) {
                                L.e("[ModuleIAM] messageDismissed couldn't remove from parsedMessage" + e.getMessage());
                            }
                        } else {
                            L.w("[ModuleIAM] messageDismissed with id" + msgId + " but not found in parsedMessage (" + parsedMessages.size() + " message)");
                        }
                    }

                    @Override
                    public void messageClicked(@Nullable String actionUrl) {
                        assert message.getId() != null;
                        String msgId = message.getId();

                        Map<String, Object> eventData = new HashMap<>();
                        eventData.put("id", msgId);
                        if (actionUrl != null && !actionUrl.isEmpty()) {
                            eventData.put("actionUrl", actionUrl);
                        }

                        _int.events().recordSystemEvent(CLICK_EVENT_KEY, eventData);

                        IAMMessage msg = parsedMessages.get(msgId);
                        if (msg != null) {
                            try {
                                msg.setShowCount(0);
                                L.i("[ModuleIAM] try to remove message on messageClicked event with message data:" + msg + "from parsedMessages");

                                parsedMessages.remove(msgId);
                                storeMessages();
                            } catch (Exception exception) {
                                L.e("[ModuleIAM] messageClicked couldn't remove from parsedMessage" + exception.getMessage());
                            }
                        } else {
                            L.w("[ModuleIAM] messageClicked with id" + msgId + " but not found in parsedMessage (" + parsedMessages.size() + " message)");
                        }
                    }

                    @Override
                    public void displayErrorEncountered(InAppMessageListener.ErrorMessageReason inAppMessagingErrorReason) {
                        assert message.getId() != null;
                        String msgId = message.getId();

                        Map<String, Object> eventData = new HashMap<>();
                        eventData.put("id", msgId);
                        eventData.put("reason", inAppMessagingErrorReason);
                        _int.events().recordSystemEvent(FAILED_EVENT_KEY, eventData);

                        IAMMessage msg = parsedMessages.get(msgId);
                        if (msg != null) {
                            try {
                                msg.setShowCount(0);
                                parsedMessages.remove(msgId);
                                storeMessages();
                            } catch (Exception e) {
                                L.e("[ModuleIAM] displayErrorEncountered couldn't remove from parsedMessage" + e.getMessage());
                            }
                        } else {
                            L.w("[ModuleIAM] displayErrorEncountered with id" + msgId + " but not found in parsedMessage (" + parsedMessages.size() + " message)");
                        }
                    }
                });
            }
        } catch (Exception e) {
            L.e("[ModuleIAM] displaying message(" + message + ")  got error" + e.getMessage());
        }
    }

    @Override
    public void onEventRaised(@NonNull final String key, @Nullable final Map<String, Object> segmentation, @Nullable UserDetails userDetails, @NonNull UtilsTime.Instant instant, boolean processedSegmentation, boolean isSystemEvent) {
        if (!isEnabled) {
            return;
        }
        flows.onEventRaised(key, segmentation, userDetails, instant, processedSegmentation, isSystemEvent);
    }

    public static class IAM {
    }
}
