package sdk.main.core

import android.app.Activity
import android.app.Application
import android.os.Bundle
import android.view.View
import androidx.annotation.Keep
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.ProcessLifecycleOwner
import ir.intrack.android.sdk.prefs.CampaignDataPrefs
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.consumeAsFlow
import kotlinx.coroutines.flow.distinctUntilChangedBy
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.shareIn
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import sdk.main.core.ext.MapStateFlow
import sdk.main.core.ext.nowMs
import sdk.main.core.ext.refreshWith
import sdk.main.core.inappmessaging.CustomScreenTitle
import sdk.main.core.inappmessaging.model.message.EventModel
import sdk.main.core.inappmessaging.model.message.IAMMessage
import sdk.main.core.inappmessaging.model.message.accepts
import sdk.main.core.inappmessaging.model.message.doesScreenCompliesTo
import java.lang.ref.WeakReference

internal val activityAndFragmentAutoTracking
    get() = CoreInternal.sharedInstance()?.config?.inAppMessagingActivityAndFragmentAutoTracking
        ?: true

internal val clock = MutableStateFlow(nowMs)
internal fun refreshClock() {
    clock.value = nowMs
}

private const val CLOCK_INVALIDATION_INTERVAL = 1000L

internal val IAMScope = CoroutineScope(SupervisorJob() + Dispatchers.Main)

@Volatile
private var clockInvalidationJob: Job? = null

@Synchronized
internal fun stopClockInvalidation() {
    clockInvalidationJob?.cancel()
    clockInvalidationJob = null
}

@Synchronized
internal fun startClockInvalidation() {
    stopClockInvalidation()
    clockInvalidationJob = IAMScope.launch(Dispatchers.Main) {
        while (isActive) {
            delay(CLOCK_INVALIDATION_INTERVAL)
            if (isActive) {
                refreshClock()
            }
        }
    }
}

private var screenDataCounter = 0

internal data class ScreenData(
    var timeSpentOn: Double = 0.0,
    var lastTimeCheck: Long = nowMs,
    var state: Lifecycle.State = Lifecycle.State.INITIALIZED,
    var count: Int = screenDataCounter++,
    var screenData: Map<String?, Any?>? = null,
) {
    override fun toString(): String {
        return "ScreenData(timeSpentOn=$timeSpentOn, lastTimeCheck=$lastTimeCheck, state=$state, count=$count)"
    }
}

internal typealias CustomScreen = String

internal enum class TrackingScreenType {
    Activity,
    Fragment,
    NamedScreen
}

internal class ScreenUnion private constructor() {
    @Deprecated("Don't use activity directly, use fromAll instead")
    private var activity: WeakReference<Activity>? = null

    @Deprecated("Don't use fragment directly, use fromAll instead")
    private var fragment: WeakReference<Fragment>? = null

    @Deprecated("Don't use customScreen directly, use fromAll instead")
    private var customScreen: CustomScreen? = null

    @Suppress("DEPRECATION")
    constructor(activity: Activity) : this() {
        this.activity = WeakReference(activity)
    }

    @Suppress("DEPRECATION")
    constructor(fragment: Fragment) : this() {
        this.fragment = WeakReference(fragment)
    }

    @Suppress("DEPRECATION")
    constructor(customScreen: CustomScreen) : this() {
        this.customScreen = customScreen
    }


    @Suppress("DEPRECATION")
    private fun <T> fromAll(
        fromActivity: (Activity) -> T,
        fromFragment: (Fragment) -> T,
        fromCustomScreen: (CustomScreen) -> T,
    ) =
        activity?.get()?.let { fromActivity(it) } ?: fragment?.get()?.let { fromFragment(it) }
        ?: customScreen?.let { fromCustomScreen(it) }

    val filledItem
        get() = fromAll(
            { act -> act },
            { fragment -> fragment },
            { customScreen -> customScreen }
        )

    override fun toString(): String {
        return "ScreenUnion{" +
                fromAll(
                    { act -> "Activity=$act" },
                    { fragment -> "Fragment=$fragment" },
                    { customScreen -> "CustomScreen=$customScreen" }
                ) + "}"
    }

    override fun equals(other: Any?) =
        if (other == null || other !is ScreenUnion)
            false
        else
            filledItem == other.filledItem

    override fun hashCode() = filledItem.hashCode()

    private val shouldUseShortNames
        get() = CoreInternal.sharedInstance()?.config?.autoTrackingUseShortName ?: false

    val screenTitle
        get() = fromAll(
            { act ->
                (act as? CustomScreenTitle)?.customScreenTitle
                    ?: if (shouldUseShortNames) act.javaClass.simpleName else act.javaClass.name
            },
            { fragment ->
                (fragment as? CustomScreenTitle)?.customScreenTitle
                    ?: if (shouldUseShortNames) fragment.javaClass.simpleName else fragment.javaClass.name
            },
            { customScreen -> customScreen }
        ) ?: ""

    val trackingScreenType
        get() = fromAll(
            fromActivity = { TrackingScreenType.Activity },
            fromFragment = { TrackingScreenType.Fragment },
            fromCustomScreen = { TrackingScreenType.NamedScreen }
        ) ?: TrackingScreenType.Activity
}

internal val Activity.screenUnion get() = ScreenUnion(this)
internal val Fragment.screenUnion get() = ScreenUnion(this)
internal val CustomScreen.screenUnion get() = ScreenUnion(this)


internal class ModuleIAMFlows(
    private val iamModule: ModuleIAM,
) : Application.ActivityLifecycleCallbacks, ModuleEvents.InTrackEventListener {

    private inline val parsedMessages: MutableMap<String, IAMMessage> get() = iamModule.parsedMessages

    private val campaignDataPrefs = CampaignDataPrefs()

    private val screenTrackerMap = MapStateFlow<ScreenUnion?, ScreenData>(HashMap())
    private val eventsChannel = Channel<EventModel>()
    private val eventsFlow = eventsChannel
        .consumeAsFlow()
        .shareIn(IAMScope, SharingStarted.Eagerly, 0)

    private val sessionTotalTime = MutableStateFlow(0L)
    private var sessionTotalTimeLastTimeCheck: Long = nowMs


    override fun onEventRaised(
        key: String,
        segmentation: MutableMap<String, Any?>?,
        userDetails: UserDetails<*>?,
        instant: UtilsTime.Instant,
        processedSegmentation: Boolean,
        isSystemEvent: Boolean,
    ) {
        try {
            val event = EventModel(key, segmentation ?: hashMapOf(), userDetails)
            eventsChannel.trySend(event)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    @Keep
    val clockObserver = IAMScope.launch(Dispatchers.Main) {
        clock.collectLatest { time ->
            for ((_, data) in screenTrackerMap) {
                if (data.state.isAtLeast(Lifecycle.State.RESUMED))
                    data.timeSpentOn = data.timeSpentOn + (time - data.lastTimeCheck)
                data.lastTimeCheck = time
            }

            if (ProcessLifecycleOwner.get().lifecycle.currentState.isAtLeast(Lifecycle.State.RESUMED))
                sessionTotalTime.value =
                    sessionTotalTime.value + (time - sessionTotalTimeLastTimeCheck)
            sessionTotalTimeLastTimeCheck = time
        }
    }

    //region union tracker
    internal fun ScreenUnion.onCreated() {
        screenTrackerMap[this] = ScreenData(state = Lifecycle.State.CREATED)
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onCreated ${this.screenTitle}")
    }

    internal fun ScreenUnion.onStarted() {
        screenTrackerMap[this]?.state = Lifecycle.State.STARTED
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onStarted ${this.screenTitle}")
    }

    internal fun ScreenUnion.onResumed() {
        screenTrackerMap[this]?.run {
            lastTimeCheck = nowMs
            state = Lifecycle.State.RESUMED

            for ((_, msg) in parsedMessages) {
                campaignDataPrefs.increaseScreenViewCount(msg)
            }
        }
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onResumed ${this.screenTitle}")
    }

    internal fun ScreenUnion.onPaused() {
        screenTrackerMap[this]?.state = Lifecycle.State.STARTED
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onPaused ${this.screenTitle}")
    }

    internal fun ScreenUnion.onStopped() {
        screenTrackerMap[this]?.state = Lifecycle.State.CREATED
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onStopped ${this.screenTitle}")
    }

    internal fun ScreenUnion.onDestroyed() {
        screenTrackerMap.remove(this)
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] onDestroyed ${this.screenTitle}")
    }

    private fun ScreenUnion.updateScreenData(screenData: Map<String?, Any?>?) {
        screenTrackerMap[this]?.screenData = screenData
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] updateScreenData ${this.screenTitle}")
    }
    //endregion

    //region Activity trackers
    private val activityToFragmentStateMonitorMap =
        HashMap<FragmentActivity, FragmentStateMonitor>()

    private fun FragmentActivity.ensureFragmentStateMonitor() {
        if (!activityToFragmentStateMonitorMap.contains(this)) {
            val fragmentStateMonitor = FragmentStateMonitor()
            activityToFragmentStateMonitorMap[this] = fragmentStateMonitor
            supportFragmentManager.registerFragmentLifecycleCallbacks(fragmentStateMonitor, true)
        }
    }

    private fun FragmentActivity.removeFragmentStateMonitor() {
        activityToFragmentStateMonitorMap.remove(this)?.let { removedMonitor ->
            supportFragmentManager.unregisterFragmentLifecycleCallbacks(removedMonitor)
        }
    }

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onCreated()
        (activity as? FragmentActivity)?.ensureFragmentStateMonitor()
    }

    override fun onActivityStarted(activity: Activity) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onStarted()
        (activity as? FragmentActivity)?.ensureFragmentStateMonitor()
    }

    override fun onActivityResumed(activity: Activity) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onResumed()
    }

    override fun onActivityPaused(activity: Activity) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onPaused()
    }

    override fun onActivityStopped(activity: Activity) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onStopped()
    }

    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
        if (!activityAndFragmentAutoTracking) return

    }

    override fun onActivityDestroyed(activity: Activity) {
        if (!activityAndFragmentAutoTracking) return
        activity.screenUnion.onDestroyed()
        (activity as? FragmentActivity)?.removeFragmentStateMonitor()
    }

    fun updateScreenData(activity: Activity, screenData: Map<String?, Any?>?) {
        activity.screenUnion.updateScreenData(screenData)
    }
    //endregion

    //region FragmentTracker
    private inner class FragmentStateMonitor : FragmentManager.FragmentLifecycleCallbacks() {

        override fun onFragmentViewCreated(
            fm: FragmentManager,
            f: Fragment,
            v: View,
            savedInstanceState: Bundle?,
        ) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onCreated()
        }

        override fun onFragmentStarted(fm: FragmentManager, f: Fragment) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onStarted()
        }

        override fun onFragmentResumed(fm: FragmentManager, f: Fragment) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onResumed()
        }

        override fun onFragmentPaused(fm: FragmentManager, f: Fragment) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onPaused()
        }

        override fun onFragmentStopped(fm: FragmentManager, f: Fragment) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onStopped()
        }

        override fun onFragmentViewDestroyed(fm: FragmentManager, f: Fragment) {
            if (!activityAndFragmentAutoTracking) return
            f.screenUnion.onDestroyed()
        }
    }

    fun updateScreenData(f: Fragment, screenData: Map<String?, Any?>?) {
        f.screenUnion.updateScreenData(screenData)
    }
    //endregion

    //region CustomScreenTracking
    @JvmOverloads
    fun customNavigatedTo(customScreen: CustomScreen, screenData: Map<String?, Any?>? = null) {
        val union = customScreen.screenUnion
        CoreInternal.sharedInstance()?.L?.d("[ModuleIAMFlows] custom navigation to $union")

        val mapKeys = screenTrackerMap.keys.toTypedArray()

        for (k in mapKeys) {
            k ?: continue //it's unexpected anyway

            // we have to remove any other custom screen in order to have only one active
            if (k.trackingScreenType == TrackingScreenType.NamedScreen) {
//                if(k != union){
                k.onPaused() //it seems unnecessarily, but I put it here if we wanted to do anything in the future :D
                k.onStopped()
                k.onDestroyed()
//                }
            }
        }

        // if we don't have this screen, it means it does not exist yet :D otherwise we won't override it :D
        if (screenTrackerMap[union] == null) {
            union.onCreated()
        }
        // Although you might have a tendency to turn this into an `else if`, please DON'T! it is rational to check it after onCreate
        // maybe we didn't have that screen in our map, so we have to ask from it once more :D
        if (screenTrackerMap[union]?.state?.isAtLeast(Lifecycle.State.STARTED) == false) { //it was newly created, or had stopped recently (from the code above)
            union.onStarted()
        }
        //same reason for NOT using else if
        if (screenTrackerMap[union]?.state?.isAtLeast(Lifecycle.State.RESUMED) == false) { //it was newly created->started, or had paused recently (again, from the code above)
            union.onResumed()
        }

        union.updateScreenData(screenData)
    }

    fun updateScreenData(customScreen: CustomScreen, screenData: Map<String?, Any?>?) {
        customScreen.screenUnion.updateScreenData(screenData)
    }
    //endregion

    private lateinit var collectScope: CoroutineScope
    private var messagesFlows: List<Flow<IAMMessage?>> = arrayListOf()
    private var messagesFlowsJobs: List<Job?> = arrayListOf()
    private var shownMessageIds = mutableListOf<String>()

    fun processMessages() {
        CoreInternal.sharedInstance()?.L?.i("[ModuleIAMFlows] start processing messages $parsedMessages")

        if (parsedMessages.isEmpty())
            stopClockInvalidation()
        else
            startClockInvalidation()

        messagesFlowsJobs.forEach {
            it?.cancel()
        }
        if (this::collectScope.isInitialized)
            collectScope.cancel()
        collectScope = CoroutineScope(SupervisorJob() + Dispatchers.Main)

        messagesFlows = parsedMessages.map { it.value.generateMessageFlow() }
            .filterNotNull()


        messagesFlowsJobs = messagesFlows.map { f ->

            collectScope.launch(Dispatchers.Main) {
                f.collectLatest {
                    if (it != null) {
                        CoreInternal.sharedInstance()?.L?.i("[ModuleIAMFlows] GoingToShow $it")
                        try {
                            iamModule.displayProperMessage(it)
                            shownMessageIds.add(it.id ?: "null")
                            CoreInternal.sharedInstance()?.L?.i("[ModuleIAMFlows] shownMessageIds $shownMessageIds")
                        } catch (e: Exception) {
                            val eventData: MutableMap<String, Any> = HashMap()
                            eventData["id"] = it.id ?: ""
                            eventData["reason"] = e.message ?: ""
                            CoreInternal.sharedInstance().events().recordSystemEvent(
                                ModuleIAM.FAILED_EVENT_KEY, eventData
                            )
                            CoreInternal.sharedInstance()?.L?.e("[ModuleIAMFlows] displayProperMessage Error $e")
                            parsedMessages.remove(it.id)
                            iamModule.storeMessages()
                        }
                    }
                }
            }
        }
    }

    private fun IAMMessage.generateMessageFlow(): Flow<IAMMessage?> {
        val conditions = whenAndWhereConditions

        val whenAndWhereFlow = if (conditions != null) {
            val matchScreenFlow = screenTrackerMap.refreshWith(clock).map { tracker ->
                val matchedScreens = tracker.filter { item ->
                    val (screenUnion, data) = item
                    if (screenUnion?.filledItem == null) return@filter false

                    data.state.isAtLeast(Lifecycle.State.RESUMED) &&
                            (conditions.activityNames?.doesScreenCompliesTo(screenUnion) ?: true) &&
                            (conditions.activityData?.accepts(data.screenData) ?: true)
                }

                matchedScreens
                //TODO: add screenData
            }


            val screenTimeFlows = matchScreenFlow.map { matchedScreens ->
                matchedScreens.mapNotNull { item ->
                    val screenData = item.value
                    val screenUnion = item.key

                    val screenViewCount = screenUnion?.let {
                        campaignDataPrefs.screenViewCount(this)
                    } ?: 0


                    if (conditions.activityViews.accepts(screenViewCount) &&
                        screenData.timeSpentOn >= conditions.timeSpentOnActivity &&
                        sessionTotalTime.value >= conditions.timeSpentOnApp
                    )
                        item
                    else
                        null
                }
            }

            val distinctMessage = screenTimeFlows
                .filter { it.isNotEmpty() }
                .distinctUntilChangedBy { entry ->
                    arrayListOf(
                        entry,
                        this@generateMessageFlow.id
                    )
                }
                .map { this@generateMessageFlow }

            val eventFilter = conditions.customEvent
            if (eventFilter != null && eventFilter.terms?.isNotEmpty() == true) {
                eventsFlow.filter { event ->
                    eventFilter.accepts(event)
                }.combine(distinctMessage) { _, dm ->
                    dm
                }
            } else {
                distinctMessage
            }
        } else {
            MutableStateFlow(this@generateMessageFlow)
        }

        val scheduleFiltered = whenAndWhereFlow.filter {
            if (it.endDate != null) {
                it.endDate?.time!! > nowMs && it.showCount > 0 && !shownMessageIds.contains(it.id)
            } else {
                it.showCount > 0 && !shownMessageIds.contains(it.id)
            }
        }

        return scheduleFiltered
    }


}
