package sdk.main.core

import android.app.Application
import android.content.Context
import android.util.Base64
import androidx.lifecycle.DefaultLifecycleObserver
import com.gojek.courier.Courier
import com.gojek.courier.Message
import com.gojek.courier.QoS
import com.gojek.courier.messageadapter.gson.GsonMessageAdapterFactory
import com.gojek.courier.streamadapter.rxjava2.RxJava2StreamAdapterFactory
import com.gojek.mqtt.auth.Authenticator
import com.gojek.mqtt.client.MqttClient
import com.gojek.mqtt.client.config.ExperimentConfigs
import com.gojek.mqtt.client.config.PersistenceOptions
import com.gojek.mqtt.client.config.v3.MqttAndroidLogger
import com.gojek.mqtt.client.config.v3.MqttV3Configuration
import com.gojek.mqtt.client.factory.MqttClientFactory
import com.gojek.mqtt.client.listener.MessageListener
import com.gojek.mqtt.client.model.ConnectionState.CONNECTED
import com.gojek.mqtt.client.model.ConnectionState.INITIALISED
import com.gojek.mqtt.client.model.MqttMessage
import com.gojek.mqtt.event.EventHandler
import com.gojek.mqtt.event.MqttEvent
import com.gojek.mqtt.model.AdaptiveKeepAliveConfig
import com.gojek.mqtt.model.MqttConnectOptions
import com.gojek.workmanager.pingsender.WorkManagerPingSenderConfig
import com.gojek.workmanager.pingsender.WorkPingSenderFactory
import io.reactivex.disposables.CompositeDisposable
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import org.json.JSONObject
import sdk.main.core.ext.HistoryStateFlow
import sdk.main.core.ext.withHistory


private val String.toAppId
    get(): Long {

        val pidBytes = Base64.decode(this, Base64.DEFAULT);

        var productId = 0L
        for (b in pidBytes) {
            productId = (productId shl 8) + (b.toInt() and 0xFF)
        }
        return productId
    }

internal object MqttController : DefaultLifecycleObserver,
    CoroutineScope by CoroutineScope(Dispatchers.Main + SupervisorJob()) {
    private lateinit var app: Application

    private var inited = false;

    private val mqttConfig = HistoryStateFlow(MqttConfig())
    private val isForeground = MutableStateFlow(false)

    private val userId = MutableStateFlow<String?>(null)
    private val deviceId = MutableStateFlow<String?>(null)

    private lateinit var mqttClient: MqttClient
    private lateinit var courierService: MqttCourierService

    private fun generateTopic() = generateTopic(userId.value, deviceId.value)
    private fun generateTopic(userId: String?, deviceId: String?): String? {
        val id = if (userId != null)
            "U$userId"
        else if (deviceId != null)
            "A$deviceId"
        else
            null //TODO: check with Farhan

        val appId = CoreInternal.sharedInstance()?.config?.appKey?.toAppId

        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: TOPIC: IN_APP/$appId/$id")

        return if (appId != null && id != null)
            "IN_APP/$appId/$id"
        else
            null
    }

    private fun generateMqttDeviceId() = generateMqttDeviceId(deviceId.value)
    private fun generateMqttDeviceId(deviceId: String?): String? {
        val id = if (deviceId != null)
            "$deviceId"
        else
            null

        val appId = CoreInternal.sharedInstance()?.config?.appKey?.toAppId

        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: DEVICE ID: ANDROID_${appId}_$id")

        return if (appId != null && id != null)
            "ANDROID_${appId}_$id"
        else
            null
    }

    @OptIn(FlowPreview::class)
    private val topic = userId.combine(deviceId) { uid, did ->
        generateTopic(uid, did)
    }
        .debounce(100)
        .stateIn(this, SharingStarted.Eagerly, null)
        .withHistory(this)

    private val shouldConnectAndConfig = mqttConfig.combine(isForeground) { c, id -> c to id }

    fun onActivityResume() {
        this.isForeground.value = true
    }

    fun onActivityPaused() {
        this.isForeground.value = false
    }

    fun updateMqttConfig(config: MqttConfig) {
        this.mqttConfig.value = config
    }

    fun updateUserId(userId: String?) {
        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: updatedUserId: $userId")
        this.userId.value = userId
    }

    fun updateDeviceId(deviceId: String?) {
        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: updatedDeviceId: $deviceId")
        this.deviceId.value = deviceId
    }

    private operator fun <T> Flow<T>.invoke(action: suspend (value: T) -> Unit) {
        launch { collectLatest(action) }
    }

    private val mqttConnectionState = MutableStateFlow(INITIALISED)
    private val eventHandler = object : EventHandler {
        override fun onEvent(mqttEvent: MqttEvent) {

            CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: Received event: $mqttEvent")
            mqttConnectionState.value = mqttClient.getCurrentState()

            (mqttEvent as? MqttEvent.MqttConnectFailureEvent)?.let {
                it.exception.printStackTrace()
            }
        }
    }

    private fun initialiseCourier() {
        val mqttConfig = MqttV3Configuration(
            logger = MqttAndroidLogger,
            eventHandler = eventHandler,
            authenticator = object : Authenticator {
                override fun authenticate(
                    connectOptions: MqttConnectOptions,
                    forceRefresh: Boolean,
                ): MqttConnectOptions {
                    return connectOptions.newBuilder()
                        .password(mqttConfig.value.password)
                        .build()
                }
            },
//            mqttInterceptorList = if(BuildConfig.DEBUG)
//                listOf(MqttChuckInterceptor(app, MqttChuckConfig(retentionPeriod = Period.ONE_HOUR)))
//            else
//                listOf(),
            mqttInterceptorList = listOf(),
            persistenceOptions = PersistenceOptions.PahoPersistenceOptions(100, false),
            experimentConfigs = ExperimentConfigs(
                adaptiveKeepAliveConfig = AdaptiveKeepAliveConfig(
                    lowerBoundMinutes = 1,
                    upperBoundMinutes = 9,
                    stepMinutes = 2,
                    optimalKeepAliveResetLimit = 10,
                    pingSender = WorkPingSenderFactory.createAdaptiveMqttPingSender(
                        app,
                        WorkManagerPingSenderConfig()
                    )
                ),
                inactivityTimeoutSeconds = 45,
                activityCheckIntervalSeconds = 30,
                incomingMessagesTTLSecs = 60,
                incomingMessagesCleanupIntervalSecs = 10,
            ),
            pingSender = WorkPingSenderFactory.createMqttPingSender(
                app,
                WorkManagerPingSenderConfig()
            )
        )
        mqttClient = MqttClientFactory.create(app, mqttConfig)

        val configuration = Courier.Configuration(
            client = mqttClient,
            streamAdapterFactories = listOf(RxJava2StreamAdapterFactory()),
            messageAdapterFactories = listOf(GsonMessageAdapterFactory()),
            logger = MqttAndroidLogger
        )
        val courier = Courier(configuration)
        courierService = courier.create()
    }

    fun init(context: Context) {
        if (inited) {
            CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: initialized before!")
            return
        }
        inited = true
        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: init connection ")
        app = context.applicationContext as Application

        updateUserId(CoreInternal.sharedInstance()?.userId)
        updateDeviceId(CoreInternal.sharedInstance()?.deviceID)

        initialiseCourier()

        //workaround for RN
        mqttClient.connect(
            mqttConnectOptions(
                generateMqttDeviceId() ?: "clientId"
            )
        ) //clientId

//        shouldConnectAndConfig { pair ->
//            val (config, shouldConnect) = pair
//
//            CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: shouldConnectAndConfig ${shouldConnect}")
//
//            if(mqttClient.getCurrentState() in listOf(CONNECTING, CONNECTED)){
//                CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: Calling Disconnect")
//                mqttClient.disconnect()
//            }
//
//            if(shouldConnect){
//                CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: Calling Connect")
//                mqttClient.connect(config.mqttConnectOptions(generateMqttDeviceId() ?: "clientId")) //clientId
//            }
//        }

        val appId = CoreInternal.sharedInstance()?.config?.appKey?.toAppId
        if (appId != null) {
            val configTopic = "APP_CONFIG/$appId"
            CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: connect to config topic: $configTopic")

            mqttClient.subscribe(configTopic to QoS.TWO);
            mqttClient.addMessageListener(configTopic, object : MessageListener {
                override fun onMessageReceived(mqttMessage: MqttMessage) {
                    try {
                        val payload = String((mqttMessage.message as Message.Bytes).value)
                        val data = JSONObject(payload)
                        CoreInternal.sharedInstance().L.d("[MQTT_CONTROLLER] dynamic config received $data")
                        CoreInternal.sharedInstance().moduleDynamicConfig.updateDynamicConfigValues(
                            data,
                            false
                        );
                    } catch (_: Exception) {
                        CoreInternal.sharedInstance().L.d("[MQTT_CONTROLLER] dynamic config failed to parse ... ")
                    }
                }
            })
        }
    }

    fun initIAM() {
        topic {
            topicAndConnectionStateChanged(true)
        }
        mqttConnectionState {
            topicAndConnectionStateChanged(false)
        }
    }

    private val compositeDisposable = CompositeDisposable()
    private fun topicAndConnectionStateChanged(isFromTopic: Boolean) {
        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: topicAndConnectionStateChanged....")
        if (mqttConnectionState.value == CONNECTED) {
//            if(isFromTopic){
//                try{
//                    CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: Calling Disconnect")
//                    mqttClient.disconnect()
//
//                    topic.value?.let {
//                        mqttClient.connect(mqttConfig.value.mqttConnectOptions(generateTopic()?.replace("/","_") ?: "clientId")) //clientId
//                    }
//
//                }catch (e: Exception){
//                    e.printStackTrace()
//                }
//            }

            try {
                topic.value?.let {
                    compositeDisposable.clear()

                    compositeDisposable.add(courierService.subscribe(it).subscribe {
                        CoreInternal.sharedInstance()?.L?.i("[MQTT_CONTROLLER] Connection: received: $it")
                        CoreInternal.sharedInstance().moduleIAM.updateMessages(listOf(it))
                    })
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }


}
