package sdk.main.core

import android.content.Context
import android.content.pm.PackageManager
import android.os.Bundle
import android.util.Base64
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.gojek.courier.Message
import com.gojek.courier.QoS
import com.gojek.mqtt.auth.Authenticator
import com.gojek.mqtt.client.MqttClient
import com.gojek.mqtt.client.config.PersistenceOptions
import com.gojek.mqtt.client.config.v3.MqttAndroidLogger
import com.gojek.mqtt.client.config.v3.MqttV3Configuration
import com.gojek.mqtt.client.factory.MqttClientFactory
import com.gojek.mqtt.client.listener.MessageListener
import com.gojek.mqtt.client.model.MqttMessage
import com.gojek.mqtt.event.EventHandler
import com.gojek.mqtt.event.MqttEvent
import com.gojek.mqtt.model.KeepAlive
import com.gojek.mqtt.model.MqttConnectOptions
import com.gojek.mqtt.model.ServerUri
import com.gojek.mqtt.pingsender.MqttPingSender
import com.gojek.workmanager.pingsender.WorkManagerPingSenderConfig
import com.gojek.workmanager.pingsender.WorkPingSenderFactory
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import java.lang.reflect.Type
import java.util.Calendar
import java.util.Date
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class PushAmpWorker(context: Context?, workerParams: WorkerParameters?) :
    Worker(context!!, workerParams!!) {

    private lateinit var mqttClient: MqttClient;
    private var startTimeMark = TimeSource.Monotonic.markNow();
    private val pingSender: MqttPingSender = WorkPingSenderFactory.createMqttPingSender(
        this.applicationContext,
        WorkManagerPingSenderConfig(100)
    );
    private lateinit var workerThread: Thread;
    val ttl: Duration = 180.seconds
    var threadIsRunning = false;

    private val eventHandler = object : EventHandler {
        override fun onEvent(mqttEvent: MqttEvent) {
            val lifetime = startTimeMark + ttl;

//            CoreInternal.sharedInstance()?.L?.i("[PUSH_AMP] WORKER Connection: Received event: $mqttEvent");
//            CoreInternal.sharedInstance()?.L?.i("[PUSH_AMP] WORKER Thread id : ${workerThread.id} should i going to sleep? ${lifetime.hasPassedNow()}. am i stop working? $threadIsRunning");

//            if(lifetime.hasPassedNow() && threadIsRunning){
//                CoreInternal.sharedInstance()?.L?.i("[PUSH_AMP] WORKER Connection: i'm going to sleep :) ${workerThread.id}")
//                try {
//                    CoreInternal.sharedInstance()?.L?.i("[PUSH_AMP] WORKER thread should stopped: ${workerThread.id} - ${workerThread.isAlive} ")
//                    pingSender.stop();
//                    mqttClient.disconnect();
//                    threadIsRunning = false;
//                    workerThread.stop();
//                }catch (_: Exception){}
//                return
//            }
        }
    }

    private fun isInDND(): Boolean {
        return try {
            val dndFrom: Int =
                CoreInternal.sharedInstance().moduleDynamicConfig.getValue("pushAmpDDFrom") as Int
            val dndTo: Int =
                CoreInternal.sharedInstance().moduleDynamicConfig.getValue("pushAmpDDTo") as Int

            val calendar: Calendar = Calendar.getInstance()
            val hourOfDay: Int = calendar.get(Calendar.HOUR_OF_DAY)
            CoreInternal.sharedInstance()?.L?.d("[PUSH_AMP] DnD from $dndFrom, DnDTo  $dndTo, hourOfDay $hourOfDay, DND APPLIED? ${hourOfDay !in (dndFrom + 1) until dndTo}")

            hourOfDay !in (dndFrom + 1) until dndTo
        } catch (_: Exception) {
            true
        }
    }

    internal fun displayMessage(
        data: Map<String, String>,
        ttl: Any?,
        messageDisplayedInThisSession: Boolean,
        app: Context,
    ): Boolean {
        if (messageDisplayedInThisSession || isInDND()) {
            val expDate = Calendar.getInstance()
            expDate.time = Date()
            try {
                if (ttl == null) {
                    expDate.add(Calendar.DATE, 28)
                } else {
                    expDate.add(Calendar.SECOND, (ttl as Double).toInt())
                }
            } catch (ignored: Exception) {
                expDate.add(Calendar.DATE, 28)
            }

            SharedPref.addCachedPushMessage(data["id"]?.let {
                PushAmpCacheMessage(
                    it,
                    false,
                    expDate.time,
                    data
                )
            }, app, false);
            return false
        } else {
            var icon: Int = app.applicationInfo.icon
            try {
                val useApp = app.packageManager.getApplicationInfo(
                    app.packageName,
                    PackageManager.GET_META_DATA
                )
                val bundle: Bundle = useApp.metaData
                icon = bundle.getInt(
                    PushHandler.DEFAULT_PUSH_ICON_META,
                    app.applicationInfo.icon
                )
            } catch (_: Exception) {
            }

            var channelId = SharedPref.getLatestPushChannel(app);
            if (channelId == null) {
                channelId = PushHandler.PUSH_AMP_CHANNEL_ID
            }

            return CoreProxy.displayMessage(app, data, channelId, icon, null)
        }
    }

    override fun doWork(): Result {
        val app = this.applicationContext
        var messageDisplayedInThisSession = false;
        CoreInternal.sharedInstance().L.d("[PUSH_AMP] WORKER is starting")
        workerThread = Thread {
            try {
                startTimeMark = TimeSource.Monotonic.markNow();
                CoreInternal.sharedInstance().L.d("[PUSH_AMP] WORKER check with backend")

                val cachedMessage = SharedPref.getACachedPushMessage(app);
                if (cachedMessage != null) {
                    messageDisplayedInThisSession = cachedMessage.data?.let {
                        displayMessage(
                            it, null, messageDisplayedInThisSession, app
                        )
                    } == false
                }

                val userId = CoreInternal.sharedInstance()?.userId
                val deviceId = CoreInternal.sharedInstance()?.deviceID
                val mqttConf = MqttConfig();

                val id = if (userId != null)
                    "U$userId"
                else if (deviceId != null)
                    "A$deviceId"
                else
                    null

                val appId = CoreInternal.sharedInstance()?.config?.appKey
                val pidBytes = Base64.decode(appId, Base64.DEFAULT);
                var productId = 0L
                for (b in pidBytes) {
                    productId = (productId shl 8) + (b.toInt() and 0xFF)
                }

                if (appId != null && deviceId != null) {
                    val mqttDeviceId = "ANDROID_${productId}_$deviceId"
                    val mqttTopic = "IN_BOX/$productId/$id"

                    val mqttConfig = MqttV3Configuration(
                        logger = MqttAndroidLogger,
                        eventHandler = eventHandler,
                        authenticator = object : Authenticator {
                            override fun authenticate(
                                connectOptions: MqttConnectOptions,
                                forceRefresh: Boolean,
                            ): MqttConnectOptions {
                                return connectOptions.newBuilder()
                                    .password(mqttConf.password)
                                    .build()
                            }
                        },
                        mqttInterceptorList = listOf(),
                        persistenceOptions = PersistenceOptions.PahoPersistenceOptions(100, false),
                        pingSender = pingSender
                    )
                    mqttClient = MqttClientFactory.create(app, mqttConfig);

                    val connectOptions = MqttConnectOptions.Builder()
                        .serverUris(
                            listOf(
                                ServerUri(
                                    mqttConf.host ?: "",
                                    mqttConf.port,
                                    if (mqttConf.port == 8883) "ssl" else "tcp"
                                )
                            )
                        )
                        .clientId(mqttDeviceId)
                        .userName(mqttConf.username)
                        .password(mqttConf.password)
                        .cleanSession(false)
                        .keepAlive(KeepAlive(timeSeconds = 30))
                        .build()

                    mqttClient.connect(connectOptions);

                    CoreInternal.sharedInstance()?.L?.i("[PUSH_AMP] Worker subscribe  TOPIC: $mqttTopic with deviceId $mqttDeviceId")
                    mqttClient.subscribe(mqttTopic to QoS.TWO);

                    mqttClient.addMessageListener(mqttTopic, object : MessageListener {
                        override fun onMessageReceived(mqttMessage: MqttMessage) {
                            try {
                                val payload = String((mqttMessage.message as Message.Bytes).value)
                                val type: Type = object : TypeToken<Map<String?, Any?>?>() {}.type
                                val data = Gson().fromJson<Map<String, Any>>(payload, type)

                                val messageData = data["data"] as MutableMap<String, String>
                                val ttl = data["time_to_live"]

                                CoreInternal.sharedInstance().L.d("[PUSH_AMP] received an inbox message with payload: $payload")

                                if (messageData["id"] == null) {
                                    return
                                }

                                messageDisplayedInThisSession = displayMessage(
                                    messageData,
                                    ttl,
                                    messageDisplayedInThisSession,
                                    app
                                )
                                CoreInternal.sharedInstance().L.sendLog("[PUSH_AMP] displayed a message (${messageDisplayedInThisSession}) with id: ${messageData["id"]}")
                            } catch (ex: Exception) {
                                CoreInternal.sharedInstance().L.e("[PUSH_AMP] parsing message failed: ${ex.message}")
                            }
                        }
                    })
                }
            } catch (ex: Exception) {
                CoreInternal.sharedInstance().L.e("[PUSH_AMP] the whole things goes wrong" + ex.message)
            }
        }
        workerThread.start()
        threadIsRunning = true;
        CoreInternal.sharedInstance().L.i("[PUSH_AMP] WORKER Thread started ${workerThread.id}")
        return Result.success()
    }
}
