package sdk.main.core;

import static android.content.Context.UI_MODE_SERVICE;

import android.app.UiModeManager;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.os.Build;
import android.util.Log;

import androidx.core.app.NotificationCompat;

import com.google.android.gms.tasks.Tasks;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import sdk.main.core.pushmessaging.PushMessage;

class Utils {

    private static final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZZZ", Locale.ENGLISH);

    private static final ExecutorService bg = Executors.newSingleThreadExecutor();
    private static final String TAG = "Utils";
    private static final int IMAGE_DOWNLOAD_TIMEOUT_SECONDS = 5;
    public static ExecutorService networkIoExecutor = Executors.newFixedThreadPool(8);

    public static Future<?> runInBackground(Runnable runnable) {
        return bg.submit(runnable);
    }

    public static <T> Future<T> runInBackground(Callable<T> runnable) {
        return bg.submit(runnable);
    }

    /**
     * Joins objects with a separator
     *
     * @param objects   objects to join
     * @param separator separator to use
     * @return resulting string
     */
    static <T> String join(Collection<T> objects, String separator) {
        StringBuilder sb = new StringBuilder();
        Iterator<T> iter = objects.iterator();
        while (iter.hasNext()) {
            sb.append(iter.next());
            if (iter.hasNext()) {
                sb.append(separator);
            }
        }
        return sb.toString();
    }

    /**
     * StringUtils.isEmpty replacement.
     *
     * @param str string to check
     * @return true if null or empty string, false otherwise
     */
    public static boolean isEmpty(String str) {
        return str == null || "".equals(str);
    }

    /**
     * StringUtils.isNotEmpty replacement.
     *
     * @param str string to check
     * @return false if null or empty string, true otherwise
     */
    public static boolean isNotEmpty(String str) {
        return !isEmpty(str);
    }

    /**
     * Returns true if the version you are checking is at or below the build version
     *
     * @param version
     * @return
     */
    public static boolean API(int version) {
        return Build.VERSION.SDK_INT >= version;
    }

    /**
     * Read stream into a byte array
     *
     * @param stream input to read
     * @return stream contents or {@code null} in case of error
     */
    public static byte[] readStream(InputStream stream) {
        if (stream == null) {
            return null;
        }

        ByteArrayOutputStream bytes = new ByteArrayOutputStream();
        try {
            byte[] buffer = new byte[1024];
            int len = 0;
            while ((len = stream.read(buffer)) != -1) {
                bytes.write(buffer, 0, len);
            }
            return bytes.toByteArray();
        } catch (IOException e) {
            CoreInternal.sharedInstance().L.e("Couldn't read stream: " + e);
            return null;
        } finally {
            try {
                bytes.close();
                stream.close();
            } catch (Throwable ignored) {
            }
        }
    }

    static String inputStreamToString(InputStream stream) {
        BufferedReader br = new BufferedReader(new InputStreamReader(stream));

        StringBuilder sbRes = new StringBuilder();

        while (true) {
            String streamLine;
            try {
                streamLine = br.readLine();
            } catch (IOException e) {
                CoreInternal.sharedInstance().L.e("", e);
                break;
            }

            if (streamLine == null) {
                break;
            }

            if (sbRes.length() > 0) {
                //if it's not empty then there has been a previous line
                sbRes.append("\n");
            }

            sbRes.append(streamLine);
        }

        return sbRes.toString();
    }

    static Map<String, Object> removeKeysFromMap(Map<String, Object> data, String[] keys) {
        if (data == null || keys == null) {
            return data;
        }

        for (String key : keys) {
            data.remove(key);
        }

        return data;
    }

    /**
     * Removes unsupported data types
     *
     * @param data
     * @return returns true if any entry had been removed
     */
    static boolean removeUnsupportedDataTypes(Map<String, Object> data) {
        if (data == null) {
            return false;
        }

        boolean removed = false;

        for (Iterator<Map.Entry<String, Object>> it = data.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<String, Object> entry = it.next();
            String key = entry.getKey();
            Object value = entry.getValue();

            if (key == null || key.isEmpty() || key.length() > 50 || !(isValidObject(value))) {
                //found unsupported data type or null key or value, removing
                it.remove();
                removed = true;
            }
        }

        if (removed) {
            CoreInternal.sharedInstance().L.w("Unsupported data types were removed from provided segmentation");
        }

        return removed;
    }

    /**
     * Used for quickly sorting segments into their respective data type
     *
     * @param allSegm
     * @param segmStr
     * @param segmInt
     * @param segmDouble
     * @param segmLong
     * @param segmBoolean
     */
    protected static synchronized void fillInSegmentation(
            Map<String, Object> allSegm,
            Map<String, String> segmStr,
            Map<String, Integer> segmInt,
            Map<String, Double> segmDouble,
            Map<String, Long> segmLong,
            Map<String, Boolean> segmBoolean,
            Map<String, JSONObject> segmObject,
            Map<String, JSONArray> segmArray,
            Map<String, Object> reminder
    ) {
        for (Map.Entry<String, Object> pair : allSegm.entrySet()) {
            String key = pair.getKey();
            Object value = pair.getValue();

            if (value instanceof Integer) {
                segmInt.put(key, (Integer) value);
            } else if (value instanceof Double) {
                segmDouble.put(key, (Double) value);
            } else if (value instanceof Long) {
                segmLong.put(key, (Long) value);
            } else if (value instanceof String) {
                segmStr.put(key, (String) value);
            } else if (value instanceof Boolean) {
                segmBoolean.put(key, (Boolean) value);
            } else if (value instanceof JSONObject) {
                segmObject.put(key, (JSONObject) value);
            } else if (value instanceof JSONArray) {
                segmArray.put(key, (JSONArray) value);
            } else {
                if (reminder != null) {
                    reminder.put(key, value);
                }
            }
        }
    }

    /**
     * Used for detecting if current device is a tablet of phone
     */
    static boolean isDeviceTablet(Context context) {
        if (context == null) {
            return false;
        }

        return (context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) >= Configuration.SCREENLAYOUT_SIZE_LARGE;
    }

    /**
     * Used for detecting if device is a tv
     *
     * @return
     */
    @SuppressWarnings("RedundantIfStatement")
    static boolean isDeviceTv(Context context) {
        if (context == null) {
            return false;
        }

        UiModeManager uiModeManager = (UiModeManager) context.getSystemService(UI_MODE_SERVICE);

        if (uiModeManager == null) {
            return false;
        }

        if (uiModeManager.getCurrentModeType() == Configuration.UI_MODE_TYPE_TELEVISION) {
            return true;
        } else {
            return false;
        }
    }

    public static String getSystemLabel(String label) {
        return ModuleEvents.SYSTEM_EVENTS_PREFIX + label;
    }

    public static String formatDate(Date date) {
        synchronized (simpleDateFormat) {
            return simpleDateFormat.format(date);
        }
    }

    public static String formatDate(Long date) {
        return formatDate(new Date(date));
    }

    public static Date parseDate(String date) {
        synchronized (simpleDateFormat) {
            try {
                return simpleDateFormat.parse(date);
            } catch (ParseException e) {
                return null;
            }
        }
    }

    private static boolean isValidObject(Object object) {
        return object instanceof Boolean ||
                object instanceof Integer ||
                object instanceof Long ||
                object instanceof Double ||
                object instanceof String ||
                object instanceof JSONObject ||
                object instanceof JSONArray;
    }

    public static Map<String, Object> mapifyJsonObject(JSONObject jsonObject) {

        Map<String, Object> result = new HashMap<>();
        Iterator<String> iterator = jsonObject.keys();
        while (iterator.hasNext()) {
            String key = iterator.next();
            try {
                Object value = jsonObject.get(key);
                if (isValidObject(value)) {
                    result.put(key, value);
                }
            } catch (JSONException ignored) {
            }
        }
        return result;
    }

    public static JSONObject objectifyMap(Map<String, Object> map) {
        Iterator<Map.Entry<String, Object>> iterator = map.entrySet().iterator();
        while (iterator.hasNext()) {
            if (!isValidObject(iterator.next().getValue())) {
                iterator.remove();
            }
        }
        return new JSONObject(map);
    }

    public static ImageDownload startImageDownloadInBackground(String imageUrl) {
        ImageDownload imageDownload = ImageDownload.create(imageUrl);
        if (imageDownload != null) {
            imageDownload.start(networkIoExecutor);
        }
        return imageDownload;
    }

    public static void waitForAndApplyImageDownload(NotificationCompat.Builder n, PushMessage msg, ImageDownload imageDownload) {
        if (imageDownload == null) {
            return;
        }
        /*
         * This blocks to wait for the image to finish downloading as this background thread is being
         * used to keep the app (via service or receiver) alive. It can't all be done on one thread
         * as the URLConnection API used to download the image is blocking, so another thread is needed
         * to enforce the timeout.
         */
        try {
            Bitmap bitmap = Tasks.await(imageDownload.getTask(), IMAGE_DOWNLOAD_TIMEOUT_SECONDS, TimeUnit.SECONDS);

            /*
             * Set large icon for non-expanded mode (shows up on the right), and the big picture for
             * expanded mode. Clear the large icon in expanded mode as having the small image on the right
             * and the large expanded image doesn't look great. This is what the Android screenshot
             * notification does as well.
             */

            n.setStyle(new NotificationCompat.BigPictureStyle()
                    .bigPicture(bitmap)
                    .setBigContentTitle(msg.getTitle())
                    .setSummaryText(msg.getMessage()));

        } catch (ExecutionException e) {
            // For all exceptions, fall through to show the notification without the image

            Log.w(TAG, "Failed to download image: " + e.getCause());
        } catch (InterruptedException e) {
            Log.w(TAG, "Interrupted while downloading image, showing notification without it");
            imageDownload.close();
            Thread.currentThread().interrupt(); // Restore the interrupted status
        } catch (TimeoutException e) {
            Log.w(TAG, "Failed to download image in time, showing notification without it");
            imageDownload.close();
            /*
             * Instead of cancelling the task, could let the download continue, and update the
             * notification if it was still showing. For this we would need to cancel the download if the
             * user opens or dismisses the notification, and make sure the notification doesn't buzz again
             * when it is updated.
             */
        }
    }

    public static boolean isSilentPushFromInTrack(Map<String, String> pushData) {
        String source = pushData.get("source");
        return pushData.keySet().size() < 3 && source != null && source.equals("inTrack");
    }

    public static boolean isMessageDebugActive(Map<String, String> pushData) {
        if (isSilentPushFromInTrack(pushData)) {
            String debugValue = pushData.get("debug");
            // checking if "debug" exists, if so check the value to be "true" or true (as boolean) or "1" or 1 (as int)
            return debugValue != null && (debugValue.equals("true") || debugValue.equals("1"));
        } else {
            return false;
        }
    }


    public static String adaptCacheMessage4Log(Map<String, PushAmpCacheMessage> cacheMessages) {
        if (cacheMessages == null) {
            return "none";
        } else {
            // append IDs
            StringBuilder joinedIDs = new StringBuilder();
            for (String id : cacheMessages.keySet()) {
                joinedIDs.append(id).append(",");
            }

            // remove latest comma
            joinedIDs.setLength(joinedIDs.length() - 1);

            return joinedIDs.toString();
        }
    }

    public static Map<String, String> extractNestedMap(String jsonString) {
        if (jsonString == null || jsonString.isEmpty()) {
            return null; // Return null for empty or null input
        }
        try {
            JSONObject jsonObject = new JSONObject(jsonString);
            Map<String, String> map = new HashMap<>();

            for (Iterator<String> it = jsonObject.keys(); it.hasNext(); ) {
                String key = it.next();
                Object value = jsonObject.get(key); // Safely get the value
                if (value instanceof String) {
                    map.put(key, (String) value);
                } else {
                    map.put(key, String.valueOf(value)); // Convert non-string values to string
                }
            }
            return map;
        } catch (Exception e) {
            System.err.println("Error parsing JSON string: " + e.getMessage());
            return null;
        }
    }
}
