package sdk.main.core.ext

import android.os.Build
import androidx.annotation.RequiresApi
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.combine

internal class RefreshStateFlow private constructor(innerFlow: MutableStateFlow<Int>) :
    MutableStateFlow<Int> by innerFlow {
    constructor() : this(MutableStateFlow(0))

    fun refresh() {
        value += 1
    }
}

internal fun <V, T : Flow<V>> T.refreshWith(r: RefreshStateFlow) = combine(this, r) { a, _ -> a }
internal fun <T> Flow<T>.refreshWith(r: Flow<*>) = combine(this, r) { a, _ -> a }


internal class MapStateFlow<K, V>(
    private val delegatedMap: MutableMap<K, V>,
    private val stateFlow: MutableStateFlow<MutableMap<K, V>> = MutableStateFlow(delegatedMap),
    private val onDataChanged: (map: MapStateFlow<K, V>) -> Unit = {},
) :
    MutableStateFlow<MutableMap<K, V>> by stateFlow, MutableMap<K, V> by delegatedMap {

    val mapData: Map<K, V> get() = delegatedMap

    fun changed() {
        onDataChanged(this)
    }

    override fun put(key: K, value: V): V? {
        val ans = delegatedMap.put(key, value)
        changed()
        return ans
    }

    override fun remove(key: K): V? {
        val ans = delegatedMap.remove(key)
        changed()
        return ans
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun remove(key: K, value: V): Boolean {
        val ans = delegatedMap.remove(key, value)
        changed()
        return ans
    }

    override fun putAll(from: Map<out K, V>) {
        delegatedMap.putAll(from)
        changed()
    }

    override fun clear() {
        delegatedMap.clear()
        changed()
    }
}