package sdk.main.core.inappmessaging.display

import android.app.Activity
import android.view.View
import androidx.annotation.VisibleForTesting
import sdk.main.core.InAppMessageListener
import sdk.main.core.InAppMessageListener.ErrorMessageReason
import sdk.main.core.inappmessaging.InAppMessagingDisplayCallbacks
import sdk.main.core.inappmessaging.display.internal.Logging
import sdk.main.core.inappmessaging.display.mraid.ViewportHtmlCallbacks
import sdk.main.core.inappmessaging.model.MessageType
import sdk.main.core.inappmessaging.model.message.IAMMessage

internal class InAppMessagingDisplay {
    @VisibleForTesting
    var currentlyBoundActivityName: String? = null
    private lateinit var inAppMessage: IAMMessage
    private var inAppListener: InAppMessageListener? = null
    private var callbacks: InAppMessagingDisplayCallbacks? = null

    fun displayMessage(
        activity: Activity,
        inAppMessage: IAMMessage,
        callbacks: InAppMessagingDisplayCallbacks?
    ) {
        this.inAppMessage = inAppMessage
        this.callbacks = callbacks
        showActiveMessage(activity)
    }

    /**
     * Sets InApp listener to receive in app message callbacks
     */
    fun setListener(listener: InAppMessageListener?) {
        this.inAppListener = listener
    }

    /**
     * Clears InApp listener
     */
    fun clearListener() {
        this.inAppListener = null
    }

    /**
     * Bind InApp listener on Activity resume.
     */
    fun onActivityResumed(activity: Activity) {
        bindMessageToActivity(activity)
    }

    /**
     * Clear InApp listener on activity paused
     */
    fun onActivityPaused(activity: Activity) {
        unbindMessageFromActivity(activity)
    }

    private fun bindMessageToActivity(activity: Activity) {
        if (currentlyBoundActivityName == null || currentlyBoundActivityName != activity.localClassName) {
            Logging.logi("Binding to activity: " + activity.localClassName)
            currentlyBoundActivityName = activity.localClassName
        }
    }

    private fun unbindMessageFromActivity(activity: Activity) {
        if (currentlyBoundActivityName != null && currentlyBoundActivityName == activity.localClassName) {
            Logging.logi("Unbinding from activity: " + activity.localClassName)
            currentlyBoundActivityName = null
        }
    }

    private fun showActiveMessage(activity: Activity) {
        if (inAppMessage.layout != null && inAppMessage.layout == MessageType.UNSUPPORTED) {
            Logging.loge("The message being triggered is not supported by this version of the sdk.")
            inAppListener?.onError(
                ErrorMessageReason.MESSAGE_NOT_SUPPORTED_ON_CURRENT_SDK_VERSION_ERROR,
                String.format(
                    "Message Type: %s, Message ID: %s",
                    inAppMessage.layout,
                    inAppMessage.id
                )
            )
            return
        }

        activity.findViewById<View>(android.R.id.content)
            .post { showHtmlInAppMessage(activity) }
    }

    private fun showHtmlInAppMessage(activity: Activity) {
        viewportHtml(
            activity,
            inAppMessage,
            object : ViewportHtmlCallbacks {
                override fun onWebViewLoaded() {
                    callbacks?.impressionDetected()
                    inAppListener?.onMessageTrigger(inAppMessage)
                }

                override fun onDismiss() {
                    callbacks?.messageDismissed(InAppMessagingDisplayCallbacks.InAppMessagingDismissType.CLICK)
                    inAppListener?.onMessageDismiss(inAppMessage)
                    callbacks = null
                }

                override fun onClicked(url: String?) {
                    callbacks?.messageClicked(url)
                    inAppListener?.onMessageClick(inAppMessage)
                }

                override fun onViewFailed(reason: MraidErrors, message: String?) {
                    when (reason) {
                        MraidErrors.HtmlNotLoaded -> {
                            callbacks?.displayErrorEncountered(ErrorMessageReason.MRAID_JS_NOT_FOUND_IN_HTML)
                            inAppListener?.onError(
                                ErrorMessageReason.MRAID_JS_NOT_FOUND_IN_HTML,
                                message
                            )
                        }

                        MraidErrors.MraidJSNotFoundInHtml -> {
                            callbacks?.displayErrorEncountered(ErrorMessageReason.HTML_LOAD_FAILED)
                            inAppListener?.onError(ErrorMessageReason.HTML_LOAD_FAILED, message)
                        }

                        MraidErrors.InvalidData -> {
                            callbacks?.displayErrorEncountered(ErrorMessageReason.INVALID_DATA)
                            inAppListener?.onError(ErrorMessageReason.INVALID_DATA, message)
                        }
                    }
                }
            }
        )
    }
}
