package sdk.main.core.inappmessaging.display

import android.app.Activity
import android.os.Build
import android.text.Html
import android.util.Log
import android.view.ViewGroup
import android.webkit.ConsoleMessage
import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.tween
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.ComposeView
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.graphics.toColorInt
import androidx.core.view.WindowCompat
import kotlinx.coroutines.launch
import sdk.main.core.inappmessaging.display.mraid.MraidCommands
import sdk.main.core.inappmessaging.display.mraid.MraidListener
import sdk.main.core.inappmessaging.display.mraid.ViewportHtmlCallbacks
import sdk.main.core.inappmessaging.display.mraid.ViewportWebView
import sdk.main.core.inappmessaging.display.mraid.utils.ViewUtil.getScreenSizeAsPixels
import sdk.main.core.inappmessaging.model.message.IAMMessage

const val MRAID_LOG_TAG = "MraidListenerTag"
const val ANIMATION_DURATION = 500 // ms

fun viewportHtml(
    activity: Activity,
    message: IAMMessage,
    viewportHtmlCallbacks: ViewportHtmlCallbacks,
) {
    val finalViewportMetric = message.config?.getFinalViewportMetric(
        getScreenSizeAsPixels(activity)[0],
        activity.resources.displayMetrics,
    )
    if (
        !message.html.isNullOrBlank() &&
        (finalViewportMetric?.height?.value ?: 0f) > 0f &&
        (finalViewportMetric?.width?.value ?: 0f) > 0f
    ) {
        message.html = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            Html.fromHtml(message.html, Html.FROM_HTML_MODE_LEGACY).toString()
        } else {
            Html.fromHtml(message.html).toString()
        }

        if (!message.html!!.contains(MraidCommands.MRAID_MINIFIED_FILE_NAME, true)) {
            viewportHtmlCallbacks.onViewFailed(
                MraidErrors.MraidJSNotFoundInHtml,
                "Does not contain mraid!"
            )
        } else {
            return setupComposeView(activity, message, viewportHtmlCallbacks)
        }
    } else {
        viewportHtmlCallbacks.onViewFailed(MraidErrors.InvalidData, null)
    }
}

private fun setupComposeView(
    activity: Activity,
    message: IAMMessage,
    viewportHtmlCallbacks: ViewportHtmlCallbacks,
) {
    WindowCompat.setDecorFitsSystemWindows(activity.window, false)
    val rootView = activity.window.decorView

    val composeView = ComposeView(activity)

    val configState = mutableStateOf(message.config!!)
    fun updateConfiguration(newConfig: Configuration) {
        configState.value = newConfig
    }

    val viewportMraidListener = object : MraidListener {
        override fun onInitFinished() {
            Log.d(MRAID_LOG_TAG, "onInitFinished called")
        }

        override fun onClose() {
            Log.d(MRAID_LOG_TAG, "onClose called")
        }

        override fun onStartLoadingHtml() {
            Log.d(MRAID_LOG_TAG, "onStartLoadingHtml called")
        }

        override fun onHtmlLoaded() {
            Log.d(MRAID_LOG_TAG, "onHtmlLoaded called")
            viewportHtmlCallbacks.onWebViewLoaded()
        }

        override fun onHtmlError(errorCode: Int?, message: String?) {
            Log.d(MRAID_LOG_TAG, "onHtmlError called with errorCode: $errorCode, message: $message")
            viewportHtmlCallbacks.onViewFailed(MraidErrors.HtmlNotLoaded, message)
        }

        override fun onJSConsole(consoleMessage: ConsoleMessage?): Boolean {
            consoleMessage?.let {
                Log.d(
                    MRAID_LOG_TAG,
                    "onJSConsole called with message: ${it.message()}, lineNumber: ${it.lineNumber()}, sourceId: ${it.sourceId()}"
                )
                return true
            }
            return false
        }

        override fun onClicked(url: String?) {
            Log.d(MRAID_LOG_TAG, "onClicked called with url: $url")
            viewportHtmlCallbacks.onClicked(url)
        }

        override fun onUnload(shouldSendCloseEvent: Boolean) {
            Log.d(MRAID_LOG_TAG, "onUnload called")
            if (shouldSendCloseEvent) {
                viewportHtmlCallbacks.onDismiss()
            }
            composeView
                .animate()
                .alpha(0f) // Fade out to transparent
                .setDuration(ANIMATION_DURATION.toLong())
                .setInterpolator(android.view.animation.AccelerateDecelerateInterpolator())
                .withEndAction {
                    (rootView as? ViewGroup)?.removeView(composeView)
                }.start()
        }

        override fun onConfigurationUpdated(newConfig: Configuration) {
            Log.d(MRAID_LOG_TAG, "onConfigurationUpdated called with newConfig: $newConfig")
            updateConfiguration(newConfig)
        }

        override fun onInjectJavaScript(uri: String) {
            Log.d(MRAID_LOG_TAG, "onInjectJavaScript called with uri: $uri")
        }
    }
    val webView = ViewportWebView(activity, message).apply {
        mraidListener = viewportMraidListener
    }

    composeView.setContent {
        val config by remember { configState }
        val alpha = remember { Animatable(0f) }
        val currentDisplayMetrics = LocalContext.current.resources.displayMetrics
        var finalViewportMetrics by remember {
            mutableStateOf(
                config.getFinalViewportMetric(
                    getScreenSizeAsPixels(activity)[0],
                    currentDisplayMetrics,
                )!!
            )
        }

        LaunchedEffect(LocalConfiguration.current, config) {
            finalViewportMetrics = config.getFinalViewportMetric(
                getScreenSizeAsPixels(activity)[0],
                currentDisplayMetrics,
            )!!
        }

        LaunchedEffect(Unit) {
            launch {
                alpha.animateTo(
                    1f,
                    animationSpec = tween(durationMillis = ANIMATION_DURATION)
                )
            }
        }

        Box(
            modifier = Modifier
                .fillMaxSize()
                .safeInsetsPadding(activity, currentDisplayMetrics)
                .let {
                    if (config.shouldShowOverlay()) {
                        it
                            .background(
                                Color(
                                    config.overlay!!.getColor().toColorInt(),
                                ).copy(alpha = config.overlay!!.getOpacity()),
                            )
                            .clickable(
                                enabled = config.overlay!!.isDismissible(),
                                onClick = { webView.unload() },
                            )
                    } else {
                        it
                    }
                }
                .alpha(alpha.value)
        ) {
            AndroidView(
                factory = { webView },
                modifier = Modifier
                    .size(
                        finalViewportMetrics.width?.toDp(currentDisplayMetrics) ?: 0.dp,
                        finalViewportMetrics.height?.toDp(currentDisplayMetrics, false) ?: 0.dp,
                    )
                    .align((config.position ?: ViewportPosition.CENTER).alignment)
                    .padding(
                        top = finalViewportMetrics.top?.toDp(currentDisplayMetrics) ?: 0.dp,
                        bottom = finalViewportMetrics.bottom?.toDp(currentDisplayMetrics) ?: 0.dp,
                        start = finalViewportMetrics.start?.toDp(currentDisplayMetrics) ?: 0.dp,
                        end = finalViewportMetrics.end?.toDp(currentDisplayMetrics) ?: 0.dp,
                    )
                    .alpha(alpha.value)
            )
        }
    }

    // Add ComposeView to the root view if it's a ViewGroup
    if (rootView is ViewGroup) {
        rootView.addView(composeView)
    }
}
