package sdk.main.core.inappmessaging.display.mraid

import android.Manifest
import android.app.AlertDialog
import android.app.DownloadManager
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Rect
import android.media.AudioManager
import android.media.MediaRouter
import android.media.MediaRouter.RouteGroup
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.util.Base64
import android.view.WindowInsets
import android.view.WindowMetrics
import ir.intrack.android.sdk.R
import org.json.JSONException
import org.json.JSONObject
import sdk.main.core.ClipboardManager
import sdk.main.core.CoreProxy
import sdk.main.core.inappmessaging.display.*
import sdk.main.core.inappmessaging.display.Configuration.Companion.fromString
import sdk.main.core.inappmessaging.display.internal.Logging
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.audioVolumeChangeEvent
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.exposureChangeEvent
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.getAppKey
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.getAuthKey
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.getDeviceId
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.getUserId
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.readyEvent
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setMaxSize
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setPlacementType
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setScreenSize
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setSupports
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.sizeChangeEvent
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.stateChangeEvent
import sdk.main.core.inappmessaging.display.mraid.Orientation.Companion.fromAndroidActivity
import sdk.main.core.inappmessaging.display.mraid.utils.Hex.hexStringToByteArray
import sdk.main.core.inappmessaging.display.mraid.utils.ViewUtil.getScreenSizeAsPixels
import sdk.main.core.inappmessaging.display.mraid.utils.ViewUtil.getTopContext
import sdk.main.core.inappmessaging.display.mraid.utils.W3CEvent.Companion.createFromJSON
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.io.UnsupportedEncodingException
import java.net.URLDecoder
import java.util.Locale

class MRAIDImplementation(private val owner: ViewportWebView) {
    private var expanded: Boolean = false
    private var resized: Boolean = false
    private var supportsPictureAPI = false
    private val supportsCalendar = false
    private var mediaRouter: MediaRouter? = null
    private var callback: MediaRouter.Callback? = null
    private var readyFired = false
    private var exposureVal = ""
    private var lastRotation = Orientation.Unspecified

    fun webViewFinishedLoading(view: ViewportWebView, state: MraidState) {
        if (!readyFired) {
            view.injectJavaScript(setPlacementType(DEFAULT_PLACEMENT_TYPE))
            setSupportsValues(view)
            setScreenSize()
            setMaxSize()
            setCurrentAppOrientation()

            owner.checkPosition()
            view.injectJavaScript(stateChangeEvent(state))
            view.injectJavaScript(readyEvent())

            readyFired = true
        }
    }

    private fun setCurrentAppOrientation(orientation: Orientation = fromAndroidActivity(owner.activity)) {
        owner.injectJavaScript(MraidCommands.setCurrentAppOrientation(orientation))
    }

    private fun setSupports(view: ViewportWebView, feature: String, value: Boolean) {
        view.injectJavaScript(setSupports(feature, value))
    }

    private fun setSupportsValues(view: ViewportWebView) {
        val pm = owner.context.packageManager

        if (pm.checkPermission(
                Manifest.permission.WRITE_EXTERNAL_STORAGE,
                owner.context.packageName
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            setSupports(view, MraidCommands.STORE_PICTURE, true)
            supportsPictureAPI = true
        }

        setSupports(view, MraidCommands.INLINE_VIDEO, false)
    }

    fun onExposureChange() {
        if (!readyFired) return
        owner.injectJavaScript(exposureChangeEvent(exposureVal))
    }

    fun onAudioVolumeChange(volumePercentage: String) {
        if (!readyFired) return
        owner.injectJavaScript(audioVolumeChangeEvent(volumePercentage))
    }

    // parameters are view properties in pixels
    fun setCurrentPosition(configuration: Configuration) {
        if (!owner.currentConfiguration.isEqual(configuration)) {
            owner.injectJavaScript(sizeChangeEvent(configuration))
        }
    }

    fun close() {
        if (owner.isConfigurationChanged) {
            owner.revertConfiguration()
        } else {
            unload(null)
        }
        owner.mraidListener?.onClose()
    }

    fun unload(jsonObject: JSONObject?) {
        var shouldSendCloseEvent = true
        if (jsonObject != null) {
            shouldSendCloseEvent = jsonObject.optBoolean("shouldSendCloseEvent", true)
        }
        owner.destroy()
        owner.mraidListener?.onUnload(shouldSendCloseEvent)
    }

    private fun expand() {
        owner.setConfiguration(
            owner.currentConfiguration.copy(
                fallbackMetric = ViewportMetrics(
                    width = Dimension(100f, UnitType.PERCENT),
                    height = Dimension(100f, UnitType.PERCENT),
                    top = null,
                    bottom = null,
                    start = null,
                    end = null,
                ),
                responsiveMetrics = mutableListOf(),
            )
        )
        expanded = true
    }

    fun dispatchMraidCall(url: String) {
        Logging.logd("MRAID-CALL", url)

        val qMarkSplit =
            // Remove the fake protocol and decode
            decode(url.replaceFirst(MraidCommands.MRAID_SCHEME.toRegex(), ""))
            // Separate the function from the parameters
            .split(MraidCommands.REGEX_SPLIT_BACK_SPLASH.toRegex())
            .dropLastWhile { it.isEmpty() }.toTypedArray()
        val func = qMarkSplit[0]
        var jsonObject: JSONObject? = null
        var jsonStr: String? = null
        if (qMarkSplit.size > 1) {
            jsonStr = qMarkSplit[1].replaceFirst("json=".toRegex(), "")
            try {
                jsonObject = JSONObject(jsonStr)
            } catch (e: JSONException) {
                Logging.loge("Exception parsing parameters: " + e.message)
            }
        }

        if (func == MraidCommands.RECORD_INTERNAL_EVENT) {
            recordInternalEvent(jsonObject)
        } else if (func == MraidCommands.AUTH_KEY) {
            authKeyJsCallback()
        } else if (func == MraidCommands.APP_KEY) {
            appKeyJsCallback()
        } else if (func == MraidCommands.DEVICE_ID) {
            deviceIdJsCallback()
        } else if (func == MraidCommands.USER_ID) {
            userIdJsCallback()
        } else if (func == MraidCommands.COPY_TO_CLIPBOARD) {
            copyToClipboard(jsonObject)
        } else if (func == MraidCommands.RECORD_NATIVE_CLICK_EVENT) {
            recordClick(jsonObject)
        } else if (func == MraidCommands.DEEPLINK) {
            deeplink(jsonObject)
        } else if (func == MraidCommands.SET_CONFIGURATION) {
            setConfiguration(jsonStr)
        } else if (func == MraidState.Expanded.label) {
            expand()
        } else if (func == MraidCommands.SET_ORIENTATION_PROPERTIES) {
            setOrientationProperties(jsonObject)
        } else if (func == MraidCommands.CLOSE) {
            close()
        } else if (func == MraidCommands.UNLOAD) {
            unload(jsonObject)
        } else if (func == MraidState.Resized.label) {
            resize(jsonStr)
        } else if (supportsCalendar && func == MraidCommands.CREATE_CALENDAR_EVENT) {
            createCalendarEvent(jsonStr)
        } else if (func == MraidCommands.PLAY_VIDEO) {
            playVideo(jsonObject)
        } else if (supportsPictureAPI && func == MraidCommands.STORE_PICTURE) {
            storePicture(jsonObject)
        } else if (func == MraidCommands.OPEN) {
            open(jsonObject)
        } else if (func == MraidCommands.AUDIO_VOLUME_CHANGE) {
            startAudioVolumeListener()
        } else if (func == MraidCommands.ENABLE) {
            owner.fireMRAIDEnabled()
        } else {
            Logging.logd(owner.context.resources.getString(R.string.unsupported_mraid, func))
        }
    }

    private fun setConfiguration(json: String?) {
        if (json.isNullOrBlank()) return

        val configuration = fromString(
            json
        )
        if (configuration != null) {
            owner.setConfiguration(configuration)
        }
    }

    private fun recordClick(jsonObject: JSONObject?) {
        if (jsonObject == null) return
        val url = jsonObject.optString(MraidCommands.URL)

        owner.mraidListener?.onClicked(url)
    }

    private fun deeplink(jsonObject: JSONObject?) {
        if (jsonObject == null) return
        val deeplink = jsonObject.optString(MraidCommands.DEEPLINK)

        if (deeplink.isNullOrBlank()) return

        val deeplinkIntent = Intent(Intent.ACTION_VIEW)
        deeplinkIntent.setData(Uri.parse(deeplink))
        deeplinkIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        deeplinkIntent.setPackage(owner.context.packageName)

        val packageManager = owner.context.packageManager
        val activities = packageManager.queryIntentActivities(deeplinkIntent, 0)

        // start activity if it is safe
        if (activities.isEmpty()) {
            Logging.loge("No application found to handle deeplink")
            return
        }

        owner.activity.startActivity(deeplinkIntent)
    }

    private fun recordInternalEvent(jsonObject: JSONObject?) {
        if (jsonObject == null) return

        val eventName = jsonObject.optString("key")
        val attributesObject = jsonObject.optJSONObject("attributes")
        val attributes: MutableMap<String, Any> = HashMap()
        if (attributesObject != null) {
            val keys = attributesObject.keys()
            while (keys.hasNext()) {
                val key = keys.next()
                val value = jsonObject.optString(key)
                if (!value.isNullOrBlank()) {
                    attributes[key] = value
                }
            }
        }

        if (!eventName.isNullOrBlank()) {
            if (attributes.isNotEmpty()) {
                CoreProxy.recordEvent(eventName, attributes)
            } else {
                CoreProxy.recordEvent(eventName)
            }
        }
    }

    private fun authKeyJsCallback() {
        owner.injectJavaScript(getAuthKey(CoreProxy.getAuthKey()))
    }

    private fun appKeyJsCallback() {
        owner.injectJavaScript(getAppKey(CoreProxy.getAppKey()))
    }

    private fun deviceIdJsCallback() {
        owner.injectJavaScript(getDeviceId(CoreProxy.getDeviceId()))
    }

    private fun userIdJsCallback() {
        owner.injectJavaScript(getUserId(CoreProxy.getUserId()))
    }

    private fun copyToClipboard(jsonObject: JSONObject?) {
        if (jsonObject == null) return
        val text = jsonObject.optString("text")

        if (!text.isNullOrBlank()) {
            val clipboardManager = ClipboardManager()
            clipboardManager.copyToClipboard(owner.context, decode(text))
        }
    }

    private fun open(jsonObject: JSONObject?) {
        if (jsonObject == null) return
        val uri = jsonObject.optString(MraidCommands.URL)

        if (!uri.isNullOrBlank()) {
            owner.fireAdClicked(uri)
        }
    }

    private fun setOrientationProperties(jsonObject: JSONObject?) {
        if (jsonObject == null) return

        val parsedOrientation = jsonObject.optString(MraidCommands.ORIENTATION)

        val orientation = Orientation.fromString(parsedOrientation)

        owner.setOrientationProperties(orientation)
    }

    private fun storePicture(jsonObject: JSONObject?) {
        if (jsonObject == null) return

        val uri = jsonObject.optString(MraidCommands.URI)

        if (uri.isNullOrBlank()) {
            Logging.logd(owner.context.resources.getString(R.string.store_picture_error))
            return
        }

        val uri_final = Uri.decode(uri)

        val builder = AlertDialog.Builder(
            getTopContext(
                owner
            )
        )
        builder.setTitle("store_picture_title")
        builder.setMessage("store_picture_message")
        builder.setPositiveButton(
            "store_picture_message"
        ) { dialog: DialogInterface?, which: Int ->
            //Check URI scheme
            if (uri_final.startsWith("data:")) {
                //Remove 'data:(//?)' and save
                var ext = ".png"
                var isBase64 = false
                //First, find file type:
                if (uri_final.contains("image/gif")) {
                    ext = ".gif"
                } else if (uri_final.contains("image/jpeg") || uri_final.contains("image/pjpeg")) {
                    ext = ".jpg"
                } else if (uri_final.contains("image/png")) {
                    ext = ".png"
                } else if (uri_final.contains("image/tiff")) {
                    ext = ".tif"
                } else if (uri_final.contains("image/svg+xml")) {
                    ext = ".svg"
                }
                if (uri_final.contains("base64")) {
                    isBase64 = true
                }
                val out = File(
                    Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES),
                    System.currentTimeMillis().toString() + ext
                )
                try {
                    FileOutputStream(out).use { outstream ->
                        val out_array: ByteArray
                        if (out.canWrite()) {
                            val content = uri_final.substring(uri_final.lastIndexOf(",") + 1)
                            out_array = if (!isBase64) {
                                hexStringToByteArray(content)
                            } else {
                                Base64.decode(content, Base64.DEFAULT)
                            }
                            outstream.write(out_array)
                        }
                    }
                } catch (e: IOException) {
                    Logging.loge(
                        owner.context.resources.getString(
                            R.string.store_picture_error
                        )
                    )
                } catch (e: IllegalArgumentException) {
                    Logging.loge(
                        owner.context.resources.getString(
                            R.string.store_picture_error
                        )
                    )
                }
            } else {
                //Use the download manager
                val dm =
                    owner.context.getSystemService(Context.DOWNLOAD_SERVICE) as DownloadManager
                val r = DownloadManager.Request(Uri.parse(uri_final))

                //Check if we're writing to internal or external
                val pm = owner.context.packageManager
                if (pm.checkPermission(
                        Manifest.permission.WRITE_EXTERNAL_STORAGE,
                        owner.context.packageName
                    ) == PackageManager.PERMISSION_GRANTED
                ) {
                    r.setDestinationInExternalPublicDir(Environment.DIRECTORY_PICTURES,
                        uri_final.split("/".toRegex()).dropLastWhile { it.isEmpty() }
                            .toTypedArray()[uri_final.split("/".toRegex())
                            .dropLastWhile { it.isEmpty() }.toTypedArray().size - 1]
                    )
                    try {
                        r.allowScanningByMediaScanner()
                        r.setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
                        dm.enqueue(r)
                    } catch (ex: IllegalStateException) {
                        Logging.logd(
                            owner.context.resources.getString(
                                R.string.store_picture_error
                            )
                        )
                    }
                } else {
                    Logging.logd(
                        owner.context.resources.getString(
                            R.string.store_picture_error
                        )
                    )
                }
            }
        }.setNegativeButton(
            "store_picture_decline"
        ) { dialog: DialogInterface?, which: Int -> }

        val d = builder.create()
        d.show()
    }

    private fun playVideo(jsonObject: JSONObject?) {
        if (jsonObject == null) return
        val uri = jsonObject.optString(MraidCommands.URI)
        if (uri.isNullOrBlank()) {
            Logging.logd(owner.context.resources.getString(R.string.play_vide_no_uri))
            return
        }
        val i = Intent(Intent.ACTION_VIEW)
        i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        try {
            i.setDataAndType(
                Uri.parse(URLDecoder.decode(uri, MraidCommands.UTF_8)),
                MraidCommands.VIDEO_MP4
            )
        } catch (e: UnsupportedEncodingException) {
            Logging.logd(owner.context.resources.getString(R.string.unsupported_encoding))
            return
        }
        try {
            owner.context.startActivity(i)
        } catch (ignored: ActivityNotFoundException) {
        }
    }

    private fun createCalendarEvent(json: String?) {
        if (json.isNullOrBlank()) return
        val event = createFromJSON(json)
        try {
            val i = event.insertIntent
            i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            owner.context.startActivity(i)
            Logging.logd(owner.context.resources.getString(R.string.create_calendar_event))
        } catch (ignored: ActivityNotFoundException) {
        }
    }

    private fun resize(jsonObject: String?) {
        if (jsonObject == null) return

        val configuration = fromString(jsonObject)
        if (configuration != null) {
            owner.setConfiguration(configuration)
            resized = true
        }
    }

    private val screenSize: FloatArray
        get() {
            val screenSize = getScreenSizeAsPixels(owner.activity)
            val maxWidth = screenSize[0]
            val maxHeight = screenSize[1]
            // never null
            return floatArrayOf(maxWidth.value!!, maxHeight.value!!)
        }

    private fun setMaxSize() {
        val screenSize = screenSize

        val statusBarHeight: Int
        val navigationBarHeight: Int
        val leftSideWidth: Int
        val rightSideWidth: Int

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            val metrics: WindowMetrics = owner.activity.windowManager.currentWindowMetrics
            val insets = metrics.windowInsets.getInsets(WindowInsets.Type.systemBars())
            statusBarHeight = insets.top
            navigationBarHeight = insets.bottom
            leftSideWidth = insets.left
            rightSideWidth = insets.right
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            statusBarHeight = owner.rootWindowInsets?.systemWindowInsetTop ?: 0
            navigationBarHeight = owner.rootWindowInsets?.systemWindowInsetBottom ?: 0
            leftSideWidth = owner.rootWindowInsets?.systemWindowInsetLeft ?: 0
            rightSideWidth = owner.rootWindowInsets?.systemWindowInsetRight ?: 0
        } else {
            val orientation = Orientation.fromAndroidActivity(owner.activity)
            val resources = owner.activity.resources
            statusBarHeight = resources.getStatusBarSizePx()
            leftSideWidth = 0

            val navigationBarSize = resources.getNavigationBarSizePx(orientation)
            when (orientation) {
                Orientation.Portrait,
                Orientation.PortraitReverse,
                Orientation.Unspecified -> {
                    navigationBarHeight = navigationBarSize
                    rightSideWidth = 0
                }

                Orientation.Landscape,
                Orientation.LandscapeReverse -> {
                    navigationBarHeight = if (resources.isSmartPhone()) 0 else navigationBarSize
                    rightSideWidth = if (resources.isSmartPhone()) navigationBarSize else 0
                }
            }
        }

        val totalWidthCutouts = leftSideWidth + rightSideWidth
        val totalHeightCutouts = navigationBarHeight + statusBarHeight

        owner.injectJavaScript(
            setMaxSize(
                screenSize[0] - totalWidthCutouts,
                screenSize[1] - totalHeightCutouts,
            ),
        )
    }

    // same as max size for android
    private fun setScreenSize() {
        val screenSize = screenSize
        owner.injectJavaScript(
            setScreenSize(
                screenSize[0],
                screenSize[1],
                MraidDisplayMetrics(owner.activity.resources.displayMetrics),
            )
        )
    }

    //@NOTE  occlusionRectangles - Not Supported.
    fun fireExposureChangeEvent(exposedPercentage: Double, visibleRect: Rect?) {
        val newExposureVal = if (visibleRect != null) {
            String.format(
                Locale.ROOT,
                "{\"exposedPercentage\":%.1f,\"visibleRectangle\":{\"x\":%d,\"y\":%d,\"width\":%d,\"height\":%d},\"occlusionRectangles\":null}",
                exposedPercentage,
                visibleRect.left,
                visibleRect.top,
                visibleRect.width(),
                visibleRect.height()
            )
        } else {
            String.format(
                Locale.ROOT,
                "{\"exposedPercentage\":%.1f,\"visibleRectangle\":null,\"occlusionRectangles\":null}",
                exposedPercentage
            )
        }

        if (exposureVal != newExposureVal) {
            exposureVal = newExposureVal
            this.onExposureChange()
        }
    }

    // Fire AudioVolumeChange event whenever there is change in audio volume of device
    fun fireAudioVolumeChangeEvent(volumePercentage: Double?) {
        val newAudioVolumeVal = if (volumePercentage == null) {
            "{\"volumePercentage\":null}"
        } else {
            String.format(
                Locale.ROOT,
                "{\"volumePercentage\":%.1f}",
                volumePercentage
            )
        }
        this.onAudioVolumeChange(newAudioVolumeVal)
    }

    fun checkOrientationChange() {
        val orientation = Orientation.fromAndroidActivity(owner.activity)
        if (lastRotation != orientation) {
            lastRotation = orientation
            setCurrentAppOrientation(lastRotation)
            setMaxSize()
            setScreenSize()
        }
    }

    private fun startAudioVolumeListener() {
        if (mediaRouter == null) {
            mediaRouter =
                owner.context.getSystemService(Context.MEDIA_ROUTER_SERVICE) as MediaRouter
            callback = object : MediaRouter.Callback() {
                override fun onRouteSelected(
                    router: MediaRouter,
                    type: Int,
                    info: MediaRouter.RouteInfo
                ) {
                    Logging.logd("onRouteSelected " + info.name.toString())
                    // This is called when the MediaRoute is changed from Phone or Headphones to Bluetooth or vice-versa
                    fireAudioVolumeChangeEvent(audioVolumePercentage)
                }

                override fun onRouteUnselected(
                    router: MediaRouter,
                    type: Int,
                    info: MediaRouter.RouteInfo
                ) {
                    Logging.logd("onRouteUnselected " + info.name.toString())
                }

                override fun onRouteAdded(router: MediaRouter, info: MediaRouter.RouteInfo) {
                    Logging.logd("onRouteAdded " + info.name.toString())
                }

                override fun onRouteRemoved(router: MediaRouter, info: MediaRouter.RouteInfo) {
                    Logging.logd("onRouteRemoved " + info.name.toString())
                }

                override fun onRouteChanged(router: MediaRouter, info: MediaRouter.RouteInfo) {
                    Logging.logd("onRouteChanged " + info.name.toString())
                    // This is called when the MediaRoute is changed from Phone to Headphone or vice-versa
                    fireAudioVolumeChangeEvent(audioVolumePercentage)
                }

                override fun onRouteGrouped(
                    router: MediaRouter,
                    info: MediaRouter.RouteInfo,
                    group: RouteGroup,
                    index: Int
                ) {
                    Logging.logd("onRouteUngrouped " + info.name.toString())
                }

                override fun onRouteUngrouped(
                    router: MediaRouter,
                    info: MediaRouter.RouteInfo,
                    group: RouteGroup
                ) {
                    Logging.logd("onRouteUngrouped " + info.name.toString())
                }

                override fun onRouteVolumeChanged(
                    router: MediaRouter,
                    info: MediaRouter.RouteInfo
                ) {
                    Logging.logd("onRouteVolumeChanged " + router.defaultRoute.category.name.toString())
                    fireAudioVolumeChangeEvent(audioVolumePercentage)
                }
            }

            mediaRouter?.addCallback(
                MediaRouter.ROUTE_TYPE_USER,
                callback,
                MediaRouter.CALLBACK_FLAG_UNFILTERED_EVENTS
            )
            fireAudioVolumeChangeEvent(audioVolumePercentage)
        }
    }

    // Deallocate AudioVolumeContentObserver when memory cleanup takes place
    private fun stopAudioVolumeListener() {
        if (mediaRouter != null && callback != null) {
            mediaRouter?.removeCallback(callback)
            mediaRouter = null
            callback = null
        }
    }

    fun destroy() {
        stopAudioVolumeListener()
    }

    private val audioVolumePercentage: Double?
        // Helper method to fetch volume level of device
        get() {
            val audioManager =
                owner.context.getSystemService(Context.AUDIO_SERVICE) as AudioManager?
                    ?: return null
            val currentVolume = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
            val maxVolume = audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC)
            return (100.0 * currentVolume) / maxVolume
        }

    companion object {
        private const val DEFAULT_PLACEMENT_TYPE = "interstitial"

        private fun decode(value: String): String {
            return try {
                URLDecoder.decode(value, MraidCommands.UTF_8)
            } catch (ignored: UnsupportedEncodingException) {
                value
            }
        }
    }
}
