package sdk.main.core.inappmessaging.display.mraid

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Intent
import android.content.MutableContextWrapper
import android.graphics.Color
import android.graphics.Rect
import android.net.Uri
import android.net.http.SslError
import android.os.Build
import android.os.Handler
import android.view.ViewTreeObserver
import android.view.ViewTreeObserver.OnScrollChangedListener
import android.webkit.*
import androidx.activity.OnBackPressedCallback
import androidx.appcompat.app.AppCompatActivity
import ir.intrack.android.sdk.R
import sdk.main.core.inappmessaging.display.Configuration
import sdk.main.core.inappmessaging.display.internal.Logging
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.pageFinished
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setEnvironment
import sdk.main.core.inappmessaging.display.mraid.MraidCommands.setMessage
import sdk.main.core.inappmessaging.display.mraid.utils.ViewUtil
import sdk.main.core.inappmessaging.model.message.IAMMessage
import java.util.Locale

@SuppressLint("ViewConstructor")
class ViewportWebView(val activity: Activity, message: IAMMessage) : WebView(
    MutableContextWrapper(
        activity.baseContext
    )
), ViewTreeObserver.OnGlobalLayoutListener, OnScrollChangedListener {
    private val handler = Handler(activity.mainLooper)
    private var initialMraidState: MraidState = MraidState.Default
    private var isMRAIDEnabled = false
    private var implementation: MRAIDImplementation? = null
    private var firstPageFinished = false

    // for viewable event
    private var checkPaused = false
    private var timeOfLastCheckPosition = System.currentTimeMillis()
    private var mWebChromeClient: WebChromeClient? = null
    var mraidListener: MraidListener? = null
    private var isDestroyTriggered = false
    private val message: IAMMessage
    private val defaultConfiguration: Configuration
    lateinit var currentConfiguration: Configuration
        private set

    private val checkRunnable: Runnable = object : Runnable {
        override fun run() {
            if (checkPaused) return
            checkPosition()
            handler.postDelayed(this, CHECK_POSITION_TIME_INTERVAL.toLong())
        }
    }

    private val onBackPressedCallback: OnBackPressedCallback =
        object : OnBackPressedCallback(true) {
            override fun handleOnBackPressed() {
                injectJavaScript(MraidCommands.close())
            }
        }

    init {
        setup()
        Handler(activity.mainLooper).post {
            mraidListener?.onInitFinished()
        }
        loadDataWithBaseURL(
            null,
            message.html!!,
            MraidCommands.TEXT_HTML,
            MraidCommands.UTF_8,
            null
        )
        setConfiguration(message.config!!)
        this.defaultConfiguration = message.config.copy()
        this.message = message.copy(html = null, config = null)
    }

    fun setConfiguration(configuration: Configuration) {
        this.currentConfiguration = configuration
        mraidListener?.onConfigurationUpdated(configuration)
    }

    val isConfigurationChanged: Boolean
        get() = !currentConfiguration.isEqual(defaultConfiguration)

    fun revertConfiguration() {
        setConfiguration(defaultConfiguration)
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun setupSettings() {
        this.settings.javaScriptEnabled = true
        this.settings.javaScriptCanOpenWindowsAutomatically = true
        this.settings.builtInZoomControls = false
        this.settings.loadWithOverviewMode = true
        this.settings.loadsImagesAutomatically = true
        this.settings.setSupportZoom(false)
        this.settings.useWideViewPort = false
        this.settings.mediaPlaybackRequiresUserGesture = true
        this.settings.mixedContentMode = WebSettings.MIXED_CONTENT_ALWAYS_ALLOW
        this.settings.allowFileAccess = false
        this.settings.allowContentAccess = false
        this.settings.allowFileAccessFromFileURLs = false
        this.settings.allowUniversalAccessFromFileURLs = false

        val cm = CookieManager.getInstance()
        if (cm != null) {
            cm.setAcceptThirdPartyCookies(this, true)
        } else {
            Logging.logd("Failed to set Webview to accept 3rd party cookie")
        }

        setHorizontalScrollbarOverlay(false)
        isHorizontalScrollBarEnabled = false
        setVerticalScrollbarOverlay(false)
        isVerticalScrollBarEnabled = false

        setBackgroundColor(Color.TRANSPARENT)
        scrollBarStyle = SCROLLBARS_INSIDE_OVERLAY
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun setup() {
        setupSettings()
        implementation = MRAIDImplementation(this)
        webChromeClient = WebChromeClient(mraidListener).also { mWebChromeClient = it }
        webViewClient = AdWebViewClient()
        if (activity is AppCompatActivity) {
            activity.onBackPressedDispatcher.addCallback(onBackPressedCallback)
        }
    }

    override fun onConfigurationChanged(newConfig: android.content.res.Configuration) {
        super.onConfigurationChanged(newConfig)
        checkPosition()
    }

    fun fireAdClicked(url: String?) {
        try {
            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK) // Ensures a new task is started for the browser
            activity.startActivity(intent)
        } catch (ignored: Exception) {
        }
    }

    fun unload() {
        implementation?.unload(null)
    }

    override fun destroy() {
        isDestroyTriggered = true

        onBackPressedCallback.remove()
        // in case `this` was not removed when destroy was called
        ViewUtil.removeChildFromParent(this)

        val wrapper = context as MutableContextWrapper
        wrapper.baseContext = wrapper.applicationContext

        mWebChromeClient?.let {
            it.onHideCustomView()
            mWebChromeClient = null
            webChromeClient = null
        }

        implementation?.destroy()
        handler.postDelayed(
            {
                try {
                    super@ViewportWebView.destroy()
                } catch (e: IllegalArgumentException) {
                    Logging.loge(context.resources.getString(R.string.apn_webview_failed_to_destroy) + e)
                }
            },
            300
        )

        this.removeAllViews()
        stopChecking()
    }

    override fun scrollTo(x: Int, y: Int) {
        // to prevent webview from scrolling
    }

    fun fireMRAIDEnabled() {
        if (isMRAIDEnabled) return

        isMRAIDEnabled = true
        if (this.firstPageFinished) {
            implementation?.webViewFinishedLoading(this, initialMraidState)
            startChecking()
        }
    }

    fun checkPosition() {
        if (!isMRAIDEnabled) {
            return
        }
        if (tooManyCheckPositionRequests()) {
            return
        }

        // holds the visible area of a InAppWebView in global (root) coordinates
        val globalClippedArea = Rect()
        val visible = this.getGlobalVisibleRect(globalClippedArea)

        val visibleViewArea = (globalClippedArea.height() * globalClippedArea.width()).toDouble()
        val totalArea = (this.height * this.width).toDouble()
        val exposedPercentage = (visibleViewArea / totalArea) * 100

        // update current position
        implementation?.let {
            it.setCurrentPosition(currentConfiguration)
            it.checkOrientationChange()

            // exposureChange event logic
            if (visible) {
                // If at-least part of view is visible, then send exposure percentage and getLocalVisibleRect
                val localClippedArea = Rect()
                this.getLocalVisibleRect(localClippedArea)
                it.fireExposureChangeEvent(exposedPercentage, localClippedArea)
            } else {
                // No part of the view is visible then we need to send exposed percentage as 0.0 and visible rectangle as null
                it.fireExposureChangeEvent(0.0, null)
            }
        }

        timeOfLastCheckPosition = System.currentTimeMillis()
    }

    private fun tooManyCheckPositionRequests(): Boolean {
        val deltaT = System.currentTimeMillis() - timeOfLastCheckPosition
        return deltaT < MIN_MS_BETWEEN_CHECK_POSITION
    }

    private fun startChecking() {
        checkPaused = false
        handler.removeCallbacks(checkRunnable)
        handler.post(checkRunnable)
    }

    private fun stopChecking() {
        checkPaused = true
        handler.removeCallbacks(checkRunnable)
    }

    override fun onDetachedFromWindow() {
        removeViewTreeObserverListeners()
        super.onDetachedFromWindow()
    }

    private fun removeViewTreeObserverListeners() {
        val treeObserver = viewTreeObserver
        if (treeObserver.isAlive) {
            treeObserver.removeOnScrollChangedListener(this)
            treeObserver.removeOnGlobalLayoutListener(this)
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        setupViewTreeObserver()
    }

    private fun setupViewTreeObserver() {
        val treeObserver = viewTreeObserver
        if (treeObserver.isAlive) {
            treeObserver.removeOnScrollChangedListener(this)
            treeObserver.removeOnGlobalLayoutListener(this)
            treeObserver.addOnScrollChangedListener(this)
            treeObserver.addOnGlobalLayoutListener(this)
        }
    }

    override fun onGlobalLayout() {
        checkPosition()
    }

    override fun onScrollChanged() {
        checkPosition()
    }

    fun injectJavaScript(uri: String) {
        try {
            mraidListener?.onInjectJavaScript(uri)
            evaluateJavascript(uri, null)
        } catch (exception: Exception) {
            // We can't do anything much here if there is an exception ignoring.
            // This is to avoid crash of users app gracefully.
            Logging.loge("InAppWebView.injectJavaScript -- Caught EXCEPTION:$exception")
        }
    }

    private fun setMessage() {
        try {
            injectJavaScript(setMessage(message))
        } catch (e: Exception) {
            Logging.loge(e.toString())
        }
    }

    private fun setMraidEnvironment() {
        try {
            injectJavaScript(setEnvironment(MraidEnvironment()))
        } catch (e: Exception) {
            Logging.loge(e.toString())
        }
    }

    private fun setDefaultMraidConfiguration() {
        try {
            injectJavaScript(MraidCommands.setConfiguration(defaultConfiguration))
        } catch (e: Exception) {
            Logging.loge(e.toString())
        }
    }

    fun setOrientationProperties(orientation: Orientation?) {
        if (activity.isFinishing) {
            return
        }

        val newOrientation = orientation ?: Orientation.Unspecified
        activity.requestedOrientation = newOrientation.androidOrientation
    }

    /**
     * AdWebViewClient for the webview
     */
    private inner class AdWebViewClient : WebViewClient() {
        override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
            if (isDestroyTriggered) {
                return false
            }

            Logging.logd(
                "MRAID-toNative",
                "Loading URL: $url"
            )

            if (url.startsWith(MraidCommands.JS)) {
                return false
            }

            if (url.startsWith(MraidCommands.MRAID_SCHEME)) {
                val host = Uri.parse(url).host
                if (isMRAIDEnabled || MraidCommands.ENABLE == host || MraidCommands.OPEN == host) {
                    implementation?.dispatchMraidCall(url)
                }
                return true
            }

            fireAdClicked(url)

            return true
        }

        override fun onPageFinished(view: WebView, url: String) {
            Logging.logd("WebView finished loading: $url")
            super.onPageFinished(view, url)

            if (!firstPageFinished) {
                injectJavaScript(pageFinished())
                if (isMRAIDEnabled) {
                    implementation?.webViewFinishedLoading(this@ViewportWebView, initialMraidState)
                    startChecking()
                }

                firstPageFinished = true
                setMraidEnvironment()
                setMessage()
                setDefaultMraidConfiguration()
            }
            mraidListener?.onHtmlLoaded()
        }

        override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail): Boolean {
            Logging.loge("WebView onRenderProcessGone: $detail")

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                if (detail.didCrash()) {
                    Logging.loge("WebView crashed: $detail")
                }
            }
            view?.destroy()
            return true
        }

        override fun onLoadResource(view: WebView, url: String) {
            Logging.logd("WebView onLoadResource: $url")

            mraidListener?.onStartLoadingHtml()

            if (url.startsWith(MraidCommands.HTTP)) {
                val hitTestResult: HitTestResult
                try {
                    hitTestResult = getHitTestResult()
                    if (hitTestResult == null) {
                        return
                    }
                    // check that the hitTestResult matches the url
                    if (hitTestResult.extra == null || hitTestResult.extra != url) {
                        return
                    }
                } catch (e: NullPointerException) {
                    return
                }

                when (hitTestResult.type) {
                    HitTestResult.ANCHOR_TYPE, HitTestResult.SRC_ANCHOR_TYPE -> view.stopLoading()
                    else -> {}
                }
            }
        }

        override fun shouldInterceptRequest(
            view: WebView,
            request: WebResourceRequest
        ): WebResourceResponse? {
            Logging.logd("WebView shouldInterceptRequest: " + request.requestHeaders)

            try {
                // This intercepts resource loading requests from a webview stop loading the mraid.js from server.
                if (request.url.toString().lowercase(Locale.getDefault()).contains(
                        MraidCommands.MRAID_MINIFIED_FILE_NAME.lowercase(
                            Locale.getDefault()
                        )
                    )
                ) {
                    return WebResourceResponse(
                        MraidCommands.TEXT_JS,
                        MraidCommands.UTF_8,
                        activity.assets.open(MraidCommands.MRAID_MINIFIED_FILE_NAME)
                    )
                }
            } catch (e: Exception) {
                Logging.loge("", e)
            }
            return super.shouldInterceptRequest(view, request)
        }

        override fun onReceivedError(
            view: WebView,
            errorCode: Int,
            description: String,
            failingURL: String
        ) {
            mraidListener?.onHtmlError(errorCode, "$failingURL -- $description")

            Logging.logi(
                context.resources.getString(
                    R.string.webview_received_error,
                    errorCode,
                    description,
                    failingURL
                )
            )
        }

        override fun onReceivedSslError(view: WebView, handler: SslErrorHandler, error: SslError) {
            handler.cancel()
            try {
                Logging.logi(
                    context.resources.getString(
                        R.string.webclient_error,
                        error.primaryError,
                        error.toString()
                    )
                )
            } catch (ignored: NullPointerException) {
            }
        }
    }

    companion object {
        private const val CHECK_POSITION_TIME_INTERVAL = 1000
        private const val MIN_MS_BETWEEN_CHECK_POSITION = 200
    }
}
