package sdk.main.core.pushmessaging

import android.app.Notification
import android.app.Service
import android.content.Intent
import android.os.Build
import android.os.CountDownTimer
import android.os.IBinder
import android.os.SystemClock
import android.view.View
import android.widget.RemoteViews
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import ir.intrack.android.sdk.R
import kotlin.math.max

class TimerNotificationService : Service() {
    private data class TimerNotification(
        val timer: CountDownTimer,
        val builder: NotificationCompat.Builder,
        val remoteViews: RemoteViews,
        val message: PushMessage
    )

    private val activeTimers = mutableMapOf<Int, TimerNotification>()

    override fun onBind(intent: Intent?): IBinder? = null

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        when (intent?.action) {
            ACTION_STOP_TIMER -> {
                val notificationId = intent.getIntExtra(EXTRA_NOTIFICATION_ID, -1)
                if (notificationId != -1) {
                    activeTimers[notificationId]?.timer?.cancel()
                    activeTimers.remove(notificationId)
                    if (activeTimers.isEmpty()) {
                        stopSelf()
                    }
                }
                return START_NOT_STICKY
            }

            else -> {
                intent?.let { handleIntent(it) }
                return START_STICKY
            }
        }
    }

    private fun handleIntent(intent: Intent) {
        val notificationMsg = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            intent.getParcelableExtra(EXTRA_NOTIFICATION, PushMessage::class.java)
        } else {
            @Suppress("DEPRECATION")
            intent.getSerializableExtra(EXTRA_NOTIFICATION) as? PushMessage
        }

        val notificationIntent = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            intent.getParcelableExtra(EXTRA_NOTIFICATION_INTENT, Intent::class.java)
        } else {
            @Suppress("DEPRECATION")
            intent.getParcelableExtra(EXTRA_NOTIFICATION_INTENT)
        }

        val notificationId = intent.getIntExtra(EXTRA_NOTIFICATION_ID, 0)
        val smallIcon = intent.getIntExtra(EXTRA_SMALL_ICON, 0)
        val channelId = intent.getStringExtra(EXTRA_CHANNEL_ID)
        val timerDuration = intent.getLongExtra(EXTRA_TIMER_DURATION, 0)
        val progressDuration = intent.getLongExtra(EXTRA_PROGRESS_DURATION, 0)

        if (notificationMsg == null || channelId == null) {
            return
        }
        activeTimers[notificationId]?.timer?.cancel()
        activeTimers.remove(notificationId)

        createAndStartTimer(
            notificationId,
            notificationMsg,
            notificationIntent,
            smallIcon,
            channelId,
            timerDuration,
            progressDuration
        )
    }

    private fun createAndStartTimer(
        notificationId: Int,
        pushMessage: PushMessage,
        notificationIntent: Intent?,
        smallIcon: Int,
        channelId: String,
        timerDuration: Long,
        progressDuration: Long
    ) {
        val builder = NotificationCompat.Builder(this, channelId)
            .setAutoCancel(true)
            .setSmallIcon(smallIcon)
            .setTicker(pushMessage.message)
            .setContentTitle(pushMessage.title)
            .setContentText(pushMessage.message)
            .setOngoing(pushMessage.sticky)

        val remoteViews = createRemoteViews(pushMessage, timerDuration)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            builder.setCustomContentView(remoteViews)
                .setStyle(NotificationCompat.DecoratedCustomViewStyle())
        } else {
            builder.setContent(remoteViews)
        }

        NotificationUtils.setNotificationButtons(
            this,
            builder,
            pushMessage,
            notificationId,
            notificationIntent
        )
        NotificationUtils.setNotificationClickAction(
            this,
            builder,
            pushMessage,
            notificationId,
            notificationIntent
        )

        val progressBarEndTime = System.currentTimeMillis() + progressDuration
        val timer = createCountDownTimer(
            notificationId,
            timerDuration,
            progressDuration,
            progressBarEndTime,
            remoteViews,
            pushMessage
        )

        activeTimers[notificationId] = TimerNotification(
            timer = timer,
            builder = builder,
            remoteViews = remoteViews,
            message = pushMessage
        )
        timer.start()
        showNotification(notificationId)
        startForeground(notificationId, builder.build())
    }

    private fun createRemoteViews(pushMessage: PushMessage, timerDuration: Long): RemoteViews {
        val timerTemplate = pushMessage.timerTemplate
        if (timerTemplate?.type == null) {
            return RemoteViews(packageName, R.layout.layout_timer)
        }

        val builder = TimerRemoteViewBuilder(timerTemplate.type.timerLayoutResId)
            .setTitle(pushMessage.title)
            .setTitleColor(pushMessage.titleColor)
            .setMessage(pushMessage.message)
            .setMessageColor(pushMessage.messageColor)
            .setChronometerTextColor(timerTemplate.timeColor)
            .setCountdownDuration(timerDuration)

        if (timerTemplate.type == TimerType.PROGRESS_BAR) {
            builder.setProgressBarColor(timerTemplate.progressBarColor)
        }

        return builder.build(applicationContext)
    }

    private fun createCountDownTimer(
        notificationId: Int,
        timerDuration: Long,
        progressDuration: Long,
        progressBarEndTime: Long,
        remoteViews: RemoteViews,
        pushMessage: PushMessage
    ): CountDownTimer {
        return object : CountDownTimer(timerDuration, 1000) {
            override fun onTick(millisUntilFinished: Long) {
                val baseTime = SystemClock.elapsedRealtime() + millisUntilFinished
                remoteViews.setChronometer(R.id.chronometer, baseTime, null, true)

                if (pushMessage.timerTemplate?.type == TimerType.PROGRESS_BAR) {
                    val remainingProgressTime =
                        max(0, progressBarEndTime - System.currentTimeMillis())
                    val progress = ((remainingProgressTime * 100) / progressDuration).toInt()
                    remoteViews.setProgressBar(R.id.notification_progressBar, 100, progress, false)
                }

                val remainingHours = millisUntilFinished / (1000 * 60 * 60)
                remoteViews.setViewVisibility(
                    R.id.hourLabel,
                    if (remainingHours < 1) View.GONE else View.VISIBLE
                )

                showNotification(notificationId)
            }

            override fun onFinish() {
                remoteViews.setChronometer(
                    R.id.chronometer,
                    SystemClock.elapsedRealtime(),
                    null,
                    false
                )
                remoteViews.setViewVisibility(R.id.hourLabel, View.GONE)
                remoteViews.setProgressBar(R.id.notification_progressBar, 100, 0, false)

                showNotification(notificationId)
                activeTimers.remove(notificationId)
                if (activeTimers.isEmpty()) {
                    stopSelf()
                }
            }
        }
    }

    private fun showNotification(notificationId: Int) {
        val timerNotification = activeTimers[notificationId] ?: return

        NotificationManagerCompat.from(this).notify(
            notificationId,
            timerNotification.builder.build().apply {
                flags =
                    flags or (Notification.FLAG_ONLY_ALERT_ONCE or Notification.FLAG_AUTO_CANCEL)
            }
        )
    }

    override fun onDestroy() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            stopForeground(STOP_FOREGROUND_REMOVE)
        } else {
            stopForeground(true)
        }
        activeTimers.values.forEach { it.timer.cancel() }
        activeTimers.clear()
        super.onDestroy()
    }

    companion object {
        const val EXTRA_NOTIFICATION = "notification"
        const val EXTRA_NOTIFICATION_ID = "notification_id"
        const val EXTRA_NOTIFICATION_INTENT = "notification_intent"
        const val EXTRA_SMALL_ICON = "small_icon"
        const val EXTRA_CHANNEL_ID = "channel_id"
        const val EXTRA_TIMER_DURATION = "timer_duration"
        const val EXTRA_PROGRESS_DURATION = "progress_duration"
        const val ACTION_STOP_TIMER = "action_stop_timer"
        const val ACTION_DISMISS_NOTIFICATION = "action_dismiss_notification_"
    }
}
