package ir.intrack.android.sdk.prefs

import android.content.Context
import android.content.SharedPreferences
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.lifecycle.MutableLiveData
import com.google.gson.annotations.Expose
import com.google.gson.annotations.SerializedName
import com.google.gson.reflect.TypeToken
import sdk.main.core.ext.fromJson
import sdk.main.core.ext.json
import sdk.main.core.ext.setOrPostValue
import java.lang.reflect.Type
import java.util.function.BiFunction
import kotlin.reflect.KProperty


data class JsonMapWrapper<Key, Value>(
    @SerializedName("k") val key: Key,
    @SerializedName("v") val value: Value?,
)

class PrefsHashMap<Key : Any, Value : Any?>(
    @field:Transient
    @field:Expose(serialize = false, deserialize = false)
    val changedCallback: (HashMap<Key, Value?>) -> Unit,
) : HashMap<Key, Value?>() {

    var internalLevel = 0
    private fun <T> withSingleChangeCallback(body: () -> T): T {
        internalLevel++
        val ans = body()
        internalLevel--
        if (internalLevel == 0)
            changedCallback(this)
        return ans
    }

    override fun clear() = withSingleChangeCallback {
        super.clear()
    }

    override fun put(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.put(key, value)
    }

    override fun putAll(from: Map<out Key, Value?>) = withSingleChangeCallback {
        super.putAll(from)
    }

    override fun remove(key: Key): Value? = withSingleChangeCallback {
        super.remove(key)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun remove(key: Key, value: Value?): Boolean = withSingleChangeCallback {
        super.remove(key, value)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replaceAll(function: BiFunction<in Key, in Value?, out Value?>) =
        withSingleChangeCallback {
            super.replaceAll(function)
        }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun putIfAbsent(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.putIfAbsent(key, value)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replace(key: Key, oldValue: Value?, newValue: Value?): Boolean =
        withSingleChangeCallback {
            super.replace(key, oldValue, newValue)
        }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replace(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.replace(key, value)
    }

//    @RequiresApi(Build.VERSION_CODES.N)
//    override fun merge(key: Key, value: Value, remappingFunction: BiFunction<in Value, in Value, out Value?>): Value? {
//        return super.merge(key, value, remappingFunction).apply {
//            changedCallback(this)
//        }
//    }
}

class PrefsLinkedHashMap<Key : Any, Value : Any?>(
    @field:Transient
    @field:Expose(serialize = false, deserialize = false)
    val changedCallback: (LinkedHashMap<Key, Value?>) -> Unit,
) : LinkedHashMap<Key, Value?>() {

    var internalLevel = 0
    private fun <T> withSingleChangeCallback(body: () -> T): T {
        internalLevel++
        val ans = body()
        internalLevel--
        if (internalLevel == 0)
            changedCallback(this)
        return ans
    }

    override fun clear() = withSingleChangeCallback {
        super.clear()
    }

    override fun put(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.put(key, value)
    }

    override fun putAll(from: Map<out Key, Value?>) = withSingleChangeCallback {
        super.putAll(from)
    }

    override fun remove(key: Key): Value? = withSingleChangeCallback {
        super.remove(key)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun remove(key: Key, value: Value?): Boolean = withSingleChangeCallback {
        super.remove(key, value)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replaceAll(function: BiFunction<in Key, in Value?, out Value?>) =
        withSingleChangeCallback {
            super.replaceAll(function)
        }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun putIfAbsent(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.putIfAbsent(key, value)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replace(key: Key, oldValue: Value?, newValue: Value?): Boolean =
        withSingleChangeCallback {
            super.replace(key, oldValue, newValue)
        }

    @RequiresApi(Build.VERSION_CODES.N)
    override fun replace(key: Key, value: Value?): Value? = withSingleChangeCallback {
        super.replace(key, value)
    }

//    @RequiresApi(Build.VERSION_CODES.N)
//    override fun merge(key: Key, value: Value, remappingFunction: BiFunction<in Value, in Value, out Value?>): Value? {
//        return super.merge(key, value, remappingFunction).apply {
//            changedCallback(this)
//        }
//    }
}

@Suppress("UNCHECKED_CAST", "unused")
abstract class PrefsWrapper(
    private var context: Context,
    private val sharedPreferencesName: String = "${context.packageName}.Prefs",
) {

    //region Listeners
    interface SharedPrefsListener {
        fun onSharedPrefChanged(property: KProperty<*>)
    }

    fun addListener(sharedPrefsListener: SharedPrefsListener) = listeners.add(sharedPrefsListener)
    fun removeListener(sharedPrefsListener: SharedPrefsListener) =
        listeners.remove(sharedPrefsListener)

    fun clearListeners() = listeners.clear()

    private fun onPrefChanged(property: KProperty<*>) {
        listeners.forEach { it.onSharedPrefChanged(property) }
    }

    private val listeners = mutableListOf<SharedPrefsListener>()
    //endregion

    val prefs: SharedPreferences by lazy {
        context.getSharedPreferences(sharedPreferencesName, Context.MODE_PRIVATE)
    }

    abstract inner class NonNullPrefDelegate<T>(var prefKey: String?) {
        operator fun getValue(thisRef: Any?, property: KProperty<*>): T {
            if (prefKey == null) {
                prefKey = property.name
            }
            return getValue(prefKey ?: property.name)
        }

        operator fun setValue(thisRef: Any?, property: KProperty<*>, value: T) {
            if (prefKey == null) {
                prefKey = property.name
            }
            setValue(prefKey ?: property.name, value)
            onPrefChanged(property)
        }

        abstract fun getValue(preferencesKey: String): T
        abstract fun setValue(preferencesKey: String, value: T)
    }

    abstract inner class NullablePrefDelegate<T>(var prefKey: String?) {
        operator fun getValue(thisRef: Any?, property: KProperty<*>): T? {
            if (prefKey == null) {
                prefKey = property.name
            }
            return getValue(prefKey ?: property.name)
        }

        operator fun setValue(thisRef: Any?, property: KProperty<*>, value: T?) {
            if (prefKey == null) {
                prefKey = property.name
            }
            setValue(prefKey ?: property.name, value)
            onPrefChanged(property)
        }

        abstract fun getValue(preferencesKey: String): T?
        abstract fun setValue(preferencesKey: String, value: T?)
    }


    fun intPref(defaultValue: Int = 0, prefKey: String? = null): NonNullPrefDelegate<Int> =
        object : NonNullPrefDelegate<Int>(prefKey) {
            override fun getValue(preferencesKey: String): Int =
                prefs.getInt(preferencesKey, defaultValue)

            override fun setValue(preferencesKey: String, value: Int) =
                prefs.edit().putInt(preferencesKey, value).apply()
        }

    fun floatPref(defaultValue: Float = 0f, prefKey: String? = null): NonNullPrefDelegate<Float> =
        object : NonNullPrefDelegate<Float>(prefKey) {
            override fun getValue(preferencesKey: String): Float =
                prefs.getFloat(preferencesKey, defaultValue)

            override fun setValue(preferencesKey: String, value: Float) =
                prefs.edit().putFloat(preferencesKey, value).apply()
        }

    fun booleanPref(
        defaultValue: Boolean = false,
        prefKey: String? = null,
    ): NonNullPrefDelegate<Boolean> = object : NonNullPrefDelegate<Boolean>(prefKey) {
        override fun getValue(preferencesKey: String): Boolean =
            prefs.getBoolean(preferencesKey, defaultValue)

        override fun setValue(preferencesKey: String, value: Boolean) =
            prefs.edit().putBoolean(preferencesKey, value).apply()
    }

    fun longPref(defaultValue: Long = 0L, prefKey: String? = null): NonNullPrefDelegate<Long> =
        object : NonNullPrefDelegate<Long>(prefKey) {
            override fun getValue(preferencesKey: String): Long =
                prefs.getLong(preferencesKey, defaultValue)

            override fun setValue(preferencesKey: String, value: Long) =
                prefs.edit().putLong(preferencesKey, value).apply()
        }


    fun stringPref(
        defaultValue: String? = null,
        prefKey: String? = null,
    ): NullablePrefDelegate<String> = object : NullablePrefDelegate<String>(prefKey) {
        override fun getValue(preferencesKey: String): String? =
            prefs.getString(preferencesKey, defaultValue)

        override fun setValue(preferencesKey: String, value: String?) =
            prefs.edit().putString(preferencesKey, value).apply()
    }

    fun stringSetPref(
        defaultValue: Set<String>? = HashSet(),
        prefKey: String? = null,
    ): NullablePrefDelegate<Set<String>> = object : NullablePrefDelegate<Set<String>>(prefKey) {
        override fun getValue(preferencesKey: String): Set<String>? =
            prefs.getStringSet(preferencesKey, defaultValue)

        override fun setValue(preferencesKey: String, value: Set<String>?) =
            prefs.edit().putStringSet(preferencesKey, value).apply()
    }

    inline fun <reified T : Any?> pref(
        defaultValue: T? = null,
        prefKey: String? = null,
    ): NullablePrefDelegate<T> = object : NullablePrefDelegate<T>(prefKey) {
        override fun getValue(preferencesKey: String): T? =
            prefs.getString(preferencesKey, "")?.fromJson<T>(T::class.java) ?: defaultValue

        override fun setValue(preferencesKey: String, value: T?) =
            prefs.edit().putString(preferencesKey, (value as? Any)?.json ?: "").apply()
    }

    inner class HashMapPrefsDelegate<Key : Any, Value : Any?>(prKey: String?, val typeToken: Type) :
        NonNullPrefDelegate<HashMap<Key, Value?>>(prKey) {
        var changedCallback: () -> Unit = {}
        var isLoopbackChange = false
        var isInit = false
        var internalMap = PrefsHashMap<Key, Value?> { changedMap ->
            if (prefKey != null && !isLoopbackChange) {
                val wrapper = ArrayList<JsonMapWrapper<Key, Value?>>(changedMap.size)
                for ((k, v) in changedMap) {
                    wrapper += JsonMapWrapper(k, v)
                }
                prefs.edit().putString(prefKey, wrapper.json).apply()
                isLoopbackChange = true
                changedCallback()
                isLoopbackChange = false
            }
        }

        override fun getValue(preferencesKey: String): HashMap<Key, Value?> {
            if (!isInit) {
                isInit = true
                this.prefKey = preferencesKey

                val parsed = prefs.getString(preferencesKey, "")
                    ?.fromJson<ArrayList<JsonMapWrapper<Key?, Value?>>>(typeToken) ?: arrayListOf()

                val linkedMap = LinkedHashMap<Key, Value?>(parsed.size)
                for ((k, v) in parsed) {
                    if (k != null)
                        linkedMap[k] = v
                }

                internalMap.putAll(linkedMap)
            }
            return internalMap
        }

        override fun setValue(preferencesKey: String, value: HashMap<Key, Value?>) {
            if (internalMap !== value) {
                val shouldDisableLoopAfter = isLoopbackChange
                isLoopbackChange = true
                internalMap.clear()
                if (!shouldDisableLoopAfter)
                    isLoopbackChange = false
                internalMap.putAll(value)
                isLoopbackChange = false
            } else if (!isLoopbackChange) {
                isLoopbackChange = true
                changedCallback()
                isLoopbackChange = false
            }
        }
    }

    inner class LinkedHashMapPrefsDelegate<Key : Any, Value : Any?>(
        prKey: String?,
        val typeToken: Type,
    ) : NonNullPrefDelegate<LinkedHashMap<Key, Value?>>(prKey) {
        var changedCallback: () -> Unit = {}
        var isLoopbackChange = false
        var isInit = false
        var internalMap = PrefsLinkedHashMap<Key, Value?> { changedMap ->
            if (prefKey != null && !isLoopbackChange) {
                val wrapper = ArrayList<JsonMapWrapper<Key, Value?>>(changedMap.size)
                for ((k, v) in changedMap) {
                    wrapper += JsonMapWrapper(k, v)
                }
                prefs.edit().putString(prefKey, wrapper.json).apply()
                isLoopbackChange = true
                changedCallback()
                isLoopbackChange = false
            }
        }

        override fun getValue(preferencesKey: String): LinkedHashMap<Key, Value?> {
            if (!isInit) {
                isInit = true
                this.prefKey = preferencesKey

                val parsed = prefs.getString(preferencesKey, "")
                    ?.fromJson<ArrayList<JsonMapWrapper<Key?, Value?>>>(typeToken) ?: arrayListOf()

                val linkedMap = LinkedHashMap<Key, Value?>(parsed.size)
                for ((k, v) in parsed) {
                    if (k != null)
                        linkedMap[k] = v
                }

                internalMap.putAll(linkedMap)
            }
            return internalMap
        }

        override fun setValue(preferencesKey: String, value: LinkedHashMap<Key, Value?>) {
            if (internalMap !== value) {
                val shouldDisableLoopAfter = isLoopbackChange
                isLoopbackChange = true
                internalMap.clear()
                if (!shouldDisableLoopAfter)
                    isLoopbackChange = false
                internalMap.putAll(value)
                isLoopbackChange = false
            } else if (!isLoopbackChange) {
                isLoopbackChange = true
                changedCallback()
                isLoopbackChange = false
            }
        }
    }

    inline fun <reified Key : Any, reified Value : Any?> hashMapPrefs(prefKey: String? = null) =
        HashMapPrefsDelegate<Key, Value?>(
            prefKey, TypeToken.getParameterized(
                ArrayList::class.java, TypeToken.getParameterized(
                    JsonMapWrapper::class.java, Key::class.java, Value::class.java
                ).type
            ).type
        )

    inline fun <reified Key : Any, reified Value : Any?> linkedHashMapPrefs(prefKey: String? = null) =
        LinkedHashMapPrefsDelegate<Key, Value?>(
            prefKey, TypeToken.getParameterized(
                ArrayList::class.java, TypeToken.getParameterized(
                    JsonMapWrapper::class.java, Key::class.java, Value::class.java
                ).type
            ).type
        )


    inner class NullableLiveDataPrefDelegate<T : Any?>(var innerValue: NullablePrefDelegate<T>) {
        private var isInitialized = false
        private val liveData: MutableLiveData<T?> = object : MutableLiveData<T?>() {
            override fun setValue(value: T?) {
                if (isInitialized) {
                    innerValue.prefKey?.let { prefKey ->
                        innerValue.setValue(prefKey, value)
                    }
                }

                super.setValue(value)
            }
        }

        operator fun getValue(thisRef: Any?, property: KProperty<*>): MutableLiveData<T?> {
            if (innerValue.prefKey == null) {
                innerValue.prefKey = property.name
            }
            if (!isInitialized) {
                liveData.setOrPostValue(innerValue.getValue(innerValue.prefKey ?: property.name))
                isInitialized = true
            }
            return liveData
        }

    }

    inner class NonNullLiveDataPrefDelegate<T : Any>(var innerValue: NonNullPrefDelegate<T>) {
        private var isInitialized = false

        init {
            (innerValue as? HashMapPrefsDelegate<*, *>)?.let { inner ->
                inner.changedCallback = {
                    liveData.setValue(inner.internalMap as T)
                }
            }
        }

        private val liveData: MutableLiveData<T> = object : MutableLiveData<T>() {
            override fun setValue(value: T) {
                if (isInitialized) {
                    innerValue.prefKey?.let { prefKey ->
                        (innerValue as? HashMapPrefsDelegate<*, *>)?.isLoopbackChange = true
                        innerValue.setValue(prefKey, value)
                        (innerValue as? HashMapPrefsDelegate<*, *>)?.isLoopbackChange = false
                    }
                }
                (innerValue as? HashMapPrefsDelegate<*, *>)?.let { inner ->
                    super.setValue(inner.internalMap as T)
                    return
                }
                super.setValue(value)
            }
        }

        operator fun getValue(thisRef: Any?, property: KProperty<*>): MutableLiveData<T> {
            if (innerValue.prefKey == null) {
                innerValue.prefKey = property.name
            }
            if (!isInitialized) {
                liveData.setOrPostValue(innerValue.getValue(innerValue.prefKey ?: property.name))
                isInitialized = true
            }
            return liveData
        }
    }

    inline fun <reified T : Any> liveData(nonNullInnerPrefs: NonNullPrefDelegate<T>) =
        NonNullLiveDataPrefDelegate(nonNullInnerPrefs)

    inline fun <reified T : Any?> liveData(nullablePrefs: NullablePrefDelegate<T>) =
        NullableLiveDataPrefDelegate(nullablePrefs)


}