package sdk.main.core;

import android.content.Context;

import java.lang.reflect.Method;

class AdvertisingIdAdapter {
    private final static String ADVERTISING_ID_CLIENT_CLASS_NAME = "com.google.android.gms.ads.identifier.AdvertisingIdClient";

    public static boolean isAdvertisingIdAvailable() {
        boolean advertisingIdAvailable = false;
        if (!CoreInternal.sharedInstance().config_.collectAdvertisingId) {
            return false;
        }
        try {
            Class.forName(ADVERTISING_ID_CLIENT_CLASS_NAME);
            advertisingIdAvailable = true;
        } catch (ClassNotFoundException ignored) {
        }
        return advertisingIdAvailable;
    }

    public static void setAdvertisingId(final Context context, final SharedPref store, final DeviceId deviceId) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    deviceId.setId(DeviceIdType.ADVERTISING_ID, getAdvertisingId(context));
                } catch (Throwable t) {
                    if (t.getCause() != null && t.getCause().getClass().toString().contains("GooglePlayServicesAvailabilityException")) {
                        // recoverable, let device ID be null, which will result in storing all requests to  server
                        // and rerunning them whenever Advertising ID becomes available
                        CoreInternal.sharedInstance().L.i("[AdvertisingIdAdapter] Advertising ID cannot be determined yet");
                    } else if (t.getCause() != null && t.getCause().getClass().toString().contains("GooglePlayServicesNotAvailableException")) {
                        // non-recoverable, fallback to OpenUDID
                        CoreInternal.sharedInstance().L.w("[AdvertisingIdAdapter] Advertising ID cannot be determined because Play Services are not available");
                        deviceId.switchToIdType(DeviceIdType.OPEN_UDID, context, store);
                    } else {
                        // unexpected
                        CoreInternal.sharedInstance().L.e("[AdvertisingIdAdapter] Couldn't get advertising ID", t);
                    }
                }
            }
        }).start();
    }

    public static String getAdvertisingId(final Context context) throws Throwable {
        final Class<?> cls = Class.forName(ADVERTISING_ID_CLIENT_CLASS_NAME);
        final Method getAdvertisingIdInfo = cls.getMethod("getAdvertisingIdInfo", Context.class);
        Object info = getAdvertisingIdInfo.invoke(null, context);
        if (info != null) {
            final Method getId = info.getClass().getMethod("getId");
            Object id = getId.invoke(info);
            return (String) id;
        }
        return null;
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private static boolean isLimitAdTrackingEnabled(final Context context) {
        //noinspection CatchMayIgnoreException
        if (!CoreInternal.sharedInstance().config_.collectAdvertisingId) {
            return true;
        }
        try {
            final Class<?> cls = Class.forName(ADVERTISING_ID_CLIENT_CLASS_NAME);
            final Method getAdvertisingIdInfo = cls.getMethod("getAdvertisingIdInfo", Context.class);
            Object info = getAdvertisingIdInfo.invoke(null, context);
            if (info != null) {
                final Method getId = info.getClass().getMethod("isLimitAdTrackingEnabled");
                Object id = getId.invoke(info);
                return (boolean) id;
            }
        } catch (Throwable t) {
            if (t.getCause() != null && t.getCause().getClass().toString().contains("java.lang.ClassNotFoundException") &&
                t.getCause().getMessage().contains("com.google.android.gms.ads.identifier.AdvertisingIdClient")) {
                CoreInternal.sharedInstance().L.w("[AdvertisingIdAdapter] Play Services are not available, while checking if limited ad tracking enabled");
            }
        }
        return false;
    }

    /**
     * Cache advertising ID for attribution
     */
    protected static void cacheAdvertisingID(final Context context, final SharedPref store) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    if (!isLimitAdTrackingEnabled(context)) {
                        String adId = getAdvertisingId(context);
                        store.setCachedAdvertisingId(adId);
                    } else {
                        store.setCachedAdvertisingId("");
                    }
                } catch (Throwable t) {
                    if (t.getCause() != null && t.getCause().getClass().toString().contains("GooglePlayServicesAvailabilityException")) {
                        CoreInternal.sharedInstance().L.i("[AdvertisingIdAdapter] Advertising ID cannot be determined yet, while caching");
                    } else if (t.getCause() != null && t.getCause().getClass().toString().contains("GooglePlayServicesNotAvailableException")) {
                        CoreInternal.sharedInstance().L.w("[AdvertisingIdAdapter] Advertising ID cannot be determined because Play Services are not available, while caching");
                    } else if (t.getCause() != null && t.getCause().getClass().toString().contains("java.lang.ClassNotFoundException") &&
                        t.getCause().getMessage().contains("com.google.android.gms.ads.identifier.AdvertisingIdClient")) {
                        CoreInternal.sharedInstance().L.w("[AdvertisingIdAdapter] Play Services are not available, while caching advertising id");
                    } else {
                        // unexpected
                        CoreInternal.sharedInstance().L.e("[AdvertisingIdAdapter] Couldn't get advertising ID, while caching", t);
                    }
                }
            }
        }).start();
    }
}
