package sdk.main.core

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.os.Build
import android.util.Log
import android.widget.Toast
import java.io.FileNotFoundException
import java.io.IOException
import java.io.InputStreamReader

class ClipboardManager {
    fun copyToClipboard(context: Context, text: String?): Boolean {
        return try {
            val clipboard = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
            val clip = ClipData.newPlainText("copied", text)
            clipboard.setPrimaryClip(clip)
            if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.S_V2)
                Toast.makeText(context, "Copied", Toast.LENGTH_SHORT).show()
            true
        } catch (e: Exception) {
            false
        }
    }

    fun readFromClipboard(context: Context): String {
        val clipboard = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        return clipboard.primaryClip?.let {
            coerceToText(context, it.getItemAt(0)).toString()
        } ?: ""
    }

    fun coerceToText(context: Context, item: ClipData.Item): CharSequence {
        // If this Item has an explicit textual value, simply return that.
        val text = item.text
        if (text != null) {
            return text
        }

        // If this Item has a URI value, try using that.
        val uri = item.uri
        if (uri != null) {
            // First see if the URI can be opened as a plain text stream
            // (of any sub-type). If so, this is the best textual
            // representation for it.
            try {
                // Ask for a stream of the desired type.
                context.contentResolver
                    .openTypedAssetFileDescriptor(uri, "text/*", null)
                    ?.createInputStream()
                    ?.use { stream ->
                        val reader = InputStreamReader(stream, "UTF-8")

                        // copy the stream into a local string and return it.
                        val builder = StringBuilder(128)
                        val buffer = CharArray(8192)
                        var len: Int
                        while (reader.read(buffer).also { len = it } > 0) {
                            builder.appendRange(buffer, 0, len)
                        }
                        return builder.toString()
                    }
            } catch (ignored: FileNotFoundException) {
                // Unable to open content URI as text... not really an
                // error, just something to ignore.
            } catch (e: IOException) {
                // Something bad has happened.
                Log.w("ClippedData", "Failure loading text", e)
                return e.toString()
            }

            // If we couldn't open the URI as a stream, then the URI itself
            // probably serves fairly well as a textual representation.
            return uri.toString()
        }

        // Finally, if all we have is an Intent, then we can just turn that
        // into text. Not the most user-friendly thing, but it's something.
        val intent = item.intent
        return if (intent != null) intent.toUri(Intent.URI_INTENT_SCHEME) else ""
    }
}
