package sdk.main.core;

import android.app.Application;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Map;

import ir.intrack.android.sdk.BuildConfig;

public class Config {

    /**
     * Internal fields and fields for testing
     */

    protected @Nullable SharedPref sharedPref = null;

    protected @Nullable AppInfoPrefs appInfoPrefs = null;

    protected boolean checkForNativeCrashDumps = true;

    //used to deliver this object to connection queue
    protected @Nullable DeviceId deviceIdInstance = null;

    /**
     * Android context.
     * Mandatory field.
     */
    protected @Nullable Context context = null;

    /**
     * URL of the server to submit data to.
     * Mandatory field.
     */
    protected @Nullable String serverURL = setServerUrlBaseOnFlavorName();

    /**
     * app key for the application being tracked; find in the  Dashboard under Management &gt; Applications.
     * Mandatory field.
     */
    protected @Nullable String appKey = null;

    protected @Nullable String authKey = null;

    protected @NonNull Boolean isDemoApp = false;
    /**
     * unique ID for the device the app is running on; note that null in deviceID means that SDK will fall back to OpenUDID, then, if it's not available, to Google Advertising ID.
     */
    protected @Nullable String deviceID = null;

    /**
     * enum value specifying which device ID generation strategy  should use: OpenUDID or Google Advertising ID.
     */
    protected @Nullable DeviceIdType idMode = null;

    /**
     * sets the limit after how many sessions, for each apps version, the automatic star rating dialog is shown.
     */
    protected int starRatingSessionLimit = 5;

    /**
     * the callback function that will be called from the automatic star rating dialog.
     */
    protected @Nullable StarRatingCallback starRatingCallback = null;

    /**
     * the shown title text for the star rating dialogs.
     */
    protected @Nullable String starRatingTextTitle = null;

    /**
     * the shown message text for the star rating dialogs.
     */
    protected @Nullable String starRatingTextMessage = null;

    /**
     * the shown dismiss button text for the shown star rating dialogs.
     */
    protected @Nullable String starRatingTextDismiss = null;

    protected boolean loggingEnabled = false;

    protected boolean enableUnhandledCrashReporting = false;

    protected boolean enableInAppMessaging = false;
    protected boolean inAppMessagingActivityAndFragmentAutoTracking = true;
    protected double sdkLocalStorageMaxFileSize = -1.0;
    protected boolean autoTrackingUseShortName = true;

    protected Class[] autoTrackingExceptions = null;

    protected @Nullable Map<String, Object> automaticViewSegmentation = null;

    protected @Nullable Map<String, String> customNetworkRequestHeaders = null;

    protected boolean pushIntentAddMetadata = false;


    protected boolean shouldRequireConsent = true;
    protected String[] enabledFeatureNames = {
            CoreProxy.SdkFeatureNames.push,
            CoreProxy.SdkFeatureNames.events,
            CoreProxy.SdkFeatureNames.sessions,
            CoreProxy.SdkFeatureNames.crashes,
            CoreProxy.SdkFeatureNames.iam
    };

    protected boolean temporaryDeviceIdEnabled = false;

    protected @Nullable String tamperingProtectionSalt = null;

    protected @Nullable Integer eventQueueSizeThreshold = null;

    protected boolean trackOrientationChange = false;

    protected boolean manualSessionControlEnabled = false;

    protected boolean recordAllThreadsWithCrash = false;

    protected boolean disableUpdateSessionRequests = false;

    protected boolean shouldIgnoreAppCrawlers = false;

    protected @Nullable String[] appCrawlerNames = null;

    protected @Nullable String[] publicKeyPinningCertificates = null;

    protected @Nullable String[] certificatePinningCertificates = null;

    protected @Nullable Boolean enableAttribution = null;

    protected @Nullable Map<String, Object> customCrashSegment = null;

    protected @Nullable Integer sessionUpdateTimerDelay = null;

    protected CrashFilterCallback crashFilterCallback;

    protected boolean starRatingDialogIsCancellable = false;

    protected boolean starRatingShownAutomatically = false;

    protected boolean starRatingDisableAskingForEachAppVersion = false;

    protected @Nullable Application application = null;

    protected boolean recordAppStartTime = false;
    protected boolean noneStandardPhoneValidation = false;
    protected @Nullable String overrideSDKName = null;
    protected @Nullable String overrideSDKVersion = null;
    protected @Nullable InAppMessageListener inAppListener = null;
    protected boolean collectAdvertisingId = true;
    boolean disableLocation = false;
    String locationCountyCode = null;
    String locationCity = null;
    String locationLocation = null;
    Double latitude = null;
    Double longitude = null;
    String locationIpAddress = null;
    Long appStartTimestampOverride = null;
    boolean appLoadedManualTrigger = false;
    boolean manualForegroundBackgroundTrigger = false;
    LogCallback providedLogCallback;

    public Config() {
    }

    public Config(@NonNull Application application, @NonNull String appKey) {
        setContext(application.getApplicationContext());
        setAppKey(appKey);
        setApplication(application);
    }

    /**
     * Android context.
     * Mandatory field.
     */
    public Config setContext(@NonNull Context context) {
        this.context = context;
        return this;
    }

    /**
     * app key for the application being tracked; find in the  Dashboard under Management &gt; Applications.
     * Mandatory field.
     */
    public Config setAppKey(@NonNull String appKey) {
        this.appKey = appKey;
        if (sharedPref != null) {
            sharedPref.setPreference("ir.[" + BuildConfig.FLAVOR + "].android.api.appkey", appKey);
        }

        return this;
    }

    public Config setAuthKey(@NonNull String authKey) {
        this.authKey = authKey;
        return this;
    }

    public Config setIsDemoApp(boolean isDemoApp) {
        this.isDemoApp = isDemoApp;
        return this;
    }

    public Config setServerUrl(@Nullable String serverURL) {
        if (isDemoApp && serverURL != null)
            this.serverURL = serverURL;
        else
            this.serverURL = setServerUrlBaseOnFlavorName();
        return this;
    }

    /**
     * unique ID for the device the app is running on; note that null in deviceID means that Sdk will fall back to OpenUDID, then, if it's not available, to Google Advertising ID.
     */
    public Config setDeviceId(@NonNull String deviceID) {
        this.deviceID = deviceID;
        return this;
    }

    /**
     * Set to true if you want to enable internal debugging logs
     * Those logs will be printed to the console
     */
    @NonNull
    public Config setLoggingEnabled(boolean enabled) {
        this.loggingEnabled = enabled;
        return this;
    }

    /**
     * enum value specifying which device ID generation strategy  should use: OpenUDID or Google Advertising ID.
     */
    @NonNull
    public Config setIdMode(@NonNull DeviceIdType idMode) {
        this.idMode = idMode;
        return this;
    }

    /**
     * Override validation of user's cell phone.
     */
    @NonNull
    public Config setNoneStandardPhoneValidation(@NonNull Boolean validate) {
        this.noneStandardPhoneValidation = validate;
        return this;
    }

    @NonNull
    public Config seCollectAdvertisingId(@NonNull Boolean value) {
        this.collectAdvertisingId = value;
        return this;
    }

    @NonNull
    public Config setInternalData(@NonNull String n, @NonNull String v) {
        this.overrideSDKName = n;
        this.overrideSDKVersion = v;
        return this;
    }

    @NonNull
    public Config setInAppListener(@NonNull InAppMessageListener listener) {
        this.inAppListener = listener;
        return this;
    }

    /**
     * Call to enable uncaught crash reporting
     */
    @NonNull
    public Config enableCrashReporting() {
        this.enableUnhandledCrashReporting = true;
        return this;
    }

    @NonNull
    public Config enableInAppMessaging() {
        this.enableInAppMessaging = true;
        return this;
    }

    @NonNull
    public Config enableInAppMessaging(boolean enableAutoTracking, boolean useActivityShortName) {
        this.enableInAppMessaging = true;
        this.inAppMessagingActivityAndFragmentAutoTracking = enableAutoTracking;
        this.autoTrackingUseShortName = useActivityShortName;
        return this;
    }

    @NonNull
    public Config disableInAppMessagingActivityAndFragmentAutoTracking() {
        this.inAppMessagingActivityAndFragmentAutoTracking = false;
        return this;
    }

    /**
     * limits max local storage file size to a threshold (few KiloBytes tolerance)
     * <br/>
     * NOT RECOMMENDED TO USE generally. contact our support if you need to use this feature.
     *
     * @param maxFileSizeInKiloBytes upper threshold of local storage file size , must be greater than zero!
     */
    public Config setSDKLocalStorageMaxFileSize(double maxFileSizeInKiloBytes) {
        if (maxFileSizeInKiloBytes > 0) {
            this.sdkLocalStorageMaxFileSize = maxFileSizeInKiloBytes;
        } else {
            CoreInternal.sharedInstance().L.i("[Config] sdkLocalStorageMaxFileSize can not be `0` or `negative value`, considered as not set!");
        }
        return this;
    }

    private Config setPushIntentAddMetadata(boolean enable) {
        pushIntentAddMetadata = enable;
        return this;
    }

    /**
     * Set if consent should be required
     *
     * @param shouldRequireConsent
     * @return
     */
    private Config setRequiresConsent(boolean shouldRequireConsent) {
        this.shouldRequireConsent = shouldRequireConsent;
        return this;
    }

    /**
     * Sets which features are enabled in case consent is required
     *
     * @param featureNames
     * @return
     */
    private Config setConsentEnabled(String[] featureNames) {
        enabledFeatureNames = featureNames;
        return this;
    }

    private Config enableTemporaryDeviceIdMode() {
        temporaryDeviceIdEnabled = true;
        return this;
    }

    private Config setCrashFilterCallback(CrashFilterCallback callback) {
        crashFilterCallback = callback;
        return this;
    }

    private Config setParameterTamperingProtectionSalt(String salt) {
        tamperingProtectionSalt = salt;
        return this;
    }

    private Config setTrackOrientationChanges(boolean shouldTrackOrientation) {
        trackOrientationChange = shouldTrackOrientation;
        return this;
    }

    private Config setRecordAllThreadsWithCrash() {
        recordAllThreadsWithCrash = true;
        return this;
    }

    private Config setEnableAttribution(boolean enableAttribution) {
        this.enableAttribution = enableAttribution;
        return this;
    }

    private Config enablePublicKeyPinning(String[] certificates) {
        publicKeyPinningCertificates = certificates;
        return this;
    }

    private Config enableCertificatePinning(String[] certificates) {
        certificatePinningCertificates = certificates;
        return this;
    }

    private Config setShouldIgnoreAppCrawlers(boolean shouldIgnore) {
        shouldIgnoreAppCrawlers = shouldIgnore;
        return this;
    }

    private Config setAppCrawlerNames(String[] appCrawlerNames) {
        this.appCrawlerNames = appCrawlerNames;
        return this;
    }

    /**
     * Set the threshold for event grouping. Event count that is bellow the
     * threshold will be send on update ticks.
     *
     * @param threshold
     * @return
     */
    private Config setEventQueueSizeToSend(int threshold) {
        eventQueueSizeThreshold = threshold;
        return this;
    }

    private Config enableManualSessionControl() {
        manualSessionControlEnabled = true;
        return this;
    }

    private Config setCustomCrashSegment(Map<String, Object> crashSegment) {
        customCrashSegment = crashSegment;
        return this;
    }

    protected Config checkForNativeCrashDumps(boolean checkForDumps) {
        checkForNativeCrashDumps = checkForDumps;
        return this;
    }

    /**
     * Sets the interval for the automatic session update calls
     * min value 1 (1 second),
     * max value 600 (10 minutes)
     *
     * @param delay in seconds
     * @return
     */
    private Config setUpdateSessionTimerDelay(int delay) {
        sessionUpdateTimerDelay = delay;
        return this;
    }

    protected Config setSharedPref(SharedPref store) {
        sharedPref = store;
        if (appKey != null) {
            sharedPref.setPreference("ir.[" + BuildConfig.FLAVOR + "].android.api.appkey", appKey);
        }

        return this;
    }

    protected Config setDisableUpdateSessionRequests(boolean disable) {
        disableUpdateSessionRequests = disable;
        return this;
    }

    /**
     * Set if the star rating dialog is cancellable
     *
     * @param isCancellable set this true if it should be cancellable
     */
    private synchronized Config setIfStarRatingDialogIsCancellable(boolean isCancellable) {
        starRatingDialogIsCancellable = isCancellable;
        return this;
    }

    /**
     * Set if the star rating should be shown automatically
     *
     * @param isShownAutomatically set it true if you want to show the app star rating dialog automatically for each new version after the specified session amount
     */
    private synchronized Config setIfStarRatingShownAutomatically(boolean isShownAutomatically) {
        starRatingShownAutomatically = isShownAutomatically;
        return this;
    }

    /**
     * Set if the star rating is shown only once per app lifetime
     *
     * @param disableAsking set true if you want to disable asking the app rating for each new app version (show it only once per apps lifetime)
     */
    private synchronized Config setStarRatingDisableAskingForEachAppVersion(boolean disableAsking) {
        starRatingDisableAskingForEachAppVersion = disableAsking;
        return this;
    }

    /**
     * Set the link to the application class
     *
     * @param application
     * @return
     */
    public synchronized Config setApplication(@NonNull Application application) {
        this.application = application;
        return this;
    }


    /**
     * Enable the recording of the app start time
     *
     * @param recordAppStartTime
     * @return
     */
    private synchronized Config setRecordAppStartTime(boolean recordAppStartTime) {
        this.recordAppStartTime = recordAppStartTime;
        return this;
    }

    /**
     * Disable location tracking
     *
     * @return
     */
    private synchronized Config setDisableLocation() {
        disableLocation = true;
        return this;
    }

    /**
     * Set location parameters.
     * This will be ignored if set together with `setDisableLocation`
     *
     * @param country_code   ISO Country code for the user's country
     * @param city           Name of the user's city
     * @param gpsCoordinates comma separate lat and lng values. For example, "56.42345,123.45325"
     */
    private synchronized Config setLocation(String country_code, String city, String gpsCoordinates, String ipAddress) {
        locationCountyCode = country_code;
        locationCity = city;
        locationLocation = gpsCoordinates;
        locationIpAddress = ipAddress;
        return this;
    }

    public synchronized Config setLatitude(@Nullable Double latitude) {
        this.latitude = latitude;
        return this;
    }

    public synchronized Config setLongitude(@Nullable Double longitude) {
        this.longitude = longitude;
        return this;
    }

    /**
     * Override the app start timestamp in case you have a more precise way to measure it
     */
    private synchronized Config setAppStartTimestampOverride(long appStartTimestampOverride) {
        this.appStartTimestampOverride = appStartTimestampOverride;
        return this;
    }

    /**
     * Set to manually trigger the moment when the app has finished loading
     */
    private synchronized Config enableManualAppLoadedTrigger() {
        appLoadedManualTrigger = true;
        return this;
    }

    /**
     * Set this in case you want to control these triggers manually
     */
    private synchronized Config enableManualForegroundBackgroundTriggerAPM() {
        manualForegroundBackgroundTrigger = true;
        return this;
    }

    /**
     * Add a log callback that will duplicate all logs done by the SDK.
     * For each message you will receive the message string and it's targeted log level.
     */
    @NonNull
    public synchronized Config setLogListener(@NonNull LogCallback logCallback) {
        providedLogCallback = logCallback;
        return this;
    }


    private String setServerUrlBaseOnFlavorName() {
        if (BuildConfig.DEBUG)
            return "https://stage-api.intrack.ir";
        else {
            return "https://api.intrack.ir";
        }
    }
}
