/*
Copyright (c) 2012, 2013, 2014 ST.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package sdk.main.core;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;

/**
 * ConnectionProcessor is a Runnable that is executed on a background
 * thread to submit session &amp; event data to a server.
 * <p>
 * NOTE: This class is only public to facilitate unit testing, because
 * of this bug in dexmaker: https://code.google.com/p/dexmaker/issues/detail?id=34
 */
class ConnectionProcessor implements Runnable {
    private static final int CONNECT_TIMEOUT_IN_MILLISECONDS = 30000;
    private static final int READ_TIMEOUT_IN_MILLISECONDS = 30000;

    private final SharedPref store_;
    private final DeviceId deviceId_;
    private final String serverURL_;
    private final SSLContext sslContext_;

    private final Map<String, String> requestHeaderCustomValues_;

    ModuleLog L;

    ConnectionProcessor(final String serverURL, final SharedPref store, final DeviceId deviceId, final SSLContext sslContext, final Map<String, String> requestHeaderCustomValues, ModuleLog logModule) {
        serverURL_ = serverURL;
        store_ = store;
        deviceId_ = deviceId;
        sslContext_ = sslContext;
        requestHeaderCustomValues_ = requestHeaderCustomValues;
        L = logModule;
    }

    synchronized public URLConnection urlConnectionForServerRequest(String requestData, final String customEndpoint) throws IOException {
        String urlEndpoint = "/api/sdk/accounts/android";
        if (customEndpoint != null) {
            urlEndpoint = customEndpoint;
        }

        String urlStr = serverURL_ + urlEndpoint;
        requestData += "&checksum=<checksum>";
        final URL url = new URL(urlStr);
        final HttpURLConnection conn;
        if (CoreInternal.publicKeyPinCertificates == null && CoreInternal.certificatePinCertificates == null) {
            conn = (HttpURLConnection) url.openConnection();
        } else {
            HttpsURLConnection c = (HttpsURLConnection) url.openConnection();
            c.setSSLSocketFactory(sslContext_.getSocketFactory());
            conn = c;
        }
        conn.setConnectTimeout(CONNECT_TIMEOUT_IN_MILLISECONDS);
        conn.setReadTimeout(READ_TIMEOUT_IN_MILLISECONDS);
        conn.setUseCaches(false);
        conn.setDoInput(true);
        conn.setRequestMethod("GET");

        if (requestHeaderCustomValues_ != null) {
            //if there are custom header values, add them
            L.v("[Connection Processor] Adding [" + requestHeaderCustomValues_.size() + "] custom header fields");
            for (Map.Entry<String, String> entry : requestHeaderCustomValues_.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                if (key != null && value != null && !key.isEmpty()) {
                    conn.addRequestProperty(key, value);
                }
            }
        }

        String picturePath = UserData.getPicturePathFromQuery(url);
        L.v("[Connection Processor] Got picturePath: " + picturePath);
        if (urlEndpoint.contains("/pollInApp")) {
            conn.setDoOutput(true);
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/json");
//            JSONObject jsonBody = convertBodyToJson(requestData);

            JSONObject requestBody = convertBodyToJson(requestData);

            OutputStream os = conn.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(os, Charset.forName("UTF-8")));
            writer.write(requestBody.toString());
            writer.flush();
            writer.close();
            os.close();

        } else if (!picturePath.equals("")) {
            //Uploading files:
            //http://stackoverflow.com/questions/2793150/how-to-use-java-net-urlconnection-to-fire-and-handle-http-requests

            File binaryFile = new File(picturePath);
            conn.setDoOutput(true);
            // Just generate some unique random value.
            String boundary = Long.toHexString(System.currentTimeMillis());
            // Line separator required by multipart/form-data.
            String CRLF = "\r\n";
            String charset = "UTF-8";
            conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
            OutputStream output = conn.getOutputStream();
            PrintWriter writer = new PrintWriter(new OutputStreamWriter(output, charset), true);
            // Send binary file.
            writer.append("--").append(boundary).append(CRLF);
            writer.append("Content-Disposition: form-data; name=\"binaryFile\"; filename=\"").append(binaryFile.getName()).append("\"").append(CRLF);
            writer.append("Content-Type: ").append(URLConnection.guessContentTypeFromName(binaryFile.getName())).append(CRLF);
            writer.append("Content-Transfer-Encoding: binary").append(CRLF);
            writer.append(CRLF).flush();
            FileInputStream fileInputStream = new FileInputStream(binaryFile);
            byte[] buffer = new byte[1024];
            int len;
            try {
                while ((len = fileInputStream.read(buffer)) != -1) {
                    output.write(buffer, 0, len);
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
            output.flush(); // Important before continuing with writer!
            writer.append(CRLF).flush(); // CRLF is important! It indicates end of boundary.
            fileInputStream.close();

            // End of multipart/form-data.
            writer.append("--").append(boundary).append("--").append(CRLF).flush();
        } else {
            conn.setDoOutput(true);
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/json");
            JSONObject jsonBody = convertBodyToJson(requestData);
            String date = "";
            String deviceId = "";
            String appKey = "";
            String appId = "";
            try {
                date = jsonBody.getString("date");
            } catch (JSONException e) {
                e.printStackTrace();
            }
            try {
                deviceId = jsonBody.getString("deviceId");
            } catch (JSONException e) {
                e.printStackTrace();
            }
            try {
                appKey = jsonBody.getString("appKey");
            } catch (JSONException e) {
                e.printStackTrace();
            }
            try {
                JSONObject device = jsonBody.getJSONObject("device");
                appId = device.getString("appId");
            } catch (JSONException e) {
                e.printStackTrace();
            }
            String cs = SdkChecksumGeneratorUtil.Companion.checksum(jsonBody.toString(), date, deviceId, appKey, appId);
            String bodyString = jsonBody.toString().replace("<checksum>", cs);
            if (L.logEnabled()) {
                L.i(bodyString);
            }
            JSONObject requestBody = new JSONObject();
            OutputStream os = conn.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(os, Charset.forName("UTF-8")));
            try {
                requestBody.put("data", LZString.compressToBase64(bodyString));
                writer.write(requestBody.toString());
            } catch (JSONException ignored) {
            }
            writer.flush();
            writer.close();
            os.close();
        }
        return conn;
    }

    private JSONObject convertBodyToJson(String requestData) {
        JSONObject body = new JSONObject();

        for (String keyValuePair : requestData.split("&")) {
            String[] keyValueArray = keyValuePair.split("=");
            if (keyValueArray.length != 2) {
                continue;
            }
            String key = UtilsNetworking.urlDecodeString(keyValueArray[0]);
            String value = UtilsNetworking.urlDecodeString(keyValueArray[1]);

            try {
                if (value.equals("true") || value.equals("false")) {
                    body.put(key, value.equals("true"));
                    continue;
                }

                if (key.equals("deviceId")) {
                    body.put(key, value);
                    continue;
                }

                try {
                    body.put(key, Double.parseDouble(value));
                    continue;
                } catch (NumberFormatException ignored) {
                }

                try {
                    body.put(key, Integer.parseInt(value));
                    continue;
                } catch (NumberFormatException ignored) {
                }

                try {
                    body.put(key, Long.parseLong(value));
                    continue;
                } catch (NumberFormatException ignored) {
                }

                try {
                    body.put(key, new JSONObject(value));
                    continue;
                } catch (JSONException ignored) {
                }

                try {
                    body.put(key, new JSONArray(value));
                    continue;
                } catch (JSONException ignored) {
                }

                body.put(key, value);

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        return body;
    }

    @Override
    public void run() {
        while (true) {
            final String[] storedEvents = store_.connections();
            int storedEventCount = storedEvents == null ? 0 : storedEvents.length;

            if (L.logEnabled()) {
                String msg = "[Connection Processor] Starting to run, there are [" + storedEventCount + "] requests stored";
                if (storedEventCount == 0) {
                    L.v(msg);
                } else {
                    L.i(msg);
                }
            }

            if (storedEvents == null || storedEventCount == 0) {
                // currently no data to send, we are done for now
                break;
            }

            // get first event from collection
            if (deviceId_.getId() == null || deviceId_.getId().isEmpty()) {
                // When device ID is supplied by OpenUDID or by Google Advertising ID.
                // In some cases it might take time for them to initialize. So, just wait for it.
                L.i("[Connection Processor] No Device ID available yet, skipping request " + storedEvents[0]);
                break;
            }

            String temporaryIdOverrideTag = "&override_id=" + DeviceId.temporaryInTrackDeviceId;
            String temporaryIdTag = "&deviceId=" + DeviceId.temporaryInTrackDeviceId;
            boolean containsTemporaryIdOverride = storedEvents[0].contains(temporaryIdOverrideTag);
            boolean containsTemporaryId = storedEvents[0].contains(temporaryIdTag);
            if (containsTemporaryIdOverride || containsTemporaryId || deviceId_.temporaryIdModeEnabled()) {
                //we are about to change ID to the temporary one or
                //the internally set id is the temporary one

                //abort and wait for exiting temporary mode
                L.i("[Connection Processor] Temporary ID detected, stalling requests. Id override:[" + containsTemporaryIdOverride + "], tmp id tag:[" + containsTemporaryId + "], temp ID set:[" + deviceId_.temporaryIdModeEnabled() + "]");
                break;
            }

            boolean deviceIdOverride = storedEvents[0].contains("&override_id="); //if the sendable data contains a override tag
            boolean deviceIdChange = storedEvents[0].contains("&deviceId="); //if the sendable data contains a device_id tag. In this case it means that we will have to change the stored device ID

            //add the device_id to the created request
            final String eventData, newId;
            if (deviceIdOverride) {
                // if the override tag is used, it means that the device_id will be changed
                // to finish the session of the previous device_id, we have cache it into the request
                // this is indicated by having the "override_id" tag. This just means that we
                // don't use the id provided in the deviceId variable as this might have changed already.

                eventData = storedEvents[0].replace("&override_id=", "&deviceId=");
                newId = null;
            } else {
                if (deviceIdChange) {
                    // this branch will be used if a new device_id is provided
                    // and a device_id merge on server has to be performed

                    final int endOfDeviceIdTag = storedEvents[0].indexOf("&deviceId=") + "&deviceId=".length();
                    newId = UtilsNetworking.urlDecodeString(storedEvents[0].substring(endOfDeviceIdTag));

                    if (newId.equals(deviceId_.getId())) {
                        // If the new device_id is the same as previous,
                        // we don't do anything to change it

                        eventData = storedEvents[0];
                        deviceIdChange = false;

                        L.d("[Connection Processor] Provided device_id is the same as the previous one used, nothing will be merged");
                    } else {
                        //new device_id provided, make sure it will be merged
                        eventData = storedEvents[0] + "&old_device_id=" + UtilsNetworking.urlEncodeString(deviceId_.getId());
                    }
                } else {
                    // this branch will be used in almost all requests.
                    // This just adds the device_id to them

                    newId = null;
                    eventData = storedEvents[0] + "&deviceId=" + UtilsNetworking.urlEncodeString(deviceId_.getId());
                }
            }

            if (!(CoreInternal.sharedInstance().isDeviceAppCrawler() && CoreInternal.sharedInstance().ifShouldIgnoreCrawlers())) {
                //continue with sending the request to the server
                URLConnection conn = null;
                InputStream connInputStream = null;
                try {
                    // initialize and open connection
                    conn = urlConnectionForServerRequest(eventData, null);
                    conn.connect();

                    int responseCode = 0;
                    String responseString = "";
                    if (conn instanceof HttpURLConnection) {
                        final HttpURLConnection httpConn = (HttpURLConnection) conn;

                        try {
                            //assume there will be no error
                            connInputStream = httpConn.getInputStream();
                        } catch (Exception ex) {
                            //in case of exception, assume there was a error in the request and change streams
                            connInputStream = httpConn.getErrorStream();
                        }

                        responseCode = httpConn.getResponseCode();
                        responseString = Utils.inputStreamToString(connInputStream);
                    }

                    L.d("[Connection Processor] code:[" + responseCode + "], response:[" + responseString + "]");

                    final RequestResult rRes;

                    if (responseCode >= 200 && responseCode < 300) {

                        if (responseString.isEmpty()) {
                            L.v("[Connection Processor] Response was empty, will retry");
                            rRes = RequestResult.RETRY;
                        } else {
                            JSONObject jsonObject;
                            try {
                                jsonObject = new JSONObject(responseString);
                            } catch (JSONException ex) {
                                //failed to parse, so not a valid json
                                jsonObject = null;
                            }

                            if (jsonObject == null) {
                                //received unparseable response, retrying
                                L.v("[Connection Processor] Response was a unknown, will retry");
                                rRes = RequestResult.RETRY;
                            } else {
                                if (jsonObject.has("result")) {
                                    //contains result entry
                                    L.v("[Connection Processor] Response was a success");
                                    rRes = RequestResult.OK;
                                } else {
                                    L.v("[Connection Processor] Response does not contain 'result', will retry");
                                    rRes = RequestResult.RETRY;
                                }
                            }
                        }
                    } else if (responseCode >= 300 && responseCode < 400) {
                        //assume redirect
                        L.d("[Connection Processor] Encountered redirect, will retry");
                        rRes = RequestResult.RETRY;
                    } else if (responseCode == 400) {
                        rRes = RequestResult.REMOVE;
                        L.w("[Connection Processor] Bad request, will not retry");
                        L.sendLog("[Connection Processor] Bad request [status: " + responseCode + "] event data: [" + eventData + "]");
                    } else if (responseCode == 404) {
                        L.w("[Connection Processor] Bad request (not found), will still retry");
                        L.sendLog("[Connection Processor] Bad request [status: " + responseCode + "] (not found)");
                        rRes = RequestResult.RETRY;
                    } else if (responseCode > 400) {
                        //server down, try again later
                        L.d("[Connection Processor] Server is down, will retry");
                        L.sendLog("[Connection Processor] Server is down [status: " + responseCode + "]");
                        rRes = RequestResult.RETRY;
                    } else {
                        L.d("[Connection Processor] Bad response code, will retry");
                        L.sendLog("[Connection Processor] Bad response code [status: " + responseCode + "] for event data: [" + eventData + "]");
                        rRes = RequestResult.RETRY;
                    }

                    // an 'if' needs to be used here so that a 'switch' statement does not 'eat' the 'break' call
                    // that is used to get out of the request loop
                    if (rRes == RequestResult.OK) {
                        // successfully submitted event data to server, so remove
                        // this one from the stored events collection
                        store_.removeConnection(storedEvents[0]);

                        if (deviceIdChange) {
                            deviceId_.changeToDeveloperProvidedId(store_, newId);
                        }

                        if (deviceIdChange || deviceIdOverride) {
                            CoreInternal.sharedInstance().notifyDeviceIdChange();
                        }
                    } else if (rRes == RequestResult.REMOVE) {
                        //bad request, will be removed
                        store_.removeConnection(storedEvents[0]);
                    } else if (rRes == RequestResult.RETRY) {
                        // warning was logged above, stop processing, let next tick take care of retrying
                        break;
                    }
                } catch (Exception e) {
                    L.w("[Connection Processor] Got exception while trying to submit event data: [" + eventData + "] [" + e + "]");
                    L.sendLog("[Connection Processor] Got exception while trying to submit event data: [" + eventData + "] [" + e + "]");
                    // if exception occurred, stop processing, let next tick take care of retrying
                    break;
                } finally {
                    // free connection resources
                    if (conn instanceof HttpURLConnection) {
                        try {
                            if (connInputStream != null) {
                                connInputStream.close();
                            }
                        } catch (Throwable ignored) {
                        }

                        ((HttpURLConnection) conn).disconnect();
                    }
                }
            } else {
                //device is identified as a app crawler and nothing is sent to the server
                L.i("[Connection Processor] Device identified as a app crawler, skipping request " + storedEvents[0]);

                //remove stored data
                store_.removeConnection(storedEvents[0]);
            }
        }
    }

    // for unit testing
    String getServerURL() {
        return serverURL_;
    }

    SharedPref getSharedPref() {
        return store_;
    }

    DeviceId getDeviceId() {
        return deviceId_;
    }

    private enum RequestResult {
        OK,         // success
        RETRY,      // retry MAX_RETRIES_BEFORE_SLEEP before switching to SLEEP
        REMOVE      // bad request, remove
    }
}
