package sdk.main.core;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.WindowManager;

import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

/**
 * This class provides several static methods to retrieve information about
 * the current device and operating environment.
 */
class DeviceInfo {

    static String deepLink;
    private static String pushKey = null;
    private static String campaignName = null; //utm_campaign
    private static String campaignMedium = null; //utm_medium
    private static String campaignSource = null; //utm_source
    private static String campaignTerm = null; //utm_term
    private static String campaignContent = null; //utm_content
    private static CoreProxy.MessagingProvider pushProvider = null;

    public static final String APP_UPDATED_EVENT_KEY = "app_updated";

    public static void setPushKey(String pushKeyIn) {
        pushKey = pushKeyIn;
    }

    public static void setCampaignName(String campaignNameIn) {
        campaignName = campaignNameIn;
    }

    public static void setCampaignMedium(String campaignMediumIn) {
        campaignMedium = campaignMediumIn;
    }

    public static void setCampaignSource(String campaignSourceIn) {
        campaignSource = campaignSourceIn;
    }

    public static void setCampaignTerm(String campaignTermIn) {
        campaignTerm = campaignTermIn;
    }

    public static void setCampaignContent(String campaignContentIn) {
        campaignContent = campaignContentIn;
    }

    public static void setPushProvider(CoreProxy.MessagingProvider pushProviderIn) {
        pushProvider = pushProviderIn;
    }

    static long getSdkVersionCode(Context context) {
        long versionCode = -1;
        try {
            PackageManager packageManager = context.getPackageManager();
            PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                versionCode = packageInfo.getLongVersionCode();
            } else {
                versionCode = packageInfo.versionCode;
            }
        } catch (PackageManager.NameNotFoundException ignored) {
        }
        return versionCode;
    }

    /**
     * Returns the display name of the current operating system.
     */
    @SuppressWarnings("SameReturnValue")
    static String getOS() {
        return "Android";
    }

    /**
     * Returns the current operating system version as a displayable string.
     */
    @SuppressWarnings("SameReturnValue")
    static String getOSVersion() {
        return android.os.Build.VERSION.RELEASE;
    }

    static long getOSVersionCode() {
        return Build.VERSION.SDK_INT;
    }

    /**
     * Returns the current device model.
     */
    @SuppressWarnings("SameReturnValue")
    static String getDeviceModel() {
        return android.os.Build.MODEL;
    }

    static String getManufacturer() {
        return Build.MANUFACTURER;
    }

    /**
     * Returns the non-scaled pixel resolution of the current default display being used by the
     * WindowManager in the specified context.
     *
     * @param context context to use to retrieve the current WindowManager
     * @return a string in the format "WxH", or the empty string "" if resolution cannot be determined
     */
    static String getResolution(final Context context) {
        // user reported NPE in this method; that means either getSystemService or getDefaultDisplay
        // were returning null, even though the documentation doesn't say they should do so; so now
        // we catch Throwable and return empty string if that happens
        String resolution = "";
        try {
            final WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            final Display display = wm.getDefaultDisplay();
            final DisplayMetrics metrics = new DisplayMetrics();
            display.getMetrics(metrics);
            resolution = metrics.widthPixels + "x" + metrics.heightPixels;
        } catch (Throwable t) {
            CoreInternal.sharedInstance().L.i("[DeviceInfo] Device resolution cannot be determined");
        }
        return resolution;
    }

    /**
     * Maps the current display density to a string constant.
     *
     * @param context context to use to retrieve the current display metrics
     * @return a string constant representing the current display density, or the
     * empty string if the density is unknown
     */
    static String getDensity(final Context context) {
        String densityStr;
        final int density = context.getResources().getDisplayMetrics().densityDpi;
        switch (density) {
            case DisplayMetrics.DENSITY_LOW:
                densityStr = "LDPI";
                break;
            case DisplayMetrics.DENSITY_MEDIUM:
                densityStr = "MDPI";
                break;
            case DisplayMetrics.DENSITY_TV:
                densityStr = "TVDPI";
                break;
            case DisplayMetrics.DENSITY_HIGH:
                densityStr = "HDPI";
                break;
            case DisplayMetrics.DENSITY_260:
            case DisplayMetrics.DENSITY_280:
            case DisplayMetrics.DENSITY_300:
            case DisplayMetrics.DENSITY_XHIGH:
                densityStr = "XHDPI";
                break;
            case DisplayMetrics.DENSITY_340:
            case DisplayMetrics.DENSITY_360:
            case DisplayMetrics.DENSITY_400:
            case DisplayMetrics.DENSITY_420:
            case DisplayMetrics.DENSITY_XXHIGH:
                densityStr = "XXHDPI";
                break;
            case DisplayMetrics.DENSITY_560:
            case DisplayMetrics.DENSITY_XXXHIGH:
                densityStr = "XXXHDPI";
                break;
            default:
                densityStr = "other";
                break;
        }
        return densityStr;
    }

    /**
     * Returns the display name of the current network operator from the
     * TelephonyManager from the specified context.
     *
     * @param context context to use to retrieve the TelephonyManager from
     * @return jsonArray containing names of network operators, currently only one carrier is read
     */
    static JSONArray getCarriers(final Context context) {
        JSONArray carriers = new JSONArray();
        final TelephonyManager manager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        if (manager != null && manager.getNetworkOperatorName() != null) {
            carriers.put(UtilsNetworking.urlEncodeString(manager.getNetworkOperatorName()));
        }
        return carriers;
    }

    static String getTimezone() {
        return TimeZone.getDefault().getID();
    }

    /**
     * Returns the current locale (ex. "en_US").
     */
    static String getLocale() {
        final Locale locale = Locale.getDefault();
        return locale.getLanguage() + "_" + locale.getCountry();
    }

    /**
     * Returns the application version string stored in the specified
     * context's package info versionName field, or "1.0" if versionName
     * is not present.
     */
    static String getAppVersion(final Context context) {
        String result = CoreInternal.DEFAULT_APP_VERSION;
        try {
            String tmpVersion = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName;
            if (tmpVersion != null) {
                result = tmpVersion;
                CoreInternal.sharedInstance().L.i("[DeviceInfo] app version name = " + result);
            }
        } catch (PackageManager.NameNotFoundException e) {
            CoreInternal.sharedInstance().L.i("[DeviceInfo] No app version name found");
        }
        return result;
    }

    public static String getSavedAppVersionName() {
        String result = null;
        try {
            String savedVersionName = CoreInternal.sharedInstance().config_.appInfoPrefs.getLatestAppVersionName();
            if (savedVersionName != null && !savedVersionName.isEmpty()) {
                result = savedVersionName;
                CoreInternal.sharedInstance().L.i("[DeviceInfo] saved app version name = " + result);
            }
        } catch (Exception ignored) {
            CoreInternal.sharedInstance().L.i("[DeviceInfo] No saved app version name found");
        }
        return result;
    }

    static int getAppVersionCode(final Context context) {
        int result = -1;
        try {
            int versionCode = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode;
            if (versionCode != -1) {
                result = versionCode;
                CoreInternal.sharedInstance().L.i("[DeviceInfo] app version code = " + result);
            }
        } catch (PackageManager.NameNotFoundException e) {
            CoreInternal.sharedInstance().L.i("[DeviceInfo] No app version code found");
        }
        return result;
    }

    public static int getSavedAppVersionCode() {
        int result = -1;
        try {
            int versionCode = CoreInternal.sharedInstance().config_.appInfoPrefs.getLatestAppVersionCode();
            if (versionCode != -1) {
                result = versionCode;
                CoreInternal.sharedInstance().L.i("[DeviceInfo] saved app version code = " + result);
            }
        } catch (Exception ignored) {
            CoreInternal.sharedInstance().L.i("[DeviceInfo] No saved app version code found");
        }
        return result;
    }

    @Nullable
    public static Map<String, Object> checkAppUpdated(Context context) {
        final String savedAppVersionName = getSavedAppVersionName();
        final int savedAppVersionCode = getSavedAppVersionCode();
        final String currentVersionName = getAppVersion(context);
        final int currentVersionCode = getAppVersionCode(context);

        if (savedAppVersionCode == -1) {
            // First launch or new install
            CoreInternal.sharedInstance().config_.appInfoPrefs.setLatestAppVersionCode(currentVersionCode);
            CoreInternal.sharedInstance().config_.appInfoPrefs.setLatestAppVersionName(currentVersionName);
            return null;
        } else if (savedAppVersionCode < currentVersionCode) {
            // Update detected
            CoreInternal.sharedInstance().config_.appInfoPrefs.setLatestAppVersionCode(currentVersionCode);
            CoreInternal.sharedInstance().config_.appInfoPrefs.setLatestAppVersionName(currentVersionName);

            return new HashMap<String, Object>() {{
                put("version", currentVersionName);
                put("build", currentVersionCode);
                put("previousVersion", savedAppVersionName);
                put("previousBuild", savedAppVersionCode);
            }};
        } else {
            // No change (same version as last launch)
            return null;
        }
    }

    /**
     * Returns the package name of the app that installed this app
     */
    static String getStore(final Context context) {
        String result = "";
        try {
            result = context.getPackageManager().getInstallerPackageName(context.getPackageName());
        } catch (Exception e) {
            CoreInternal.sharedInstance().L.d("[DeviceInfo, getStore] Can't get Installer package ");
        }
        if (result == null || result.length() == 0) {
            result = "";
            CoreInternal.sharedInstance().L.d("[DeviceInfo, getStore] No store found");
        }
        return result;
    }

    /**
     * Returns what kind of device this is. The potential values are:
     * ["console", "mobile", "tablet", "smarttv", "wearable", "embedded", "desktop"]
     * Currently the Android SDK differentiates between ["mobile", "tablet", "smarttv"]
     */
    static String getDeviceType(final Context context) {
        if (Utils.isDeviceTv(context)) {
            return "smarttv";
        }

        if (Utils.isDeviceTablet(context)) {
            return "tablet";
        }

        return "mobile";
    }

    static String getDeviceBrand() {
        return Build.BRAND;
    }

    static String getAppId(Context context) {
        return context.getPackageName();
    }

    static String getApiVersion() {
        return String.valueOf(Build.VERSION.SDK_INT);
    }

    static String getAppInstallTime(Context context) {
        try {
            return String.valueOf(context.getPackageManager().getPackageInfo(context.getPackageName(), 0).firstInstallTime);
        } catch (PackageManager.NameNotFoundException e) {
            return null;
        }
    }

    static Map<String, Object> getPushMetrics() {

        Map<String, Object> result = new HashMap<>();

        putValue(result, "pushKey", pushKey);
        putValue(result, "pushProvider", pushProvider != null ? pushProvider.name() : null);

        return result;
    }

    static boolean getInAppEnabled() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && CoreInternal.sharedInstance().config_.enableInAppMessaging;
    }

    static Map<String, Object> getDeviceMetrics(final Context context) {

        Map<String, Object> result = new HashMap<>();

        if (CoreInternal.sharedInstance().config_.collectAdvertisingId) {
            putValue(result, "advertisingId", CoreInternal.sharedInstance().config_.sharedPref.getCachedAdvertisingId());
        }
        putValue(result, "deviceManufacturer", getManufacturer());
        putValue(result, "deviceBrand", getDeviceBrand());
        putValue(result, "deviceModel", getDeviceModel());
        putValue(result, "carrier", getCarriers(context));
        putValue(result, "os", getOS());
        putValue(result, "androidId", null);
        putValue(result, "appId", getAppId(context));
        putValue(result, "appVersion", getAppVersion(context));
        putValue(result, "apiVersion", getApiVersion());
        putValue(result, "vendorId", null);
        putValue(result, "appInstalledDate", getAppInstallTime(context));
        putValue(result, "locale", getLocale());
        putValue(result, "latitude", CoreInternal.sharedInstance().config_.latitude);
        putValue(result, "longitude", CoreInternal.sharedInstance().config_.longitude);

        putValue(result, "sdkVersion", CoreInternal.sharedInstance().INTRACK_SDK_VERSION_STRING);
        putValue(result, "sdkName", CoreInternal.sharedInstance().INTRACK_SDK_NAME);
        putValue(result, "osVersion", getOSVersion());
        putValue(result, "resolution", getResolution(context));
        putValue(result, "density", getDensity(context));
        putValue(result, "store", getStore(context));
        putValue(result, "deepLink", deepLink);
        putValue(result, "deviceType", getDeviceType(context));

        putValue(result, "campaignMedium", campaignMedium);
        putValue(result, "campaignName", campaignName);
        putValue(result, "campaignSource", campaignSource);
        putValue(result, "campaignTerm", campaignTerm);
        putValue(result, "campaignContent", campaignContent);

        putValue(result, "acquisitionSource", null);
        putValue(result, "referrerHost", null);
        putValue(result, "referrerUrl", null);

        putValue(result, "inAppEnabled", getInAppEnabled());

        return result;
    }

    private static void putValue(Map<String, Object> map, String key, Object object) {
        if (object != null) {
            map.put(key, object);
        }
    }

    /**
     * Utility method to fill JSONObject with supplied objects for supplied keys.
     * Fills json only with non-null and non-empty key/value pairs.
     *
     * @param json    JSONObject to fill
     * @param objects varargs of this kind: key1, value1, key2, value2, ...
     */
    static void fillJSONIfValuesNotEmpty(final JSONObject json, final String... objects) {
        try {
            if (objects.length > 0 && objects.length % 2 == 0) {
                for (int i = 0; i < objects.length; i += 2) {
                    final String key = objects[i];
                    final String value = objects[i + 1];
                    if (value != null && value.length() > 0) {
                        json.put(key, value);
                    }
                }
            }
        } catch (JSONException ignored) {
            // shouldn't ever happen when putting String objects into a JSONObject,
            // it can only happen when putting NaN or INFINITE doubles or floats into it
        }
    }
}
