/*
Copyright (c) 2012, 2013, 2014 ST.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package sdk.main.core;

import android.os.Build;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

/**
 * This class holds the data for a single custom event instance.
 * It also knows how to read & write itself to the custom event JSON syntax.
 */
class Event {
    private static final String EVENT_DATA_KEY = "eventData";
    private static final String KEY_KEY = "eventName";
    private static final String TIMESTAMP_KEY = "eventTime";

    public String key;
    public long timestamp;

    // Event data
    public Map<String, String> segmentation;
    public Map<String, Integer> segmentationInt;
    public Map<String, Double> segmentationDouble;
    public Map<String, Boolean> segmentationBoolean;
    public Map<String, Long> segmentationLong;
    public Map<String, JSONObject> segmentationObject;
    public Map<String, JSONArray> segmentationArray;

    // User details
    public UserDetails userDetails;


    Event(String key) {
        UtilsTime.Instant instant = UtilsTime.getCurrentInstant();

        this.key = key;
        this.timestamp = CoreProxy.getCurrentTimeMs();

    }

    /**
     * Factory method to create an Event from its JSON representation.
     *
     * @param json JSON object to extract event data from
     * @return Event object built from the data in the JSON or null if the "key" value is not
     * present or the empty string, or if a JSON exception occurs
     */
    static Event fromJSON(final JSONObject json) {
        Event event = new Event(null);

        try {
            if (!json.isNull(KEY_KEY)) {
                event.key = json.getString(KEY_KEY);
            }
            try {
                event.timestamp = Utils.parseDate(json.optString(TIMESTAMP_KEY)).getTime();
            } catch (Exception ignored) {
            }

            UserDetails userDetails = new UserDetails();
            if (json.has(UserDetails.firstNameKey)) {
                userDetails.firstName = json.getString(UserDetails.firstNameKey);
            }
            if (json.has(UserDetails.lastNameKey)) {
                userDetails.lastName = json.getString(UserDetails.lastNameKey);
            }
            if (json.has(UserDetails.emailKey)) {
                userDetails.email = json.getString(UserDetails.emailKey);
            }
            if (json.has(UserDetails.phoneKey)) {
                userDetails.phone = json.getString(UserDetails.phoneKey);
            }
            if (json.has(UserDetails.countryKey)) {
                userDetails.country = json.getString(UserDetails.countryKey);
            }
            if (json.has(UserDetails.stateKey)) {
                userDetails.state = json.getString(UserDetails.stateKey);
            }
            if (json.has(UserDetails.cityKey)) {
                userDetails.city = json.getString(UserDetails.cityKey);
            }
            if (json.has(UserDetails.genderKey)) {
                userDetails.gender = ApiUserGender.valueOf(json.getString(UserDetails.genderKey).toUpperCase());
            }
            if (json.has(UserDetails.birthdayKey)) {
                userDetails.birthday = Utils.parseDate(json.getString(UserDetails.birthdayKey));
            }
            if (json.has(UserDetails.companyKey)) {
                userDetails.company = json.getString(UserDetails.companyKey);
            }
            if (json.has(UserDetails.hashedPhoneKey)) {
                userDetails.hashedPhone = json.getString(UserDetails.hashedPhoneKey);
            }
            if (json.has(UserDetails.hashedEmailKey)) {
                userDetails.hashedEmail = json.getString(UserDetails.hashedEmailKey);
            }
            if (json.has(UserDetails.smsOptInKey)) {
                userDetails.smsOptIn = json.getBoolean(UserDetails.smsOptInKey);
            }
            if (json.has(UserDetails.emailOptInKey)) {
                userDetails.emailOptIn = json.getBoolean(UserDetails.emailOptInKey);
            }
            if (json.has(UserDetails.pushOptInKey)) {
                userDetails.pushOptIn = json.getBoolean(UserDetails.pushOptInKey);
            }
            if (json.has(UserDetails.webPushOptInKey)) {
                userDetails.webPushOptIn = json.getBoolean(UserDetails.webPushOptInKey);
            }
            if (json.has(UserDetails.userIdKey)) {
                userDetails.userId = json.getString(UserDetails.userIdKey);
            }
            if (json.has(UserDetails.userAttributesKey)) {
                JSONObject attributesObject = json.getJSONObject(UserDetails.userAttributesKey);
                Iterator<String> iterator = attributesObject.keys();
                while (iterator.hasNext()) {
                    String userAttrKey = iterator.next();
                    userDetails.userAttributes.put(userAttrKey, attributesObject.get(userAttrKey));
                }
            }
            event.userDetails = userDetails;

            if (!json.isNull(EVENT_DATA_KEY)) {
                JSONObject eventData = json.getJSONObject(EVENT_DATA_KEY);

                final HashMap<String, String> segmentation = new HashMap<>();
                final HashMap<String, Integer> segmentationInt = new HashMap<>();
                final HashMap<String, Double> segmentationDouble = new HashMap<>();
                final HashMap<String, Boolean> segmentationBoolean = new HashMap<>();
                final HashMap<String, Long> segmentationLong = new HashMap<>();
                final HashMap<String, JSONObject> segmentationObject = new HashMap<>();
                final HashMap<String, JSONArray> segmentationArray = new HashMap<>();

                final Iterator<String> nameItr = eventData.keys();
                while (nameItr.hasNext()) {
                    final String key = nameItr.next();
                    if (!eventData.isNull(key)) {
                        Object obj = eventData.opt(key);

                        if (obj instanceof Double) {
                            //in case it's a double
                            segmentationDouble.put(key, eventData.getDouble(key));
                        } else if (obj instanceof Integer) {
                            //in case it's an integer
                            segmentationInt.put(key, eventData.getInt(key));
                        } else if (obj instanceof Long) {
                            //in case it's an Long
                            segmentationLong.put(key, eventData.getLong(key));
                        } else if (obj instanceof Boolean) {
                            //in case it's a boolean
                            segmentationBoolean.put(key, eventData.getBoolean(key));
                        } else if (obj instanceof JSONObject) {
                            //in case it's a jsonObject
                            segmentationObject.put(key, eventData.getJSONObject(key));
                        } else if (obj instanceof JSONArray) {
                            //in case it's a jsonArray
                            segmentationArray.put(key, eventData.getJSONArray(key));
                        } else {
                            //assume it's String
                            segmentation.put(key, eventData.getString(key));
                        }
                    }
                }
                event.segmentation = segmentation;
                event.segmentationDouble = segmentationDouble;
                event.segmentationInt = segmentationInt;
                event.segmentationLong = segmentationLong;
                event.segmentationBoolean = segmentationBoolean;
                event.segmentationObject = segmentationObject;
                event.segmentationArray = segmentationArray;
            }
        } catch (JSONException e) {
            CoreInternal.sharedInstance().L.w("Got exception converting JSON to an Event", e);
            event = null;
        }

        return (event != null && event.key != null && event.key.length() > 0) ? event : null;
    }

    /**
     * Creates and returns a JSONObject containing the event data from this object.
     *
     * @return a JSONObject containing the event data from this object
     */
    JSONObject toJSON() {
        final JSONObject json = new JSONObject();

        try {
            json.put(KEY_KEY, key);
            json.put(TIMESTAMP_KEY, Utils.formatDate(timestamp));

            if (userDetails != null) {
                if (userDetails.firstName != null) {
                    json.put(UserDetails.firstNameKey, userDetails.firstName);
                }
                if (userDetails.lastName != null) {
                    json.put(UserDetails.lastNameKey, userDetails.lastName);
                }
                if (userDetails.email != null) {
                    json.put(UserDetails.emailKey, userDetails.email);
                }
                if (userDetails.phone != null) {
                    json.put(UserDetails.phoneKey, userDetails.phone);
                }
                if (userDetails.country != null) {
                    json.put(UserDetails.countryKey, userDetails.country);
                }
                if (userDetails.state != null) {
                    json.put(UserDetails.stateKey, userDetails.state);
                }
                if (userDetails.city != null) {
                    json.put(UserDetails.cityKey, userDetails.city);
                }
                if (userDetails.gender != null) {
                    json.put(UserDetails.genderKey, userDetails.gender.toString().toLowerCase());
                }
                if (userDetails.birthday != null) {
                    json.put(UserDetails.birthdayKey, Utils.formatDate(userDetails.birthday));
                }
                if (userDetails.company != null) {
                    json.put(UserDetails.companyKey, userDetails.company);
                }
                if (userDetails.hashedPhone != null) {
                    json.put(UserDetails.hashedPhoneKey, userDetails.hashedPhone);
                }
                if (userDetails.hashedEmail != null) {
                    json.put(UserDetails.hashedEmailKey, userDetails.hashedEmail);
                }
                if (userDetails.smsOptIn != null) {
                    json.put(UserDetails.smsOptInKey, userDetails.smsOptIn);
                }
                if (userDetails.emailOptIn != null) {
                    json.put(UserDetails.emailOptInKey, userDetails.emailOptIn);
                }
                if (userDetails.pushOptIn != null) {
                    json.put(UserDetails.pushOptInKey, userDetails.pushOptIn);
                }
                if (userDetails.webPushOptIn != null) {
                    json.put(UserDetails.webPushOptInKey, userDetails.webPushOptIn);
                }
                if (userDetails.userId != null) {
                    json.put(UserDetails.userIdKey, userDetails.userId);
                }
                if (userDetails.userAttributes != null) {
                    json.put(UserDetails.userAttributesKey, Utils.objectifyMap(userDetails.userAttributes));
                }
            }

            JSONObject eventData = new JSONObject();

            if (segmentation != null) {
                for (Map.Entry<String, String> pair : segmentation.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationInt != null) {
                for (Map.Entry<String, Integer> pair : segmentationInt.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationDouble != null) {
                for (Map.Entry<String, Double> pair : segmentationDouble.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationBoolean != null) {
                for (Map.Entry<String, Boolean> pair : segmentationBoolean.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationLong != null) {
                for (Map.Entry<String, Long> pair : segmentationLong.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationObject != null) {
                for (Map.Entry<String, JSONObject> pair : segmentationObject.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (segmentationArray != null) {
                for (Map.Entry<String, JSONArray> pair : segmentationArray.entrySet()) {
                    eventData.put(pair.getKey(), pair.getValue());
                }
            }

            if (eventData.keys().hasNext()) {
                json.put(EVENT_DATA_KEY, eventData);
            }

        } catch (JSONException e) {
            CoreInternal.sharedInstance().L.w("Got exception converting an Event to JSON", e);
        }

        return json;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == null || !(o instanceof Event)) {
            return false;
        }

        final Event e = (Event) o;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            return Objects.equals(key, e.key) &&
                timestamp == e.timestamp &&
                (Objects.equals(segmentation, e.segmentation));
        } else {
            return key.hashCode() == e.key.hashCode() &&
                timestamp == e.timestamp &&
                ((segmentation == null && e.segmentation == null) || segmentation.hashCode() == e.segmentation.hashCode());
        }
    }

    @Override
    public int hashCode() {
        return (key != null ? key.hashCode() : 1) ^
            (segmentation != null ? segmentation.hashCode() : 1) ^
            (timestamp != 0 ? (int) timestamp : 1);
    }
}
