package sdk.main.core

import androidx.annotation.Keep
import sdk.main.core.InAppMessageListener.ErrorMessageReason
import sdk.main.core.inappmessaging.model.message.IAMMessage

/**
 * Listener interface to be notified of in app messaging events. Use to register your listener.
 *
 * This works as follows
 *
 *  * [InAppMessageListener.onMessageTrigger] is called before the message is rendered. The method
 * is called repeatedly for a message if it needs to be re-rendered during activity
 * transitions
 *  * [InAppMessageListener.onMessageClick] is called when a message with a configured action is
 * clicked. If the clicked message does not have a configured action, it is dismissed and
 * [InAppMessageListener.onMessageDismiss] is invoked
 *  * Called when the message is dismissed either automatically after a timeout or by the user or
 * when a clicked message has no associated action
 *  * [InAppMessageListener.onError] is called when any part faces an error. the error
 * then is reported through [ErrorMessageReason] enum to indicate it's type. if there is additional data (like message ID) is provided
 * it'll reported as a nullable String message.
 *
 * @hide
 */
@Keep
interface InAppMessageListener {
    fun onMessageTrigger(details: IAMMessage?)

    fun onMessageClick(details: IAMMessage?)

    fun onMessageDismiss(details: IAMMessage?)

    fun onError(reason: ErrorMessageReason?, message: String?)

    enum class ErrorMessageReason {
        // when the message type does not fit into defined binding methods
        MISSING_BINDING_ERROR,

        MESSAGE_NOT_SUPPORTED_ON_CURRENT_SDK_VERSION_ERROR,

        DEVICE_FAILED_TO_RESOLVE_INTENT_ERROR,

        // Generic error
        UNSPECIFIED_IMAGE_RENDER_ERROR,

        // Failure to fetch the image
        IMAGE_FETCH_ERROR,

        // Failure to display the image
        IMAGE_DISPLAY_ERROR,

        // Image has an unsupported format
        IMAGE_UNSUPPORTED_FORMAT,

        // mraid js is not imported
        MRAID_JS_NOT_FOUND_IN_HTML,

        // html render fail
        HTML_LOAD_FAILED,

        // Message data is invalid
        INVALID_DATA,
    }
}
