package sdk.main.core;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import ir.intrack.android.sdk.BuildConfig;

class ModuleEvents extends ModuleBase {
    public static final String SYSTEM_EVENTS_PREFIX = "[INT]_";
    public static final String DUMMY_EVENT_KEY = "dummy";
    static final Map<String, Event> timedEvents = new HashMap<>();
    static final String[] reservedSegmentationKeys = new String[]{"aaaaaaaaaaaaaaaaaaaaInTrack"};//just a test key that no one should realistically use
    //interface for SDK users
    final Events eventsInterface;
    private final HashSet<InTrackEventListener> eventListeners = new HashSet<>();
    ModuleLog L;

    ModuleEvents(CoreInternal coreInternal, Config config) {
        super(coreInternal);

        L = coreInternal.L;

        L.v("[ModuleEvents] Initialising");

        eventsInterface = new Events();
    }

    void checkCachedPushData(Context context) {
        L.d("[ModuleEvents] Starting cache call");

        Set<String> deliveryEvents = SharedPref.getCachedDeliveryEvents(context);
        Set<String> clickedEvents = SharedPref.getCachedClickedEvents(context);
        Set<String> dismissEvents = SharedPref.getCachedDismissedEvents(context);
        Set<String> rejectedEvents = SharedPref.getCachedRejectedEvents(context);
        Set<String> customChannelDelivery = SharedPref.getCachedCustomChannelDelivery(context);
        Set<String> customChannelClicked = SharedPref.getCachedCustomChannelClick(context);

        for (String eventString : deliveryEvents) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 2) {
                continue;
            }
            Map<String, Object> pushDeliveryData = new HashMap<>();
            pushDeliveryData.put("i", eventFields[0]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[1]));
            eventsInterface.recordSystemEvent(ModulePush.PUSH_DELIVERY_EVENT_KEY, pushDeliveryData, instant, null);
        }
        for (String eventString : clickedEvents) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 3) {
                continue;
            }
            Map<String, Object> pushClickData = new HashMap<>();
            pushClickData.put("i", eventFields[0]);
            pushClickData.put("b", eventFields[1]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[2]));
            eventsInterface.recordSystemEvent(ModulePush.PUSH_CLICK_EVENT_KEY, pushClickData, instant, null);
        }
        for (String eventString : dismissEvents) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 2) {
                continue;
            }
            Map<String, Object> pushDismissData = new HashMap<>();
            pushDismissData.put("i", eventFields[0]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[1]));
            eventsInterface.recordSystemEvent(ModulePush.PUSH_DISMISSED_EVENT_KEY, pushDismissData, instant, null);
        }
        for (String eventString : rejectedEvents) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 2) {
                continue;
            }
            Map<String, Object> pushRejectionData = new HashMap<>();
            pushRejectionData.put("i", eventFields[0]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[1]));
            eventsInterface.recordSystemEvent(ModulePush.PUSH_REJECTED_EVENT_KEY, pushRejectionData, instant, null);
        }

        for (String eventString : customChannelClicked) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 3) {
                continue;
            }

            Map<String, Object> eventData = new HashMap<>();
            eventData.put("i", eventFields[0]);
            eventData.put("url", eventFields[1]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[1]));
            eventsInterface.recordSystemEvent(CustomChannelHandler.CUSTOM_CHANNEL_CLICK_EVENT_KEY, eventData, instant, null);
        }

        for (String eventString : customChannelDelivery) {
            String[] eventFields = eventString.split("@");
            if (eventFields.length != 2) {
                continue;
            }
            Map<String, Object> eventData = new HashMap<>();
            eventData.put("i", eventFields[0]);
            UtilsTime.Instant instant = new UtilsTime.Instant(Long.parseLong(eventFields[1]));
            eventsInterface.recordSystemEvent(CustomChannelHandler.CUSTOM_CHANNEL_DELIVERY_EVENT_KEY, eventData, instant, null);
        }

        SharedPref.clearCachedDeliveryEvents(context);
        SharedPref.clearCachedClickedEvents(context);
        SharedPref.clearCachedDismissEvents(context);
        SharedPref.clearCachedRejectedEvents(context);
        SharedPref.clearCachedCustomChannelClick(context);
        SharedPref.clearCachedCustomChannelDelivery(context);
    }

    /**
     * @param key
     * @param segmentation
     * @param instant
     * @param processedSegmentation if segmentation has been processed and reserved keywords should not be removed
     */
    synchronized void recordEventInternal(
            final String key,
            final Map<String, Object> segmentation,
            UserDetails userDetails,
            UtilsTime.Instant instant,
            boolean processedSegmentation
    ) {
        L.v("[ModuleEvents] calling 'recordEventInternal'");
        if (key == null || key.length() == 0) {
            throw new IllegalArgumentException("Valid [" + BuildConfig.FLAVOR + "] event key is required");
        }

        L.d("[ModuleEvents] Recording event with key: [" + key + "]");

        if (!_int.isInitialized()) {
            throw new IllegalStateException("CoreProxy.sharedInstance().init must be called before recordEvent");
        }

        for (InTrackEventListener listener : eventListeners) {
            listener.onEventRaised(key, segmentation, userDetails,
                    instant != null ? instant : UtilsTime.getCurrentInstant(),
                    processedSegmentation,
                    key.startsWith(SYSTEM_EVENTS_PREFIX));
        }

        Map<String, String> segmentationString = null;
        Map<String, Integer> segmentationInt = null;
        Map<String, Double> segmentationDouble = null;
        Map<String, Long> segmentationLong = null;
        Map<String, Boolean> segmentationBoolean = null;
        Map<String, JSONObject> segmentationObjects = null;
        Map<String, JSONArray> segmentationArrays = null;

        if (segmentation != null) {
            segmentationString = new HashMap<>();
            segmentationInt = new HashMap<>();
            segmentationDouble = new HashMap<>();
            segmentationLong = new HashMap<>();
            segmentationBoolean = new HashMap<>();
            segmentationObjects = new HashMap<>();
            segmentationArrays = new HashMap<>();
            Map<String, Object> segmentationReminder = new HashMap<>();

            Utils.removeUnsupportedDataTypes(segmentation);
            if (!processedSegmentation) {
                Utils.removeKeysFromMap(segmentation, ModuleEvents.reservedSegmentationKeys);
            }
            Utils.fillInSegmentation(
                    segmentation,
                    segmentationString,
                    segmentationInt,
                    segmentationDouble,
                    segmentationLong,
                    segmentationBoolean,
                    segmentationObjects,
                    segmentationArrays,
                    segmentationReminder
            );

            if (segmentationReminder.size() > 0) {
                if (L.logEnabled()) {
                    L.w("[ModuleEvents] Event contains events segments with unsupported types:");

                    for (String k : segmentationReminder.keySet()) {
                        if (k != null) {
                            Object obj = segmentationReminder.get(k);
                            if (obj != null) {
                                L.w("[ModuleEvents] Event segmentation key:[" + k + "], value type:[" + obj.getClass().getCanonicalName() + "]");
                            }
                        }
                    }
                }
            }

            for (String k : segmentationString.keySet()) {
                if (k == null || k.length() == 0) {
                    L.e("[ModuleEvents] [" + BuildConfig.FLAVOR + "] event segmentation key cannot be null or empty, skipping");
                    continue;
                }
                if (segmentationString.get(k) == null) {
                    L.e("[ModuleEvents] [" + BuildConfig.FLAVOR + "] event segmentation value cannot be null, skipping");
                    continue;
                }
            }
        }

        if (userDetails == null) {
            userDetails = new UserDetails();
        }
        userDetails.userId = _int.getUserId();

        switch (key) {
            case ModuleFeedback.NPS_EVENT_KEY:
            case ModuleFeedback.SURVEY_EVENT_KEY:
                if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.feedback)) {
                    _int.eventQueue_.recordEvent(key, segmentationString, segmentationInt, segmentationDouble, segmentationLong, segmentationBoolean, segmentationObjects, segmentationArrays, userDetails, instant);
                    _int.sendEventsForced();
                }
                break;
            case ModuleRatings.STAR_RATING_EVENT_KEY:
                if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.starRating)) {
                    _int.eventQueue_.recordEvent(key, segmentationString, segmentationInt, segmentationDouble, segmentationLong, segmentationBoolean, segmentationObjects, segmentationArrays, userDetails, instant);
                    _int.sendEventsIfNeeded();
                }
                break;
            default:
                if (CoreInternal.sharedInstance().consent().getConsent(CoreProxy.SdkFeatureNames.events)) {
                    _int.eventQueue_.recordEvent(key, segmentationString, segmentationInt, segmentationDouble, segmentationLong, segmentationBoolean, segmentationObjects, segmentationArrays, userDetails, instant);
                    _int.sendEventsIfNeeded();
                }
                break;
        }
    }

    synchronized boolean startEventInternal(final String key) {
        if (key == null || key.length() == 0) {
            L.e("[ModuleEvents] Can't start event with a null or empty key");
            return false;
        }
        if (timedEvents.containsKey(key)) {
            return false;
        }
        L.d("[ModuleEvents] Starting event: [" + key + "]");
        timedEvents.put(key, new Event(key));
        return true;
    }

    synchronized boolean endEventInternal(final String key, final Map<String, Object> segmentation, final int count, final double sum) {
        L.d("[ModuleEvents] Ending event: [" + key + "]");

        if (key == null || key.length() == 0) {
            L.e("[ModuleEvents] Can't end event with a null or empty key");
            return false;
        }

        Event event = timedEvents.remove(key);

        if (event != null) {
            if (!_int.getConsent(CoreProxy.SdkFeatureNames.events)) {
                return true;
            }

            if (key == null || key.length() == 0) {
                throw new IllegalArgumentException("Valid [" + BuildConfig.FLAVOR + "] event key is required");
            }
            if (count < 1) {
                throw new IllegalArgumentException("[" + BuildConfig.FLAVOR + "] event count should be greater than zero");
            }
            L.d("[ModuleEvents] Ending event: [" + key + "]");

            UtilsTime.Instant instant = new UtilsTime.Instant(event.timestamp);

            eventsInterface.recordSystemEvent(key, segmentation, instant, null);
            return true;
        } else {
            return false;
        }
    }

    synchronized boolean cancelEventInternal(final String key) {
        if (key == null || key.length() == 0) {
            L.e("[ModuleEvents] Can't cancel event with a null or empty key");
            return false;
        }

        Event event = timedEvents.remove(key);

        return event != null;
    }

    @Override
    void initFinished(Config config) {
        checkCachedPushData(_int.context_);
    }

    @Override
    void halt() {
        timedEvents.clear();
    }

    boolean registerEventListener(@NonNull InTrackEventListener listener) {
        return eventListeners.add(listener);
    }

    boolean unregisterEventListener(@NonNull InTrackEventListener listener) {
        return eventListeners.remove(listener);
    }

    interface InTrackEventListener {
        void onEventRaised(@NonNull final String key, @Nullable final Map<String, Object> segmentation, @Nullable UserDetails userDetails, @NonNull UtilsTime.Instant instant, boolean processedSegmentation, boolean isSystemEvent);
    }

    public class Events {
        /**
         * Record a event with a custom timestamp.
         * Use this in case you want to record events that you have tracked
         * and stored internally
         *
         * @param key          event key
         * @param segmentation custom segmentation you want to set, leave null if you don't want to add anything
         * @param timestamp    unix timestamp in miliseconds of when the event occurred
         */
        public void recordPastEvent(final String key, final Map<String, Object> segmentation, long timestamp) {
            synchronized (_int) {
                if (timestamp == 0) {
                    throw new IllegalStateException("Provided timestamp has to be greater that zero");
                }

                recordPastEvent(key, segmentation, 1, 0, 0, timestamp);
            }
        }

        /**
         * Record a event with a custom timestamp.
         * Use this in case you want to record events that you have tracked
         * and stored internally
         *
         * @param key          event key
         * @param segmentation custom segmentation you want to set, leave null if you don't want to add anything
         * @param count        how many of these events have occured, default value is "1"
         * @param sum          set sum if needed, default value is "0"
         * @param dur          duration of the event, default value is "0"
         * @param timestamp    unix timestamp in miliseconds of when the event occurred
         */
        public void recordPastEvent(final String key, final Map<String, Object> segmentation, final int count, final double sum, final double dur, long timestamp) {
            synchronized (_int) {
                L.i("[Events] Calling recordPastEvent: [" + key + "]");

                if (timestamp == 0) {
                    throw new IllegalStateException("Provided timestamp has to be greater that zero");
                }

                UtilsTime.Instant instant = UtilsTime.Instant.get(timestamp);
                eventsInterface.recordSystemEvent(key, segmentation, instant, null);
            }
        }

        /**
         * Start timed event with a specified key
         *
         * @param key name of the custom event, required, must not be the empty string or null
         * @return true if no event with this key existed before and event is started, false otherwise
         */
        public boolean startEvent(final String key) {
            synchronized (_int) {
                if (!_int.isInitialized()) {
                    throw new IllegalStateException("CoreProxy.sharedInstance().init must be called before startEvent");
                }

                return startEventInternal(key);
            }
        }

        /**
         * End timed event with a specified key
         *
         * @param key name of the custom event, required, must not be the empty string or null
         * @return true if event with this key has been previously started, false otherwise
         */
        public boolean endEvent(final String key) {
            synchronized (_int) {
                return endEvent(key, null, 1, 0);
            }
        }

        /**
         * End timed event with a specified key
         *
         * @param key          name of the custom event, required, must not be the empty string
         * @param segmentation segmentation dictionary to associate with the event, can be null
         * @param count        count to associate with the event, should be more than zero, default value is 1
         * @param sum          sum to associate with the event, default value is 0
         * @return true if event with this key has been previously started, false otherwise
         * @throws IllegalStateException    if  SDK has not been initialized
         * @throws IllegalArgumentException if key is null or empty, count is less than 1, or if segmentation contains null or empty keys or values
         */
        public boolean endEvent(final String key, final Map<String, Object> segmentation, final int count, final double sum) {
            synchronized (_int) {
                if (!_int.isInitialized()) {
                    throw new IllegalStateException("CoreProxy.sharedInstance().init must be called before endEvent");
                }

                if (segmentation != null) {
                    Utils.removeKeysFromMap(segmentation, ModuleEvents.reservedSegmentationKeys);
                }

                return endEventInternal(key, segmentation, count, sum);
            }
        }

        /**
         * Cancel timed event with a specified key
         *
         * @return true if event with this key has been previously started, false otherwise
         **/
        public boolean cancelEvent(final String key) {
            synchronized (_int) {
                L.i("[Events] Calling cancelEvent: [" + key + "]");

                return cancelEventInternal(key);
            }
        }

        public void recordEvent(String key) {
            recordEvent(key, new HashMap<String, Object>());
        }

        public void recordEvent(String key, Map<String, Object> segmentation) {
            if (key.startsWith(SYSTEM_EVENTS_PREFIX)) {
                key = key.replaceAll(SYSTEM_EVENTS_PREFIX, "");
                L.i("[Events] custom event key starting with system event prefix, removing the prefix");
            }
            recordEventInternal(key, segmentation, null, null, true);
        }

        void recordSystemEvent(String key) {
            recordSystemEvent(key, new HashMap<String, Object>());
        }

        void recordSystemEvent(String key, Map<String, Object> attributes) {
            recordSystemEvent(key, attributes, null, null);
        }

        void recordSystemEvent(String key, UserDetails userDetails) {
            recordSystemEvent(key, null, null, userDetails);
        }

        void recordSystemEvent(String key, Map<String, Object> attributes, UtilsTime.Instant instant, UserDetails userDetails) {
            recordEventInternal(Utils.getSystemLabel(key), attributes, userDetails, instant, true);
        }
    }
}
