package sdk.main.core;

class ModuleSessions extends ModuleBase {
    static final String LOGIN_EVENT_KEY = "login";
    static final String UPDATE_PROFILE_EVENT_KEY = ModuleEvents.DUMMY_EVENT_KEY;
    static final String LOGOUT_EVENT_KEY = "logout";
    static final String BEGIN_SESSION_EVENT_KEY = "begin_session";
    static final String APP_INSTALL_EVENT_KEY = "app_installed";
    static final String UPDATE_SESSION_EVENT_KEY = "update_session";
    static final String END_SESSION_EVENT_KEY = "end_session";
    static final String USER_ID_PREFERENCE_KEY = "intrack.user_id";
    final Sessions sessionInterface;
    boolean manualSessionControlEnabled;
    long prevSessionDurationStartTime_ = 0;
    boolean noneStandardPhoneValidation;
    ModuleLog L;

    ModuleSessions(CoreInternal coreInternal, Config config) {
        super(coreInternal);

        L = coreInternal.L;

        L.v("[ModuleSessions] Initialising");

        manualSessionControlEnabled = config.manualSessionControlEnabled;
        _int.disableUpdateSessionRequests_ = config.disableUpdateSessionRequests;
        noneStandardPhoneValidation = config.noneStandardPhoneValidation;

        sessionInterface = new Sessions();
    }

    void beginSessionInternal() {
        L.d("[ModuleSessions] 'beginSessionInternal'");

        prevSessionDurationStartTime_ = System.nanoTime();
        _int.connectionQueue_.beginSession();
    }

    void updateSessionInternal() {
        L.d("[ModuleSessions] 'updateSessionInternal'");

        if (!_int.disableUpdateSessionRequests_) {
            _int.connectionQueue_.updateSession(roundedSecondsSinceLastSessionDurationUpdate());
        }
    }

    /**
     * @param deviceIdOverride used when switching deviceID to a different one and ending the previous session
     */
    void endSessionInternal(String deviceIdOverride) {
        L.d("[ModuleSessions] 'endSessionInternal'");

        _int.connectionQueue_.endSession(roundedSecondsSinceLastSessionDurationUpdate(), deviceIdOverride);
        prevSessionDurationStartTime_ = 0;

        _int.sendEventsIfExist();
    }

    /**
     * Calculates the unsent session duration in seconds, rounded to the nearest int.
     */
    int roundedSecondsSinceLastSessionDurationUpdate() {
        final long currentTimestampInNanoseconds = System.nanoTime();
        final long unsentSessionLengthInNanoseconds = currentTimestampInNanoseconds - prevSessionDurationStartTime_;
        prevSessionDurationStartTime_ = currentTimestampInNanoseconds;
        return (int) Math.round(unsentSessionLengthInNanoseconds / 1000000000.0d);
    }

    @Override
    void halt() {
        prevSessionDurationStartTime_ = 0;
    }

    public class Sessions {
        public void beginSession() {
            synchronized (_int) {
                if (!_int.isInitialized()) {
                    L.e("CoreProxy.sharedInstance().init must be called before beginSession");
                    return;
                }

                L.i("[Sessions] Calling 'beginSession', manual session control enabled:[" + manualSessionControlEnabled + "]");

                if (!manualSessionControlEnabled) {
                    L.w("[Sessions] 'beginSession' will be ignored since manual session control is not enabled");
                    return;
                }

                beginSessionInternal();
            }
        }

        public void updateSession() {
            synchronized (_int) {
                if (!_int.isInitialized()) {
                    L.e("CoreProxy.sharedInstance().init must be called before updateSession");
                    return;
                }

                L.i("[Sessions] Calling 'updateSession', manual session control enabled:[" + manualSessionControlEnabled + "]");

                if (!manualSessionControlEnabled) {
                    L.w("[Sessions] 'updateSession' will be ignored since manual session control is not enabled");
                    return;
                }

                L.i("[Sessions] Calling 'updateSession'");

                updateSessionInternal();
            }
        }

        public void endSession() {
            synchronized (_int) {
                if (!_int.isInitialized()) {
                    L.e("CoreProxy.sharedInstance().init must be called before endSession");
                    return;
                }

                L.i("[Sessions] Calling 'endSession', manual session control enabled:[" + manualSessionControlEnabled + "]");

                if (!manualSessionControlEnabled) {
                    L.w("[Sessions] 'endSession' will be ignored since manual session control is not enabled");
                    return;
                }

                endSessionInternal(null);
            }
        }

        void recordLogin(UserDetails userDetails) {

            if (userDetails.userId == null) {
                L.e("User Id must be a non-null value");
                return;
            }

            if (!userDetails.validate(noneStandardPhoneValidation, false)) {
                return;
            }

            synchronized (_int) {
                storeUserId(userDetails.userId);
                _int.events().recordSystemEvent(LOGIN_EVENT_KEY, userDetails);
            }
        }

        void recordLogout() {
            synchronized (_int) {
                _int.events().recordSystemEvent(LOGOUT_EVENT_KEY);
                storeUserId(null);
            }
        }

        void updateProfile(UserDetails userDetails) {

//            if (userDetails.userId == null) {
//                L.e("User Id must be a non-null value");
//                return;
//            }

            if (!userDetails.validate(noneStandardPhoneValidation, true)) {
                return;
            }

            synchronized (_int) {
                if (userDetails.userId != null) {
                    storeUserId(userDetails.userId);
                }
                _int.events().recordSystemEvent(UPDATE_PROFILE_EVENT_KEY, userDetails);
            }
        }

        private void storeUserId(String userId) {
            _int.config_.sharedPref.setPreference(USER_ID_PREFERENCE_KEY, userId);
            _int.moduleIAM.updateUserId(userId);
        }
    }
}
