package sdk.main.core;

import androidx.annotation.Nullable;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

public class UserDetails<T> {
    public static final String firstNameKey = "firstName";
    public static final String lastNameKey = "lastName";
    public static final String emailKey = "email";
    public static final String phoneKey = "phone";
    public static final String countryKey = "country";
    public static final String stateKey = "state";
    public static final String cityKey = "city";
    public static final String genderKey = "gender";
    public static final String birthdayKey = "birthday";
    public static final String companyKey = "company";
    public static final String hashedPhoneKey = "hashedPhone";
    public static final String hashedEmailKey = "hashedEmail";
    public static final String smsOptInKey = "smsOptIn";
    public static final String emailOptInKey = "emailOptIn";
    public static final String pushOptInKey = "pushOptIn";
    public static final String webPushOptInKey = "webPushOptIn";
    public static final String userIdKey = "userId";
    public static final String userAttributesKey = "attributes";
    private static final Pattern emailPattern = Pattern.compile("^[\\w!#$%&'*+/=?`{|}~^-]+(?:\\.[\\w!#$%&'*+/=?`{|}~^-]+)*@(?:[a-zA-Z0-9-]+\\.)+[a-zA-Z]{2,6}$");
    private static final Pattern standardPhonePattern = Pattern.compile("^\\+([0-9]){6,19}[0-9]$");
    private static final Pattern noneStandardPhonePattern = Pattern.compile("^\\+?([0-9]){6,19}[0-9]$");
    public @Nullable String firstName;
    public @Nullable String lastName;
    public @Nullable String email;
    public @Nullable String phone;
    // Location
    public @Nullable String country;
    public @Nullable String state;
    public @Nullable String city;
    // System Attributes
    public @Nullable T gender;
    public @Nullable Date birthday;
    public @Nullable String company;
    // Hashed Contact Info
    public @Nullable String hashedPhone;
    public @Nullable String hashedEmail;
    // Opt In
    public @Nullable Boolean smsOptIn;
    public @Nullable Boolean emailOptIn;
    public @Nullable Boolean pushOptIn;
    public @Nullable Boolean webPushOptIn;
    // Identifiers
    public @Nullable String userId;
    public Map<String, Object> userAttributes = new HashMap<>();

    public UserDetails() {
    }

    public UserDetails(JSONObject jsonObject) {

        try {
            if (jsonObject.has(firstNameKey)) {
                firstName = jsonObject.getString(firstNameKey);
            }
            if (jsonObject.has(lastNameKey)) {
                lastName = jsonObject.getString(lastNameKey);
            }
            if (jsonObject.has(emailKey)) {
                email = jsonObject.getString(emailKey);
            }
            if (jsonObject.has(phoneKey)) {
                phone = jsonObject.getString(phoneKey);
            }
            if (jsonObject.has(countryKey)) {
                country = jsonObject.getString(countryKey);
            }
            if (jsonObject.has(stateKey)) {
                state = jsonObject.getString(stateKey);
            }
            if (jsonObject.has(cityKey)) {
                city = jsonObject.getString(cityKey);
            }
            if (jsonObject.has(genderKey)) {
                gender = (T) ApiUserGender.valueOf(jsonObject.getString(genderKey).toUpperCase());
            }
            if (jsonObject.has(birthdayKey)) {
                birthday = Utils.parseDate(jsonObject.getString(birthdayKey));
            }
            if (jsonObject.has(companyKey)) {
                company = jsonObject.getString(companyKey);
            }
            if (jsonObject.has(hashedPhoneKey)) {
                hashedPhone = jsonObject.getString(hashedPhoneKey);
            }
            if (jsonObject.has(hashedEmailKey)) {
                hashedEmail = jsonObject.getString(hashedEmailKey);
            }
            if (jsonObject.has(smsOptInKey)) {
                smsOptIn = jsonObject.getBoolean(smsOptInKey);
            }
            if (jsonObject.has(emailOptInKey)) {
                emailOptIn = jsonObject.getBoolean(emailOptInKey);
            }
            if (jsonObject.has(pushOptInKey)) {
                pushOptIn = jsonObject.getBoolean(pushOptInKey);
            }
            if (jsonObject.has(webPushOptInKey)) {
                webPushOptIn = jsonObject.getBoolean(webPushOptInKey);
            }
            if (jsonObject.has(userIdKey)) {
                userId = jsonObject.getString(userIdKey);
            }
            if (jsonObject.has(userAttributesKey)) {
                userAttributes = Utils.mapifyJsonObject(jsonObject.getJSONObject(userAttributesKey));
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public boolean validate(Boolean useNoneStandardPhoneValidation, Boolean isUpdate) {
        if (!isUpdate && (this.userId == null || this.userId.isEmpty() || this.userId.length() > 100)) {
            CoreInternal.sharedInstance().L.w("UserDetails userId is not valid!");
            return false;
        }

        if (this.firstName != null && this.firstName.length() > 1000) {
            CoreInternal.sharedInstance().L.w("UserDetails firstName is not valid!");
            return false;
        }
        if (this.lastName != null && this.lastName.length() > 1000) {
            CoreInternal.sharedInstance().L.w("UserDetails lastName is not valid!");
            return false;
        }

        if (this.email != null && !this.email.isEmpty() && (this.email.length() > 255 || !emailPattern.matcher(this.email).matches())) {
            CoreInternal.sharedInstance().L.w("UserDetails email is not valid!");
            return false;
        }

        Pattern phonePattern = useNoneStandardPhoneValidation ? noneStandardPhonePattern : standardPhonePattern;
        if (this.phone != null && (this.phone.length() > 255 || !phonePattern.matcher(this.phone).matches())) {
            CoreInternal.sharedInstance().L.w("UserDetails phone is not valid!");
            return false;
        }

        if (this.country != null && this.country.length() > 255) {
            CoreInternal.sharedInstance().L.w("UserDetails country is not valid!");
            return false;
        }
        if (this.state != null && this.state.length() > 255) {
            CoreInternal.sharedInstance().L.w("UserDetails state is not valid!");
            return false;
        }
        if (this.city != null && this.city.length() > 255) {
            CoreInternal.sharedInstance().L.w("UserDetails city is not valid!");
            return false;
        }

        if (this.userAttributes != null) {
            Utils.removeUnsupportedDataTypes(this.userAttributes);
        }

        return true;
    }

    @Override
    public String toString() {
        return "UserDetails{" +
                "firstName='" + firstName + '\'' +
                ", lastName='" + lastName + '\'' +
                ", email='" + email + '\'' +
                ", phone='" + phone + '\'' +
                ", country='" + country + '\'' +
                ", state='" + state + '\'' +
                ", city='" + city + '\'' +
                ", gender=" + gender +
                ", birthday=" + birthday +
                ", company='" + company + '\'' +
                ", hashedPhone='" + hashedPhone + '\'' +
                ", hashedEmail='" + hashedEmail + '\'' +
                ", smsOptIn=" + smsOptIn +
                ", emailOptIn=" + emailOptIn +
                ", pushOptIn=" + pushOptIn +
                ", webPushOptIn=" + webPushOptIn +
                ", userId='" + userId + '\'' +
                ", userAttributes=" + userAttributes +
                '}';
    }
}
