/*
 * 2stepverification Android SDK
 *
 * @version     1.0.0
 * @copyright   Copyright © 2017, 2stepverification.ir.
 * @license     http://opensource.org/licenses/mit-license.php The MIT License (MIT).
 * @author      Amir Moradabadi
 * @repository  https://github.com/2stepverification/android-sdk
 * @website     http://www.2stepverification.ir
 * @support     info@2stepverification.ir
 *
 */

package ir.twostepverification.sdk;

import android.os.Bundle;
import android.os.Message;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;

class AsyncHttpRequest implements Runnable {

    private final String sdkversion = "android.v1";
    private final Service service;
    private final String apiversion = "v1";
    private final String apibase = "https://api.2stepverification.ir/" + apiversion;


    private final AsyncHttpResponse asyncHttpResponse;

    private final String apikey;
    private final String apiauthorization;
    private final String requestId;
    private final String data;

    AsyncHttpRequest(Service service, ResponseHandlerInterface responseHandlerInterface, String apikey, String apiauthorization, String requestId, String data) {
        this.service = service;
        this.apikey = urlencode(apikey);
        this.apiauthorization = apiauthorization;
        this.requestId = urlencode(requestId);
        this.data = urlencode(data);
        this.asyncHttpResponse = new AsyncHttpResponse(responseHandlerInterface, requestId, service);
    }


    @Override
    public void run() {
        Bundle bundle = new Bundle();
        bundle.putInt("service", this.service.getValue());

        HttpModel httpModel = new HttpModel();

        switch (this.service) {
            case RequestCall:
                String api_request_call = apibase + "/call/request";
                httpModel = post(api_request_call, buildRequestBody());
                break;
            case VerifyCall:
                String api_verify_call = apibase + "/call/verify";
                httpModel = post(api_verify_call, buildVerifyBody());
                break;

            case RequestSMS:
                String api_request_sms = apibase + "/sms/request";
                httpModel = post(api_request_sms, buildRequestBody());
                break;
            case VerifySMS:
                String api_verify_sms = apibase + "/sms/verify";
                httpModel = post(api_verify_sms, buildVerifyBody());
                break;

            case RequestEmail:
                String api_request_email = apibase + "/email/request";
                httpModel = post(api_request_email, buildRequestEmailBody());
                break;
            case VerifyEmail:
                String api_verify_email = apibase + "/email/verify";
                httpModel = post(api_verify_email, buildVerifyEmailBody());
                break;

            case RequestTelegram:
                String api_request_telegram = apibase + "/telegram/request";
                httpModel = post(api_request_telegram, buildRequestBody());
                break;
            case VerifyTelegram:
                String api_verify_telegram = apibase + "/telegram/verify";
                httpModel = post(api_verify_telegram, buildVerifyBody());
                break;

            case RequestWhatsapp:
                String api_request_whatsapp = apibase + "/whatsapp/request";
                httpModel = post(api_request_whatsapp, buildRequestBody());
                break;
            case VerifyWhatsapp:
                String api_verify_whatsapp = apibase + "/whatsapp/verify";
                httpModel = post(api_verify_whatsapp, buildVerifyBody());
                break;
        }

        bundle.putByteArray("data", httpModel.getData());
        bundle.putString("message", httpModel.getMessage());

        Message msg = asyncHttpResponse.obtainMessage();
        msg.setData(bundle);
        asyncHttpResponse.sendMessage(msg);
    }

    private String buildRequestBody() {
        return String.format("apikey=%s&requestid=%s&number=%s&sdkversion=%s", apikey, requestId, data, sdkversion);
    }

    private String buildVerifyBody() {
        return String.format("apikey=%s&requestid=%s&code=%s&sdkversion=%s", apikey, requestId, data, sdkversion);
    }

    private String buildRequestEmailBody() {
        return String.format("apikey=%s&requestid=%s&email=%s&sdkversion=%s", apikey, requestId, data, sdkversion);
    }

    private String buildVerifyEmailBody() {
        return String.format("apikey=%s&requestid=%s&code=%s&sdkversion=%s", apikey, requestId, data, sdkversion);
    }

    private String urlencode(String data) {
        try {
            return URLEncoder.encode(data, "UTF-8");
        } catch (Exception e) {
            return data;
        }
    }

    private HttpModel post(String u, String body) {
        try {
            URL url = new URL(u);
            HttpURLConnection urlConnection = (HttpURLConnection) url.openConnection();
            urlConnection.setRequestMethod("POST");
            urlConnection.setDoOutput(true);
            urlConnection.setRequestProperty("Accept", "application/octet-stream");
            urlConnection.setRequestProperty("Authorization", apiauthorization);
            urlConnection.setConnectTimeout(10000);
            urlConnection.setReadTimeout(10000);
            OutputStreamWriter out = new OutputStreamWriter(urlConnection.getOutputStream());
            out.write(body);
            out.close();

            InputStream is = urlConnection.getErrorStream();
            if (is == null) {
                is = urlConnection.getInputStream();
            }
            return new HttpModel(toByteArray(is), "");
        } catch (Exception e) {
            return new HttpModel(null, e.getMessage());
        }
    }

    private byte[] toByteArray(InputStream input) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        copy((InputStream) input, (OutputStream) output);
        return output.toByteArray();
    }

    private int copy(InputStream input, OutputStream output) throws IOException {
        long count = copyLarge(input, output);
        return count > 2147483647L ? -1 : (int) count;
    }

    private long copyLarge(InputStream input, OutputStream output) throws IOException {
        return copy(input, output, 4096);
    }

    private long copy(InputStream input, OutputStream output, int bufferSize) throws IOException {
        return copyLarge(input, output, new byte[bufferSize]);
    }

    private long copyLarge(InputStream input, OutputStream output, byte[] buffer) throws IOException {
        long count;
        int n;
        for (count = 0L; -1 != (n = input.read(buffer)); count += (long) n) {
            output.write(buffer, 0, n);
        }
        return count;
    }
}
